//  Copyright John Maddock 2015.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifdef _MSC_VER
#  pragma warning (disable : 4224)
#endif

#include <boost/math/special_functions/ellint_1.hpp>
#include <boost/array.hpp>
#include <boost/lexical_cast.hpp>
#include "../../test/table_type.hpp"
#include "table_helper.hpp"
#include "performance.hpp"
#include <iostream>

typedef double T;
#define SC_(x) static_cast<double>(x)
static const boost::array<boost::array<T, 2>, 9> data2 = { {
   { { SC_(0.0), SC_(1.5707963267948966192313216916397514420985846996876) } },
   { { SC_(0.125), SC_(1.5769867712158131421244030532288080803822271060839) } },
   { { SC_(0.25), SC_(1.5962422221317835101489690714979498795055744578951) } },
   { { T(300) / 1024, SC_(1.6062331054696636704261124078746600894998873503208) } },
   { { T(400) / 1024, SC_(1.6364782007562008756208066125715722889067992997614) } },
   { { SC_(-0.5), SC_(1.6857503548125960428712036577990769895008008941411) } },
   { { SC_(-0.75), SC_(1.9109897807518291965531482187613425592531451316788) } },
   { { 1 - T(1) / 8, SC_(2.185488469278223686913080323730158689730428415766) } },
   { { 1 - T(1) / 1024, SC_(4.5074135978990422666372495313621124487894807327687) } },
   } };

int main()
{
#include "ellint_k_data.ipp"

   add_data(data2);
   add_data(ellint_k_data);

   unsigned data_total = data.size();

   screen_data([](const std::vector<double>& v){  return boost::math::ellint_1(v[0]);  }, [](const std::vector<double>& v){ return v[1];  });


#if defined(TEST_LIBSTDCXX) && !defined(COMPILER_COMPARISON_TABLES)
   screen_data([](const std::vector<double>& v){  return std::tr1::comp_ellint_1(v[0]);  }, [](const std::vector<double>& v){ return v[1];  });
#endif
#if defined(TEST_GSL) && !defined(COMPILER_COMPARISON_TABLES)
   screen_data([](const std::vector<double>& v){  return gsl_sf_ellint_Kcomp(v[0], GSL_PREC_DOUBLE);  }, [](const std::vector<double>& v){ return v[1];  });
#endif

   unsigned data_used = data.size();
   std::string function = "ellint_1 (complete)[br](" + boost::lexical_cast<std::string>(data_used) + "/" + boost::lexical_cast<std::string>(data_total) + " tests selected)";
   std::string function_short = "ellint_1 (complete)";

   double time;

   time = exec_timed_test([](const std::vector<double>& v){  return boost::math::ellint_1(v[0]);  });
   std::cout << time << std::endl;
#if !defined(COMPILER_COMPARISON_TABLES) && (defined(TEST_GSL) || defined(TEST_RMATH) || defined(TEST_LIBSTDCXX))
   report_execution_time(time, std::string("Library Comparison with ") + std::string(compiler_name()) + std::string(" on ") + platform_name(), function, boost_name());
#endif
   report_execution_time(time, std::string("Compiler Comparison on ") + std::string(platform_name()), function_short, compiler_name() + std::string("[br]") + boost_name());
   //
   // Boost again, but with promotion to long double turned off:
   //
#if !defined(COMPILER_COMPARISON_TABLES)
   if(sizeof(long double) != sizeof(double))
   {
      time = exec_timed_test([](const std::vector<double>& v){  return boost::math::ellint_1(v[0], boost::math::policies::make_policy(boost::math::policies::promote_double<false>()));  });
      std::cout << time << std::endl;
#if !defined(COMPILER_COMPARISON_TABLES) && (defined(TEST_GSL) || defined(TEST_RMATH) || defined(TEST_LIBSTDCXX))
      report_execution_time(time, std::string("Library Comparison with ") + std::string(compiler_name()) + std::string(" on ") + platform_name(), function, boost_name() + "[br]promote_double<false>");
#endif
      report_execution_time(time, std::string("Compiler Comparison on ") + std::string(platform_name()), function_short, compiler_name() + std::string("[br]") + boost_name() + "[br]promote_double<false>");
   }
#endif


#if defined(TEST_LIBSTDCXX) && !defined(COMPILER_COMPARISON_TABLES)
   time = exec_timed_test([](const std::vector<double>& v){  return std::tr1::comp_ellint_1(v[0]);  });
   std::cout << time << std::endl;
   report_execution_time(time, std::string("Library Comparison with ") + std::string(compiler_name()) + std::string(" on ") + platform_name(), function, "tr1/cmath");
#endif
#if defined(TEST_GSL) && !defined(COMPILER_COMPARISON_TABLES)
   time = exec_timed_test([](const std::vector<double>& v){  return gsl_sf_ellint_Kcomp(v[0], GSL_PREC_DOUBLE);  });
   std::cout << time << std::endl;
   report_execution_time(time, std::string("Library Comparison with ") + std::string(compiler_name()) + std::string(" on ") + platform_name(), function, "GSL " GSL_VERSION);
#endif

   return 0;
}

