/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qzioncanvas.h"

#include <QDebug>
#include <QRegion>
#include <QPainter>
#include <QPaintEvent>
#include <QApplication>
#include <QGraphicsView>
#include <QGraphicsScene>
#include <QGraphicsItemGroup>
#include <QGraphicsSceneContextMenuEvent>

#include "qzionmacros.h"
#include "qgraphicsziongroup.h"
#include "qzionabstractcanvas_p.h"

class QZionCanvasPrivate : public QGraphicsView
{
public:
    QZionCanvasPrivate(QZionCanvas *owner, QWidget *parent = 0);
    virtual ~QZionCanvasPrivate();

    QZionCanvas *owner;
    QGraphicsZionGroup *group;

protected:
    virtual void paintEvent(QPaintEvent *event);
    virtual void keyPressEvent(QKeyEvent *event);
    virtual void keyReleaseEvent(QKeyEvent *event);

private:
    virtual void resizeEvent(QResizeEvent *event);
};


QZionCanvasPrivate::QZionCanvasPrivate(QZionCanvas *owner, QWidget *parent)
    : QGraphicsView(new QGraphicsScene, parent), owner(owner)
{
    group = new QGraphicsZionGroup;
    group->owner = owner;
    group->setClipping(false);
    group->show();

    // Always top most. When children are removed from this main group,
    // it's zValue are adjusted to zero to not block canvas events.
    group->setZValue(999999);

    scene()->addItem(group);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
}


QZionCanvasPrivate::~QZionCanvasPrivate()
{
    delete group;
}

void QZionCanvasPrivate::paintEvent(QPaintEvent *event)
{
    QGraphicsView::paintEvent(event);
    owner->paintEvent(event);
}

void QZionCanvasPrivate::resizeEvent(QResizeEvent *event)
{
    QGraphicsView::resizeEvent(event);
    scene()->setSceneRect(QRect(0, 0, size().width(), size().height()));

    QZionAbstractCanvasPrivate *da = owner->_QZionAbstractCanvas_data;
    foreach (QZionObject *obj, da->items)
        obj->canvasResizeEvent(event);
}

void QZionCanvasPrivate::keyPressEvent(QKeyEvent *event)
{
    owner->keyPressEvent(event);
}

void QZionCanvasPrivate::keyReleaseEvent(QKeyEvent *event)
{
    owner->keyReleaseEvent(event);
}

QZionCanvas::QZionCanvas(QWidget *parent)
    : QObject(), QZionAbstractCanvas()
{
    _QZionCanvas_data = new QZionCanvasPrivate(this, parent);

    QZDP(QZionAbstractCanvas, da);
    da->canvasGroup = _QZionCanvas_data->group;
}

QZionCanvas::~QZionCanvas()
{
    delete _QZionCanvas_data;
}

QSize QZionCanvas::size() const
{
    QZD(QZionCanvas);
    return d->size();
}

void QZionCanvas::setSize(const QSize &size)
{
    QZD(QZionCanvas);
    d->resize(size);
}

QWidget *QZionCanvas::widget()
{
    return _QZionCanvas_data;
}

void QZionCanvas::keyPressEvent(QKeyEvent *)
{

}

void QZionCanvas::keyReleaseEvent(QKeyEvent *)
{

}

void QZionCanvas::paintEvent(QPaintEvent *)
{
    QZD(QZionCanvas);
    emit updated(QRegion(d->scene()->sceneRect().toRect()));
}


//////////////////////////////////////////////////////////////
void QZionCanvas::updateChanges() {}
void QZionCanvas::ensurePendingUpdate() {}
void QZionCanvas::invalidate(const QRect &, bool) {}
void QZionCanvas::invalidate(const QRegion &, bool) {}
