;;; SIMULATE NETWORK
;;; ----------------

;(defmacro glue (feature &rest ))

;(defmacro funcname (&rest names)
;  `(pack-intern "ACL" (,@names)))

;(defmacro callme (name &rest args)
;  (let ((call
;	 (cons (funcname name) args)))
;    `(QUOTE ,call)))

; Usage:
;  (glue outgoing
;    (decrypt (deliver verify))
;
; Well, try this instead:
;  (glue
;    (email-decrypt-outgoing
;      (deliver email-verify-outgoing)
;    (email-verify-outgoing
;      (deliver email-auto-outgoing))

(defun msgtomessage (msg)
  (begin
   (cw "msgtomessage")
  (let ((head (car msg))
	(from (cadr msg))
	(to   (caddr msg))
	(body (cadddr msg)))
    (if (equal 'smessage head)
	(mk-message
	 (mk-email (car from) (cadr from))
	 (mk-email (car to)   (cadr to  ))
	 '()
	 (list body))
      '()))))
 
(defstructure action
  (type (:assert (member type '(command init send mail deliver
                                  ok deliver-ok))))
  (name (:assert (and (not (null name)) (symbolp name))))
  (arg1 (:assert (or (equal 'init type) 
		     (if (member-equal type
				       '(send mail deliver))
			 (message-p arg1)
		       (not (null arg1))))))
  (arg2 (:assert (or (not (equal type 'command))
		     (atom-listp arg2))))
  (:options :guards))


(defun action-listp (lst)
  (if (ATOM LST) 
      (EQ LST NIL)
    (AND (action-P (CAR LST))
	 (action-LISTP (CDR LST)))))

(defun mk-action4 (type name arg1 arg2)
  (make-action :type type
	       :name name
	       :arg1 arg1
	       :arg2 arg2))

(defmacro mk-action (type &optional name arg1 arg2)
  (mk-action4 type name arg1 arg2))

(defun send-ok (msg user)
  (begin
   (cw "send-ok: ~%")
   (mv 'deliver msg (set-var 'sent-msg msg user))))

(defun send-abort (msg user)
  (mv 'abort msg user))

(defun deliver-ok (msg user)
  (mv 'deliver-ok msg (set-var 'recv-msg msg user)))

(defalias deliver-abort deliver-ok)

(defun recv-ok (msg user)
  (mv 'ok msg user))

(defun recv-abort (msg user)
  (mv 'abort msg user))

(fif address
     (defalias user-send-begin  user-send-address)
     (defalias user-send-begin  user-send-address-next))

(fif sign
     (defalias user-send-address-next user-send-sign)
     (defalias user-send-address-next user-send-sign-next))

(fif encrypt
     (defalias user-send-sign-next user-send-encrypt)
     (defalias user-send-sign-next user-send-encrypt-next))

(defalias user-send-encrypt-next	user-send-ok)

(fif decrypt
     (defalias user-deliver-begin user-deliver-decrypt)
     (defalias user-deliver-begin user-deliver-decrypt-next))

(fif verify
     (defalias user-deliver-decrypt-next user-deliver-verify)
     (defalias user-deliver-decrypt-next user-deliver-verify-next))

(fif auto
     (defalias user-deliver-verify-next user-deliver-auto)
     (defalias user-deliver-verify-next user-deliver-auto-next))

(defalias user-deliver-auto-next user-deliver-ok)


(fif host
     (defalias host-send-begin host-send-host)
     (defalias host-send-begin host-send-host-next))

(defalias host-send-host-next host-send-ok)

;;(user-init user) -> user
;;
(defun user-init (user)
;  (declare (xargs :guard (symbol-alistp user)))
  (let-seq user
           (fif sign    (email-sign-init             user) user)
           (fif verify  (email-verify-init           user) user)
	   (fif encrypt (email-encrypt-init          user) user)
	   (fif decrypt (email-decrypt-init          user) user)
	   (fif auto    (email-auto-init             user) user)
	   (fif address (email-address-init          user) user)
           user))

(defun user-command-dummy (cmd args user)
  (declare (ignore cmd args))
  user)

;;(user-command user list[<anything>]) -> user
;;run through all commands
(defun user-command (act user)
;  (declare (xargs :guard (and (symbol-alistp user)
;			      (true-listp    args))))
  (let ((cmd  (action-arg1 act))
	(args (action-arg2 act)))
    (let-seq user
	     (fif address (email-address-command cmd args user)
		  user)
	     (fif auto    (email-auto-command    cmd args user)
		  user)
	     (fif sign    (email-sign-command    cmd args user)
		  user)
	     (fif encrypt (email-encrypt-command cmd args user)
		  user)
	     (fif decrypt (email-decrypt-command cmd args user)
		  user)
	     (fif verify  (email-verify-command  cmd args user)
		  user)
	     (user-command-dummy cmd args user)
	     user)))


;;;(user-send-*
;;;send user message

(defun user-send-ok (msg user)
  (begin 
  (cw "user-send-ok: Message mailed (~x0): ~x1~%" msg user)
  (send-ok msg user)))

(defun user-send-abort (msg user)
  (begin 
   (cw "Message aborted~%")
   (send-abort msg user)))

(fif encrypt
     (defund user-send-encrypt (msg user)
       (begin (cw "user-send-encrypt: ~x0 ~% ~x1~%" msg user)
              (mv-let (status new-msg new-user)
                      (email-encrypt-outgoing msg user)
                      (cond ((equal 'mail status)
                             (user-send-encrypt-next new-msg new-user))
                            (t (user-send-abort new-msg new-user)))))))


(fif sign
     (defun user-send-sign (msg user)
       (mv-let (status msg user)
	       (email-sign-outgoing msg user)
	       (let ((user (set-var 'sent-msg msg user)))
		 (if (equal 'mail status)
                      (user-send-sign-next msg user)
		      (user-send-abort     msg user))))))

(fif address
     (defun user-send-address (msg user)
       (mv-let (status msg user)
	       (email-address-outgoing msg user)    
	       (let ((user (set-var 'sent-msg msg user)))
		 (user-send-address-next msg user)))))

;;(user-send user msg) -> user
;;
(defun user-send (msg user)
      (begin (cw "user-send: ~x0 / ~x1~%" user (stringify msg))
  (user-send-begin msg user)))

;;;(user-deliver
(defun user-deliver-ok (msg user)
  (begin (cw "Message delivered~%")
    (deliver-ok msg user)))
;    (deliver-ok msg (set-var 'recv-msg msg user))))

(defun user-deliver-abort (msg user)
   (deliver-abort msg user))
;   (deliver-abort msg (set-var 'recv-msg msg user)))

(defmacro do-options (on-abort options)
  (let ((opt (car options))
	(rst (cdr options)))
    (cond ((endp rst) `(if (equal ',(car opt) status)
			   (,(cadr opt) msg user)
			 (,on-abort msg user)))
	   (t `(if (equal ',(car opt) status)
		   (,(cadr opt) msg user)
		 (do-options ,on-abort ,rst))))))

(defmacro do-next (fn callee on-abort options)
  `(defun ,fn (msg user)
     (mv-let (status msg user)
	     (,callee msg user)
	     (do-options ,on-abort ,options))))

(fif auto
     (defun user-deliver-auto-reply (msg user)
       (begin (cw "[user-deliver-auto-reply] Auto ~x0~%" msg)
	      (mv 'mail msg user))))
     
(fif auto
     (defun user-deliver-auto (msg user)
       (mv-let (status new-msg new-user)
	       (email-auto-incoming msg user)
	       (begin 
		(cond ((equal 'deliver status)
		       (user-deliver-auto-next  new-msg new-user))
		      ((equal 'mail status)
		       (user-deliver-auto-reply new-msg (set-var 'recv-msg msg new-user)))
		      (t 
		       (user-deliver-abort      new-msg new-user))))))
     )

; should do the same as above...
;(do-next user-deliver-auto
;	 email-auto-incoming
;	 user-deliver-abort
;	 ((deliver user-deliver-auto-next)
;	  (mail    user-deliver-auto-reply)))

(fif verify 
     (defun user-deliver-verify (msg user)
       (mv-let (status msg user)
	       (email-verify-incoming msg user)
	       (let* ((user (set-var 'recv-msg msg user)))
		 (user-deliver-verify-next msg user)))))

(fif decrypt
     (defund user-deliver-decrypt (msg user)
       (mv-let (status new-msg new-user)
               (email-decrypt-incoming          msg user)
               (if status
                   (user-deliver-decrypt-next       new-msg new-user)
                   (user-deliver-decrypt-next       new-msg new-user)))))
     

     
;;(user-deliver symbol message) -> message
;;
(defun user-deliver (msg user)
  (begin
    (cw "    [user-deliver]~x0 to ~x1~%" msg user)
    (user-deliver-begin msg user)))

;;(clean-msg message) -> message
;;
;(defun clean-msg (msg)
;  (begin
;   ;(cw (string-append (string-append "MSG-->" (stringify msg)) "~%"))
;   (if (message-p msg)
;       msg
;     (let ((msg msg))
;       (begin
;        ;(cw (string-append (string-append "MSG2-->" (stringify msg)) "~%"))
;        (make-message 
;         (nth 1 msg)
;         (nth 2 msg)
;         '()
;         (nth 3 msg)))))))
  


;;(init-users list[users]) -> list[users]
;; 
(defun init-users (users)
  (if (not (endp users))
      (let ((name (caar users))
            (user (cdar users)))
        (acons name (user-init user)
               (init-users (cdr users))))
    nil))


;;(clean-act act ->
;(defun clean-act (act)
;  (if (equal 'mail (car act))
;      (let ((msg (cadr act)))
;	(begin
;	 (cw "msg: ~x0--~x1~%" msg (car (sender msg)))
;	 (list (car (sender msg))
;	       'send
;	       msg))
;	)
;    act))

;;;; HOST FUNCTIONS
;;;;

(defun host-init (host)
  (let-seq host
           (fif host (email-host-init host) host)
           host))

(defun host-command (act host)
  (let ((cmd  (action-arg1 act))
	(args (action-arg2 act)))
    (let-seq host
	     (fif host 
		  (email-host-command  cmd args host) 
		  host)
	     host)))

(defun host-send-ok (msg host)
  (begin 
    (cw "MAILED~%")
    (send-ok msg host)))

(defun host-send-delivered (msg host)
  (begin 
    (cw "host-send-delivered DELIVERED~%")
    (mv 'deliver msg (set-var 'recv-msg msg host))))

(defun host-send-abort (msg host)
  (send-abort msg host))

(fif host
     (defun host-send-host (msg host)
       (mv-let (status new-msg new-host) 
	       (email-host-outgoing msg host)
;	       (let ((host (set-var 'sent-msg msg host)))
	       (cond ((equal 'deliver status)
		      (host-send-delivered new-msg new-host))
		     ((equal 'mail    status)
		      (host-send-host-next new-msg new-host))
		     (t (host-send-abort new-msg new-host)))))
)

(defun host-send (msg host)
  (host-send-begin msg host))

; init-hosts list[hosts] -> list[hosts]
; 
(defun init-hosts (hosts)
  (if (not (endp hosts))
      (let ((name (caar hosts))
            (host (cdar hosts)))
        (acons name (host-init host) 
               (init-hosts (cdr hosts))))
    nil))

;;;; OTHER 

(defun add-user-mail-action (type msg rest) 
  (cons (mk-action
	 type
	 (email-user (message-sender msg))
	 msg)
	rest))

(defun add-sender-mail-action (type msg rest) 
  (cons (mk-action
	 type
	 (email-user (message-sender msg))
	 msg)
	rest))


(defun add-host-mail-action (type msg rest) 
  (cons (mk-action
	 type
	 (email-host (recipient msg))
	 msg)
	rest))

;;(do-init action list[actions] list[user] list[host]) -> 
;;
(defun do-init (action rest users hosts)
  (declare (ignore action))
  (begin (cw "do-init~%")
  (mv rest
      (init-users users)
      (init-hosts hosts))))

;;(do-command action list[actions] list[user] list[host]) -> 
; (BOB COMMAND "SET_USER" ("bob"))
(defun do-command (action rest users hosts)
  (begin
   (cw "do-command ~x0~%" action)
  (let* ((name (action-name  action)))
    (cond ((get-var name users)  ; USER COMMAND
           (let* ((user      (get-var name users))
                  (new-user  (user-command action user))
                  (new-users (set-var name new-user users)))
             (mv rest new-users hosts)))
          ((get-var name hosts)  ; HOST COMMAND
           (let* ((host      (get-var name hosts))
                  (new-host  (host-command action host))
                  (new-hosts (set-var name new-host hosts)))
             (mv rest users new-hosts)))
          (t (mv rest users hosts))))))

(defun do-user-send (action rest users hosts)
  (let* ((name (action-name action))
	 (msg  (action-arg1 action)))  
    (mv-let (status new-msg new-user)
	    (user-send msg (get-var name users))
	    (let ((new-users (set-var name new-user users)))
	      (begin
	       (cw "will send(~x0) ~x1, user: ~x2~%" status new-msg
		   (email-host (recipient new-msg)))
	       (if status
		   (mv (add-host-mail-action 'send new-msg rest)
		       new-users
		       hosts)
		 (mv rest new-users hosts)))))))



(defun do-host-send (action rest users hosts)
  (let* ((name (action-name action))
	 (msg  (action-arg1 action)))
    (mv-let (status new-msg new-host)
	    (host-send msg (get-var name hosts))
	    (let ((new-hosts (set-var name new-host hosts)))
	      (cond ((equal 'deliver status)
		     (mv (add-user-mail-action 'deliver new-msg rest)
		    	 users
		    	 new-hosts))
		    ((equal 'mail    status)
		     ; "mail" to another host == "deliver" on other host
 		     (mv (add-user-mail-action 'deliver new-msg rest)
		    	 users
		    	 new-hosts))
		    (t (mv rest users new-hosts)))))))
		    

;;(do-send action list[actions] list[user] list[host]) -> 
; (BOB SEND (SMESSAGE (bob host) (rjh host) "Body line 1")))))
(defun do-send (action rest users hosts)
  (begin
   (cw "do-send:  ~x0~%" (stringify action))
   (let* ((name (action-name action))
	  (msg  (action-arg1 action)))
    (cond ((get-var name users)  ; USER SEND
	   (begin (cw "trying ~x0 ~x1~%" (get-var name users) msg)
		  (do-user-send action rest users hosts)))
          ((get-var name hosts)
	   (do-host-send action rest users hosts))
          (t (mv rest users hosts))))))

(defun add-mail-action (msg rest)
  (cons (mk-action
	 'mail
	 (email-user (message-sender msg))
	 msg)
	rest))

;;(do-deliver action list[actions] list[user] list[host]) -> 
(defun do-deliver (action rest users hosts)
  (begin
   (cw "Doing delivery of ~x0~%" (stringify action))
  (let* ((name (action-name action))
         (msg  (action-arg1 action)))
    (cond ((get-var name users)  ; USER DELIVER
           (mv-let (status new-msg new-user)
                   (user-deliver msg (get-var name users))
                   (let ((new-users (set-var name new-user users)))
                     (if (equal 'mail status)
			 (begin
			  (cw "acted as ~x0 ~x1, user: ~x2~%" status new-msg 
			      (email-user (message-sender msg)))
;			  (mv (add-user-mail-action 'mail new-msg rest)
			  (mv (add-user-mail-action 'send new-msg rest)
			      new-users
			      hosts))
                     (mv rest new-users hosts)))))
          (t (mv rest users hosts))))))

;;;(do-mail action list[actions] list[user] list[host]) -> 
(defun do-mail (action rest users hosts)
  (let* ((name (action-name action))
         (msg  (action-arg1 action)))
    (cond ((get-var name users)  ; USER MAIL
           (mv-let (status msg new-user)
                   (user-send msg (get-var name users))
                   (let ((new-users (set-var name new-user users)))
                     (if status
			 (begin
			  (cw "here be dragons (~x2) ~x0 with ~x1 ~%" (message-sender msg) msg status)
			  (mv 
			   (if (equal status 'mail)
			       (add-sender-mail-action 'mail
						       msg
						       rest)
			     (if (equal status 'deliver)
				  (add-user-mail-action 'deliver
														msg
							rest)
			       rest))
			   new-users
			   hosts))
		       (mv rest new-users hosts)))))
          (t (mv rest users hosts)))))


(defun do-action-cond (action rest users hosts)
  (let ((type   (action-type action)))
    (cond ((equal 'init    type)
	   (do-init    action rest users hosts))
	  ((equal 'send    type)
	   (do-send    action rest users hosts))
	  ((equal 'deliver type)
	   (do-deliver action rest users hosts))
	  ((equal 'mail type)
	   (do-mail    action rest users hosts))
	  ((equal 'command type)
	   (do-command action rest users hosts))
	  (t (begin
	      (cw "Can't execute command ~x0~%" action)
	      (mv rest users hosts))))))

;;(do-actions action list[actions] list[user] list[host]) -> 
(defun do-actions (actions count users hosts)
  (declare (xargs :measure (acl2-count count)))
  (if (and (> count 0) (integerp count))
      (begin
       (cw "do-actions: ~x0~%" (stringify (car actions)))
      (cond ((endp actions) (mv 'end users hosts))
            (t (let* ((action (car         actions))
                      (rest   (cdr         actions)))
		 (mv-let (new-actions new-users new-hosts)
			 (do-action-cond action rest users hosts)
			 (do-actions new-actions (- count 1) new-users new-hosts ))))))
    (mv 'error users hosts)))

(defun parse-actions (actions)
  (begin
   (cond ((endp actions) '())
	 (t 
	  (cons
	   (let ((act (car actions)))
	     (if (equal 'init (car act))
		 (mk-action 'init)
	       (if (equal 'command (cadr act))
		   (mk-action
		    'command
		    (car act)
		    (caddr act)
		    (cadddr act))
		 (mk-action
		  'send
		  (car act)
		  (msgtomessage (caddr act))))))
	   (parse-actions (cdr actions)))))))

;;(simulate-network list[actions])
;;sets up initial env and calls recursive simulation func 
(defun simulate-network (actions)
  (do-actions (parse-actions actions) 20 *users* *hosts*))

;;(simulate-network-with-env list[action] env)
;;
(defun simulate-network-with-env (actions env)
  (do-actions (parse-actions actions)
	      20
              (get-var 'users env) 
              (get-var 'hosts env)
              ))
