/*
 * Copyright (c) 2021, 2023 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sve_hybrid_fp32_mla_6x4VL_a64fx (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<float> A_arg,
    size_t M, size_t N, const float *B_ptr, IndirectOutputArg<float> output_arg,
    const float *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        float maxval = static_cast<float>(std::numeric_limits<float>::infinity());
        float minval = - static_cast<float>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const float *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
    } ka;

    unsigned long flags=0;
    void *output_ptr;
    void *input_ptr;

    if (output_arg.is_indirect) {
        output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<float>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p4.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 61f\n"
      "cmp %x[M], #0x4\n"
      "bgt 49f\n"
      "beq 37f\n"
      "cmp %x[M], #0x2\n"
      "bgt 25f\n"
      "beq 13f\n"
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0x0\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "incw x20\n"
      "whilelt p0.s, x20, x11\n"
      "cbz x12, 3f\n"
      "ld1w { z8.s }, p4/Z, [x12]\n"
      "ld1w { z9.s }, p4/Z, [x12, #1, MUL VL]\n"
      "ld1w { z10.s }, p4/Z, [x12, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x12, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "b 5f\n"
      "3:"  // Height 1: no bias
      "tbz %x[flags], #0, 4f\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "b 5f\n"
      "4:"  // Height 1: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "5:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "6:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 7f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 8f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "b 8f\n"
      "7:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "8:"  // Height 1: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ble 10f\n"
      "9:"  // Height 1: Multiply loop: Main loop
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "ld1w { z17.s }, p4/Z, [x10, #2, MUL VL]\n"
      "ld1w { z16.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "add x26, x26, #0x4\n"
      "fmla z10.s, p4/M, z17.s, z0.s\n"
      "fmla z11.s, p4/M, z16.s, z0.s\n"
      "subs x27, x27, #0x1\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "bgt 9b\n"
      "10:"  // Height 1: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "ld1w { z17.s }, p4/Z, [x10, #2, MUL VL]\n"
      "ld1w { z16.s }, p4/Z, [x10, #3, MUL VL]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "fmla z10.s, p4/M, z17.s, z0.s\n"
      "fmla z11.s, p4/M, z16.s, z0.s\n"
      "addvl x10, x10, #4\n"
      "bne 6b\n"
      "tbz %x[flags], #1, 11f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rw { z17.s }, p4/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z16.s }, p4/Z, [x20]\n"
      "fmin z8.s, p4/M, z8.s, z17.s\n"
      "fmin z9.s, p4/M, z9.s, z17.s\n"
      "fmin z10.s, p4/M, z10.s, z17.s\n"
      "fmin z11.s, p4/M, z11.s, z17.s\n"
      "fmax z8.s, p4/M, z8.s, z16.s\n"
      "fmax z9.s, p4/M, z9.s, z16.s\n"
      "fmax z10.s, p4/M, z10.s, z16.s\n"
      "fmax z11.s, p4/M, z11.s, z16.s\n"
      "11:"  // Height 1: No activation
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "12:"  // Height 1: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 2b\n"
      "b 74f\n"
      "13:"  // Height 2
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "14:"  // Height 2: Column loop
      "mov x20, #0x0\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "incw x20\n"
      "whilelt p0.s, x20, x11\n"
      "cbz x12, 15f\n"
      "ld1w { z8.s }, p4/Z, [x12]\n"
      "ld1w { z9.s }, p4/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1w { z10.s }, p4/Z, [x12, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "addvl x12, x12, #4\n"
      "b 17f\n"
      "15:"  // Height 2: no bias
      "tbz %x[flags], #0, 16f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x20, x9, x20, LSL #2\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p3/Z, [x20]\n"
      "ld1w { z13.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z14.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 17f\n"
      "16:"  // Height 2: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "17:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "18:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 19f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 20f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "b 20f\n"
      "19:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "20:"  // Height 2: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ble 22f\n"
      "21:"  // Height 2: Multiply loop: Main loop
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "ld1w { z17.s }, p4/Z, [x10, #2, MUL VL]\n"
      "add x26, x26, #0x4\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "ld1w { z16.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "subs x27, x27, #0x1\n"
      "add x25, x25, #0x4\n"
      "fmla z10.s, p4/M, z17.s, z0.s\n"
      "fmla z14.s, p4/M, z17.s, z1.s\n"
      "fmla z11.s, p4/M, z16.s, z0.s\n"
      "fmla z15.s, p4/M, z16.s, z1.s\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "bgt 21b\n"
      "22:"  // Height 2: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "ld1w { z17.s }, p4/Z, [x10, #2, MUL VL]\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "ld1w { z16.s }, p4/Z, [x10, #3, MUL VL]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "fmla z10.s, p4/M, z17.s, z0.s\n"
      "fmla z14.s, p4/M, z17.s, z1.s\n"
      "addvl x10, x10, #4\n"
      "fmla z11.s, p4/M, z16.s, z0.s\n"
      "fmla z15.s, p4/M, z16.s, z1.s\n"
      "bne 18b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #2\n"
      "tbz %x[flags], #1, 23f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rw { z17.s }, p4/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z16.s }, p4/Z, [x20]\n"
      "fmin z8.s, p4/M, z8.s, z17.s\n"
      "fmin z9.s, p4/M, z9.s, z17.s\n"
      "fmin z10.s, p4/M, z10.s, z17.s\n"
      "fmin z11.s, p4/M, z11.s, z17.s\n"
      "fmin z12.s, p4/M, z12.s, z17.s\n"
      "fmin z13.s, p4/M, z13.s, z17.s\n"
      "fmin z14.s, p4/M, z14.s, z17.s\n"
      "fmin z15.s, p4/M, z15.s, z17.s\n"
      "fmax z8.s, p4/M, z8.s, z16.s\n"
      "fmax z9.s, p4/M, z9.s, z16.s\n"
      "fmax z10.s, p4/M, z10.s, z16.s\n"
      "fmax z11.s, p4/M, z11.s, z16.s\n"
      "fmax z12.s, p4/M, z12.s, z16.s\n"
      "fmax z13.s, p4/M, z13.s, z16.s\n"
      "fmax z14.s, p4/M, z14.s, z16.s\n"
      "fmax z15.s, p4/M, z15.s, z16.s\n"
      "23:"  // Height 2: No activation
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z12.s }, p3, [x25]\n"
      "st1w { z13.s }, p2, [x25, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x25, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x25, #3, MUL VL]\n"
      "24:"  // Height 2: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 14b\n"
      "b 74f\n"
      "25:"  // Height 3
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "26:"  // Height 3: Column loop
      "mov x20, #0x0\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "incw x20\n"
      "whilelt p0.s, x20, x11\n"
      "cbz x12, 27f\n"
      "ld1w { z8.s }, p4/Z, [x12]\n"
      "ld1w { z9.s }, p4/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1w { z10.s }, p4/Z, [x12, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "addvl x12, x12, #4\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "b 29f\n"
      "27:"  // Height 3: no bias
      "tbz %x[flags], #0, 28f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x21, x9, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p3/Z, [x21]\n"
      "ld1w { z13.s }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1w { z14.s }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1w { z16.s }, p3/Z, [x20]\n"
      "ld1w { z17.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z18.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z19.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 29f\n"
      "28:"  // Height 3: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "29:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "30:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 31f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 32f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "b 32f\n"
      "31:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "32:"  // Height 3: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ble 34f\n"
      "33:"  // Height 3: Multiply loop: Main loop
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x1\n"
      "fmla z16.s, p4/M, z6.s, z2.s\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "ld1w { z21.s }, p4/Z, [x10, #2, MUL VL]\n"
      "add x25, x25, #0x4\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "fmla z17.s, p4/M, z7.s, z2.s\n"
      "ld1w { z20.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "add x24, x24, #0x4\n"
      "fmla z10.s, p4/M, z21.s, z0.s\n"
      "fmla z14.s, p4/M, z21.s, z1.s\n"
      "fmla z18.s, p4/M, z21.s, z2.s\n"
      "fmla z11.s, p4/M, z20.s, z0.s\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "fmla z15.s, p4/M, z20.s, z1.s\n"
      "fmla z19.s, p4/M, z20.s, z2.s\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "bgt 33b\n"
      "34:"  // Height 3: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "add x28, x28, #0x1\n"
      "fmla z16.s, p4/M, z6.s, z2.s\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "ld1w { z21.s }, p4/Z, [x10, #2, MUL VL]\n"
      "cmp x28, x20\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "fmla z17.s, p4/M, z7.s, z2.s\n"
      "ld1w { z20.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.s, p4/M, z21.s, z0.s\n"
      "fmla z14.s, p4/M, z21.s, z1.s\n"
      "fmla z18.s, p4/M, z21.s, z2.s\n"
      "fmla z11.s, p4/M, z20.s, z0.s\n"
      "fmla z15.s, p4/M, z20.s, z1.s\n"
      "fmla z19.s, p4/M, z20.s, z2.s\n"
      "bne 30b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "tbz %x[flags], #1, 35f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rw { z21.s }, p4/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z20.s }, p4/Z, [x20]\n"
      "fmin z8.s, p4/M, z8.s, z21.s\n"
      "fmin z9.s, p4/M, z9.s, z21.s\n"
      "fmin z10.s, p4/M, z10.s, z21.s\n"
      "fmin z11.s, p4/M, z11.s, z21.s\n"
      "fmin z12.s, p4/M, z12.s, z21.s\n"
      "fmin z13.s, p4/M, z13.s, z21.s\n"
      "fmin z14.s, p4/M, z14.s, z21.s\n"
      "fmin z15.s, p4/M, z15.s, z21.s\n"
      "fmin z16.s, p4/M, z16.s, z21.s\n"
      "fmin z17.s, p4/M, z17.s, z21.s\n"
      "fmin z18.s, p4/M, z18.s, z21.s\n"
      "fmin z19.s, p4/M, z19.s, z21.s\n"
      "fmax z8.s, p4/M, z8.s, z20.s\n"
      "fmax z9.s, p4/M, z9.s, z20.s\n"
      "fmax z10.s, p4/M, z10.s, z20.s\n"
      "fmax z11.s, p4/M, z11.s, z20.s\n"
      "fmax z12.s, p4/M, z12.s, z20.s\n"
      "fmax z13.s, p4/M, z13.s, z20.s\n"
      "fmax z14.s, p4/M, z14.s, z20.s\n"
      "fmax z15.s, p4/M, z15.s, z20.s\n"
      "fmax z16.s, p4/M, z16.s, z20.s\n"
      "fmax z17.s, p4/M, z17.s, z20.s\n"
      "fmax z18.s, p4/M, z18.s, z20.s\n"
      "fmax z19.s, p4/M, z19.s, z20.s\n"
      "35:"  // Height 3: No activation
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z12.s }, p3, [x25]\n"
      "st1w { z13.s }, p2, [x25, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x25, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x25, #3, MUL VL]\n"
      "st1w { z16.s }, p3, [x24]\n"
      "st1w { z17.s }, p2, [x24, #1, MUL VL]\n"
      "st1w { z18.s }, p1, [x24, #2, MUL VL]\n"
      "st1w { z19.s }, p0, [x24, #3, MUL VL]\n"
      "36:"  // Height 3: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 26b\n"
      "b 74f\n"
      "37:"  // Height 4
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "38:"  // Height 4: Column loop
      "mov x20, #0x0\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "incw x20\n"
      "whilelt p0.s, x20, x11\n"
      "cbz x12, 39f\n"
      "ld1w { z8.s }, p4/Z, [x12]\n"
      "ld1w { z9.s }, p4/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1w { z10.s }, p4/Z, [x12, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "addvl x12, x12, #4\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z20.d, z8.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "b 41f\n"
      "39:"  // Height 4: no bias
      "tbz %x[flags], #0, 40f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x22, x9, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p3/Z, [x22]\n"
      "ld1w { z13.s }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1w { z14.s }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1w { z16.s }, p3/Z, [x21]\n"
      "ld1w { z17.s }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1w { z18.s }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1w { z19.s }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1w { z20.s }, p3/Z, [x20]\n"
      "ld1w { z21.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z22.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z23.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 41f\n"
      "40:"  // Height 4: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "41:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "42:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 43f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 44f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "b 44f\n"
      "43:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "44:"  // Height 4: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ble 46f\n"
      "45:"  // Height 4: Multiply loop: Main loop
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x1\n"
      "fmla z16.s, p4/M, z6.s, z2.s\n"
      "fmla z20.s, p4/M, z6.s, z3.s\n"
      "ld1w { z25.s }, p4/Z, [x10, #2, MUL VL]\n"
      "add x25, x25, #0x4\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "add x24, x24, #0x4\n"
      "add x23, x23, #0x4\n"
      "fmla z17.s, p4/M, z7.s, z2.s\n"
      "fmla z21.s, p4/M, z7.s, z3.s\n"
      "ld1w { z24.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.s, p4/M, z25.s, z0.s\n"
      "fmla z14.s, p4/M, z25.s, z1.s\n"
      "fmla z18.s, p4/M, z25.s, z2.s\n"
      "fmla z22.s, p4/M, z25.s, z3.s\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "fmla z11.s, p4/M, z24.s, z0.s\n"
      "fmla z15.s, p4/M, z24.s, z1.s\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "fmla z19.s, p4/M, z24.s, z2.s\n"
      "fmla z23.s, p4/M, z24.s, z3.s\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "bgt 45b\n"
      "46:"  // Height 4: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "add x28, x28, #0x1\n"
      "fmla z16.s, p4/M, z6.s, z2.s\n"
      "fmla z20.s, p4/M, z6.s, z3.s\n"
      "ld1w { z25.s }, p4/Z, [x10, #2, MUL VL]\n"
      "cmp x28, x20\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "fmla z17.s, p4/M, z7.s, z2.s\n"
      "fmla z21.s, p4/M, z7.s, z3.s\n"
      "ld1w { z24.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.s, p4/M, z25.s, z0.s\n"
      "fmla z14.s, p4/M, z25.s, z1.s\n"
      "fmla z18.s, p4/M, z25.s, z2.s\n"
      "fmla z22.s, p4/M, z25.s, z3.s\n"
      "fmla z11.s, p4/M, z24.s, z0.s\n"
      "fmla z15.s, p4/M, z24.s, z1.s\n"
      "fmla z19.s, p4/M, z24.s, z2.s\n"
      "fmla z23.s, p4/M, z24.s, z3.s\n"
      "bne 42b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "tbz %x[flags], #1, 47f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rw { z25.s }, p4/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z24.s }, p4/Z, [x20]\n"
      "fmin z8.s, p4/M, z8.s, z25.s\n"
      "fmin z9.s, p4/M, z9.s, z25.s\n"
      "fmin z10.s, p4/M, z10.s, z25.s\n"
      "fmin z11.s, p4/M, z11.s, z25.s\n"
      "fmin z12.s, p4/M, z12.s, z25.s\n"
      "fmin z13.s, p4/M, z13.s, z25.s\n"
      "fmin z14.s, p4/M, z14.s, z25.s\n"
      "fmin z15.s, p4/M, z15.s, z25.s\n"
      "fmin z16.s, p4/M, z16.s, z25.s\n"
      "fmin z17.s, p4/M, z17.s, z25.s\n"
      "fmin z18.s, p4/M, z18.s, z25.s\n"
      "fmin z19.s, p4/M, z19.s, z25.s\n"
      "fmin z20.s, p4/M, z20.s, z25.s\n"
      "fmin z21.s, p4/M, z21.s, z25.s\n"
      "fmin z22.s, p4/M, z22.s, z25.s\n"
      "fmin z23.s, p4/M, z23.s, z25.s\n"
      "fmax z8.s, p4/M, z8.s, z24.s\n"
      "fmax z9.s, p4/M, z9.s, z24.s\n"
      "fmax z10.s, p4/M, z10.s, z24.s\n"
      "fmax z11.s, p4/M, z11.s, z24.s\n"
      "fmax z12.s, p4/M, z12.s, z24.s\n"
      "fmax z13.s, p4/M, z13.s, z24.s\n"
      "fmax z14.s, p4/M, z14.s, z24.s\n"
      "fmax z15.s, p4/M, z15.s, z24.s\n"
      "fmax z16.s, p4/M, z16.s, z24.s\n"
      "fmax z17.s, p4/M, z17.s, z24.s\n"
      "fmax z18.s, p4/M, z18.s, z24.s\n"
      "fmax z19.s, p4/M, z19.s, z24.s\n"
      "fmax z20.s, p4/M, z20.s, z24.s\n"
      "fmax z21.s, p4/M, z21.s, z24.s\n"
      "fmax z22.s, p4/M, z22.s, z24.s\n"
      "fmax z23.s, p4/M, z23.s, z24.s\n"
      "47:"  // Height 4: No activation
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z12.s }, p3, [x25]\n"
      "st1w { z13.s }, p2, [x25, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x25, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x25, #3, MUL VL]\n"
      "st1w { z16.s }, p3, [x24]\n"
      "st1w { z17.s }, p2, [x24, #1, MUL VL]\n"
      "st1w { z18.s }, p1, [x24, #2, MUL VL]\n"
      "st1w { z19.s }, p0, [x24, #3, MUL VL]\n"
      "st1w { z20.s }, p3, [x23]\n"
      "st1w { z21.s }, p2, [x23, #1, MUL VL]\n"
      "st1w { z22.s }, p1, [x23, #2, MUL VL]\n"
      "st1w { z23.s }, p0, [x23, #3, MUL VL]\n"
      "48:"  // Height 4: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 38b\n"
      "b 74f\n"
      "49:"  // Height 5
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "50:"  // Height 5: Column loop
      "mov x20, #0x0\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "incw x20\n"
      "whilelt p0.s, x20, x11\n"
      "cbz x12, 51f\n"
      "ld1w { z8.s }, p4/Z, [x12]\n"
      "ld1w { z9.s }, p4/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1w { z10.s }, p4/Z, [x12, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "addvl x12, x12, #4\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z20.d, z8.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "mov z24.d, z8.d\n"
      "mov z25.d, z9.d\n"
      "mov z26.d, z10.d\n"
      "mov z27.d, z11.d\n"
      "b 53f\n"
      "51:"  // Height 5: no bias
      "tbz %x[flags], #0, 52f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x23, x9, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "add x21, x22, x20, LSL #2\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p3/Z, [x23]\n"
      "ld1w { z13.s }, p2/Z, [x23, #1, MUL VL]\n"
      "ld1w { z14.s }, p1/Z, [x23, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x23, #3, MUL VL]\n"
      "ld1w { z16.s }, p3/Z, [x22]\n"
      "ld1w { z17.s }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1w { z18.s }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1w { z19.s }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1w { z20.s }, p3/Z, [x21]\n"
      "ld1w { z21.s }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1w { z22.s }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1w { z23.s }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1w { z24.s }, p3/Z, [x20]\n"
      "ld1w { z25.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z26.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z27.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 53f\n"
      "52:"  // Height 5: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "53:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "54:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 55f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 56f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "b 56f\n"
      "55:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "56:"  // Height 5: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1rw { z4.s }, p4/Z, [x22]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ble 58f\n"
      "57:"  // Height 5: Multiply loop: Main loop
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x1\n"
      "fmla z16.s, p4/M, z6.s, z2.s\n"
      "fmla z20.s, p4/M, z6.s, z3.s\n"
      "add x25, x25, #0x4\n"
      "add x24, x24, #0x4\n"
      "fmla z24.s, p4/M, z6.s, z4.s\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "ld1w { z29.s }, p4/Z, [x10, #2, MUL VL]\n"
      "add x23, x23, #0x4\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "fmla z17.s, p4/M, z7.s, z2.s\n"
      "add x22, x22, #0x4\n"
      "fmla z21.s, p4/M, z7.s, z3.s\n"
      "fmla z25.s, p4/M, z7.s, z4.s\n"
      "ld1w { z28.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.s, p4/M, z29.s, z0.s\n"
      "fmla z14.s, p4/M, z29.s, z1.s\n"
      "fmla z18.s, p4/M, z29.s, z2.s\n"
      "fmla z22.s, p4/M, z29.s, z3.s\n"
      "fmla z26.s, p4/M, z29.s, z4.s\n"
      "fmla z11.s, p4/M, z28.s, z0.s\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "fmla z15.s, p4/M, z28.s, z1.s\n"
      "fmla z19.s, p4/M, z28.s, z2.s\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "fmla z23.s, p4/M, z28.s, z3.s\n"
      "fmla z27.s, p4/M, z28.s, z4.s\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1rw { z4.s }, p4/Z, [x22]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "bgt 57b\n"
      "58:"  // Height 5: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "add x28, x28, #0x1\n"
      "fmla z16.s, p4/M, z6.s, z2.s\n"
      "fmla z20.s, p4/M, z6.s, z3.s\n"
      "cmp x28, x20\n"
      "fmla z24.s, p4/M, z6.s, z4.s\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "ld1w { z29.s }, p4/Z, [x10, #2, MUL VL]\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "fmla z17.s, p4/M, z7.s, z2.s\n"
      "fmla z21.s, p4/M, z7.s, z3.s\n"
      "fmla z25.s, p4/M, z7.s, z4.s\n"
      "ld1w { z28.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.s, p4/M, z29.s, z0.s\n"
      "fmla z14.s, p4/M, z29.s, z1.s\n"
      "fmla z18.s, p4/M, z29.s, z2.s\n"
      "fmla z22.s, p4/M, z29.s, z3.s\n"
      "fmla z26.s, p4/M, z29.s, z4.s\n"
      "fmla z11.s, p4/M, z28.s, z0.s\n"
      "fmla z15.s, p4/M, z28.s, z1.s\n"
      "fmla z19.s, p4/M, z28.s, z2.s\n"
      "fmla z23.s, p4/M, z28.s, z3.s\n"
      "fmla z27.s, p4/M, z28.s, z4.s\n"
      "bne 54b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "tbz %x[flags], #1, 59f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rw { z29.s }, p4/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z28.s }, p4/Z, [x20]\n"
      "fmin z8.s, p4/M, z8.s, z29.s\n"
      "fmin z9.s, p4/M, z9.s, z29.s\n"
      "fmin z10.s, p4/M, z10.s, z29.s\n"
      "fmin z11.s, p4/M, z11.s, z29.s\n"
      "fmin z12.s, p4/M, z12.s, z29.s\n"
      "fmin z13.s, p4/M, z13.s, z29.s\n"
      "fmin z14.s, p4/M, z14.s, z29.s\n"
      "fmin z15.s, p4/M, z15.s, z29.s\n"
      "fmin z16.s, p4/M, z16.s, z29.s\n"
      "fmin z17.s, p4/M, z17.s, z29.s\n"
      "fmin z18.s, p4/M, z18.s, z29.s\n"
      "fmin z19.s, p4/M, z19.s, z29.s\n"
      "fmin z20.s, p4/M, z20.s, z29.s\n"
      "fmin z21.s, p4/M, z21.s, z29.s\n"
      "fmin z22.s, p4/M, z22.s, z29.s\n"
      "fmin z23.s, p4/M, z23.s, z29.s\n"
      "fmin z24.s, p4/M, z24.s, z29.s\n"
      "fmin z25.s, p4/M, z25.s, z29.s\n"
      "fmin z26.s, p4/M, z26.s, z29.s\n"
      "fmin z27.s, p4/M, z27.s, z29.s\n"
      "fmax z8.s, p4/M, z8.s, z28.s\n"
      "fmax z9.s, p4/M, z9.s, z28.s\n"
      "fmax z10.s, p4/M, z10.s, z28.s\n"
      "fmax z11.s, p4/M, z11.s, z28.s\n"
      "fmax z12.s, p4/M, z12.s, z28.s\n"
      "fmax z13.s, p4/M, z13.s, z28.s\n"
      "fmax z14.s, p4/M, z14.s, z28.s\n"
      "fmax z15.s, p4/M, z15.s, z28.s\n"
      "fmax z16.s, p4/M, z16.s, z28.s\n"
      "fmax z17.s, p4/M, z17.s, z28.s\n"
      "fmax z18.s, p4/M, z18.s, z28.s\n"
      "fmax z19.s, p4/M, z19.s, z28.s\n"
      "fmax z20.s, p4/M, z20.s, z28.s\n"
      "fmax z21.s, p4/M, z21.s, z28.s\n"
      "fmax z22.s, p4/M, z22.s, z28.s\n"
      "fmax z23.s, p4/M, z23.s, z28.s\n"
      "fmax z24.s, p4/M, z24.s, z28.s\n"
      "fmax z25.s, p4/M, z25.s, z28.s\n"
      "fmax z26.s, p4/M, z26.s, z28.s\n"
      "fmax z27.s, p4/M, z27.s, z28.s\n"
      "59:"  // Height 5: No activation
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z12.s }, p3, [x25]\n"
      "st1w { z13.s }, p2, [x25, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x25, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x25, #3, MUL VL]\n"
      "st1w { z16.s }, p3, [x24]\n"
      "st1w { z17.s }, p2, [x24, #1, MUL VL]\n"
      "st1w { z18.s }, p1, [x24, #2, MUL VL]\n"
      "st1w { z19.s }, p0, [x24, #3, MUL VL]\n"
      "st1w { z20.s }, p3, [x23]\n"
      "st1w { z21.s }, p2, [x23, #1, MUL VL]\n"
      "st1w { z22.s }, p1, [x23, #2, MUL VL]\n"
      "st1w { z23.s }, p0, [x23, #3, MUL VL]\n"
      "st1w { z24.s }, p3, [x22]\n"
      "st1w { z25.s }, p2, [x22, #1, MUL VL]\n"
      "st1w { z26.s }, p1, [x22, #2, MUL VL]\n"
      "st1w { z27.s }, p0, [x22, #3, MUL VL]\n"
      "60:"  // Height 5: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 50b\n"
      "b 74f\n"
      "61:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "mov x20, #0x18\n"
      "mov x12, %x[bias]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "mov x9, %x[output_ptr]\n"
      "madd %x[output_ptr], x21, x20, %x[output_ptr]\n"
      "62:"  // Height 6: Column loop
      "mov x20, #0x0\n"
      "whilelt p3.s, x20, x11\n"
      "incw x20\n"
      "whilelt p2.s, x20, x11\n"
      "incw x20\n"
      "whilelt p1.s, x20, x11\n"
      "incw x20\n"
      "whilelt p0.s, x20, x11\n"
      "cbz x12, 63f\n"
      "ld1w { z8.s }, p4/Z, [x12]\n"
      "ld1w { z9.s }, p4/Z, [x12, #1, MUL VL]\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "ld1w { z10.s }, p4/Z, [x12, #2, MUL VL]\n"
      "ld1w { z11.s }, p4/Z, [x12, #3, MUL VL]\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "addvl x12, x12, #4\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z20.d, z8.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "mov z24.d, z8.d\n"
      "mov z25.d, z9.d\n"
      "mov z26.d, z10.d\n"
      "mov z27.d, z11.d\n"
      "mov z28.d, z8.d\n"
      "mov z29.d, z9.d\n"
      "mov z30.d, z10.d\n"
      "mov z31.d, z11.d\n"
      "b 65f\n"
      "63:"  // Height 6: no bias
      "tbz %x[flags], #0, 64f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x24, x9, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "ld1w { z8.s }, p3/Z, [x9]\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "ld1w { z9.s }, p2/Z, [x9, #1, MUL VL]\n"
      "ld1w { z10.s }, p1/Z, [x9, #2, MUL VL]\n"
      "add x20, x21, x20, LSL #2\n"
      "ld1w { z11.s }, p0/Z, [x9, #3, MUL VL]\n"
      "ld1w { z12.s }, p3/Z, [x24]\n"
      "ld1w { z13.s }, p2/Z, [x24, #1, MUL VL]\n"
      "ld1w { z14.s }, p1/Z, [x24, #2, MUL VL]\n"
      "ld1w { z15.s }, p0/Z, [x24, #3, MUL VL]\n"
      "ld1w { z16.s }, p3/Z, [x23]\n"
      "ld1w { z17.s }, p2/Z, [x23, #1, MUL VL]\n"
      "ld1w { z18.s }, p1/Z, [x23, #2, MUL VL]\n"
      "ld1w { z19.s }, p0/Z, [x23, #3, MUL VL]\n"
      "ld1w { z20.s }, p3/Z, [x22]\n"
      "ld1w { z21.s }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1w { z22.s }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1w { z23.s }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1w { z24.s }, p3/Z, [x21]\n"
      "ld1w { z25.s }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1w { z26.s }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1w { z27.s }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1w { z28.s }, p3/Z, [x20]\n"
      "ld1w { z29.s }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1w { z30.s }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1w { z31.s }, p0/Z, [x20, #3, MUL VL]\n"
      "b 65f\n"
      "64:"  // Height 6: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "mov z29.b, #0x0\n"
      "mov z30.b, #0x0\n"
      "mov z31.b, #0x0\n"
      "65:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "66:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 67f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 68f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #2\n"
      "add x25, x25, x20, LSL #2\n"
      "add x24, x24, x20, LSL #2\n"
      "add x23, x23, x20, LSL #2\n"
      "add x22, x22, x20, LSL #2\n"
      "add x21, x21, x20, LSL #2\n"
      "b 68f\n"
      "67:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #2\n"
      "add x24, x25, x21, LSL #2\n"
      "add x23, x24, x21, LSL #2\n"
      "add x22, x23, x21, LSL #2\n"
      "add x21, x22, x21, LSL #2\n"
      "68:"  // Height 6: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "ld1rw { z4.s }, p4/Z, [x22]\n"
      "ld1rw { z5.s }, p4/Z, [x21]\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "ble 70f\n"
      "69:"  // Height 6: Multiply loop: Main loop
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "add x26, x26, #0x4\n"
      "subs x27, x27, #0x1\n"
      "fmla z16.s, p4/M, z6.s, z2.s\n"
      "fmla z20.s, p4/M, z6.s, z3.s\n"
      "add x25, x25, #0x4\n"
      "add x24, x24, #0x4\n"
      "fmla z24.s, p4/M, z6.s, z4.s\n"
      "fmla z28.s, p4/M, z6.s, z5.s\n"
      "ld1w { z6.s }, p4/Z, [x10, #2, MUL VL]\n"
      "add x23, x23, #0x4\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "add x22, x22, #0x4\n"
      "add x21, x21, #0x4\n"
      "fmla z17.s, p4/M, z7.s, z2.s\n"
      "fmla z21.s, p4/M, z7.s, z3.s\n"
      "fmla z25.s, p4/M, z7.s, z4.s\n"
      "fmla z29.s, p4/M, z7.s, z5.s\n"
      "ld1w { z7.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.s, p4/M, z6.s, z0.s\n"
      "fmla z14.s, p4/M, z6.s, z1.s\n"
      "fmla z18.s, p4/M, z6.s, z2.s\n"
      "fmla z22.s, p4/M, z6.s, z3.s\n"
      "fmla z26.s, p4/M, z6.s, z4.s\n"
      "fmla z30.s, p4/M, z6.s, z5.s\n"
      "ld1w { z6.s }, p4/Z, [x10]\n"
      "fmla z11.s, p4/M, z7.s, z0.s\n"
      "fmla z15.s, p4/M, z7.s, z1.s\n"
      "ld1rw { z0.s }, p4/Z, [x26]\n"
      "ld1rw { z1.s }, p4/Z, [x25]\n"
      "fmla z19.s, p4/M, z7.s, z2.s\n"
      "fmla z23.s, p4/M, z7.s, z3.s\n"
      "ld1rw { z2.s }, p4/Z, [x24]\n"
      "ld1rw { z3.s }, p4/Z, [x23]\n"
      "fmla z27.s, p4/M, z7.s, z4.s\n"
      "fmla z31.s, p4/M, z7.s, z5.s\n"
      "ld1rw { z4.s }, p4/Z, [x22]\n"
      "ld1rw { z5.s }, p4/Z, [x21]\n"
      "ld1w { z7.s }, p4/Z, [x10, #1, MUL VL]\n"
      "bgt 69b\n"
      "70:"  // Height 6: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.s, p4/M, z6.s, z0.s\n"
      "fmla z12.s, p4/M, z6.s, z1.s\n"
      "add x28, x28, #0x1\n"
      "fmla z16.s, p4/M, z6.s, z2.s\n"
      "fmla z20.s, p4/M, z6.s, z3.s\n"
      "cmp x28, x20\n"
      "fmla z24.s, p4/M, z6.s, z4.s\n"
      "fmla z28.s, p4/M, z6.s, z5.s\n"
      "ld1w { z6.s }, p4/Z, [x10, #2, MUL VL]\n"
      "fmla z9.s, p4/M, z7.s, z0.s\n"
      "fmla z13.s, p4/M, z7.s, z1.s\n"
      "fmla z17.s, p4/M, z7.s, z2.s\n"
      "fmla z21.s, p4/M, z7.s, z3.s\n"
      "fmla z25.s, p4/M, z7.s, z4.s\n"
      "fmla z29.s, p4/M, z7.s, z5.s\n"
      "ld1w { z7.s }, p4/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "fmla z10.s, p4/M, z6.s, z0.s\n"
      "fmla z14.s, p4/M, z6.s, z1.s\n"
      "fmla z18.s, p4/M, z6.s, z2.s\n"
      "fmla z22.s, p4/M, z6.s, z3.s\n"
      "fmla z26.s, p4/M, z6.s, z4.s\n"
      "fmla z30.s, p4/M, z6.s, z5.s\n"
      "fmla z11.s, p4/M, z7.s, z0.s\n"
      "fmla z15.s, p4/M, z7.s, z1.s\n"
      "fmla z19.s, p4/M, z7.s, z2.s\n"
      "fmla z23.s, p4/M, z7.s, z3.s\n"
      "fmla z27.s, p4/M, z7.s, z4.s\n"
      "fmla z31.s, p4/M, z7.s, z5.s\n"
      "bne 66b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x25, x9, x20, LSL #2\n"
      "add x24, x25, x20, LSL #2\n"
      "add x23, x24, x20, LSL #2\n"
      "add x22, x23, x20, LSL #2\n"
      "add x21, x22, x20, LSL #2\n"
      "tbz %x[flags], #1, 71f\n"
      "add x20, %x[args_ptr], %[offset_max]\n"
      "ld1rw { z1.s }, p4/Z, [x20]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rw { z0.s }, p4/Z, [x20]\n"
      "fmin z8.s, p4/M, z8.s, z1.s\n"
      "fmin z9.s, p4/M, z9.s, z1.s\n"
      "fmin z10.s, p4/M, z10.s, z1.s\n"
      "fmin z11.s, p4/M, z11.s, z1.s\n"
      "fmin z12.s, p4/M, z12.s, z1.s\n"
      "fmin z13.s, p4/M, z13.s, z1.s\n"
      "fmin z14.s, p4/M, z14.s, z1.s\n"
      "fmin z15.s, p4/M, z15.s, z1.s\n"
      "fmin z16.s, p4/M, z16.s, z1.s\n"
      "fmin z17.s, p4/M, z17.s, z1.s\n"
      "fmin z18.s, p4/M, z18.s, z1.s\n"
      "fmin z19.s, p4/M, z19.s, z1.s\n"
      "fmin z20.s, p4/M, z20.s, z1.s\n"
      "fmin z21.s, p4/M, z21.s, z1.s\n"
      "fmin z22.s, p4/M, z22.s, z1.s\n"
      "fmin z23.s, p4/M, z23.s, z1.s\n"
      "fmin z24.s, p4/M, z24.s, z1.s\n"
      "fmin z25.s, p4/M, z25.s, z1.s\n"
      "fmin z26.s, p4/M, z26.s, z1.s\n"
      "fmin z27.s, p4/M, z27.s, z1.s\n"
      "fmin z28.s, p4/M, z28.s, z1.s\n"
      "fmin z29.s, p4/M, z29.s, z1.s\n"
      "fmin z30.s, p4/M, z30.s, z1.s\n"
      "fmin z31.s, p4/M, z31.s, z1.s\n"
      "fmax z8.s, p4/M, z8.s, z0.s\n"
      "fmax z9.s, p4/M, z9.s, z0.s\n"
      "fmax z10.s, p4/M, z10.s, z0.s\n"
      "fmax z11.s, p4/M, z11.s, z0.s\n"
      "fmax z12.s, p4/M, z12.s, z0.s\n"
      "fmax z13.s, p4/M, z13.s, z0.s\n"
      "fmax z14.s, p4/M, z14.s, z0.s\n"
      "fmax z15.s, p4/M, z15.s, z0.s\n"
      "fmax z16.s, p4/M, z16.s, z0.s\n"
      "fmax z17.s, p4/M, z17.s, z0.s\n"
      "fmax z18.s, p4/M, z18.s, z0.s\n"
      "fmax z19.s, p4/M, z19.s, z0.s\n"
      "fmax z20.s, p4/M, z20.s, z0.s\n"
      "fmax z21.s, p4/M, z21.s, z0.s\n"
      "fmax z22.s, p4/M, z22.s, z0.s\n"
      "fmax z23.s, p4/M, z23.s, z0.s\n"
      "fmax z24.s, p4/M, z24.s, z0.s\n"
      "fmax z25.s, p4/M, z25.s, z0.s\n"
      "fmax z26.s, p4/M, z26.s, z0.s\n"
      "fmax z27.s, p4/M, z27.s, z0.s\n"
      "fmax z28.s, p4/M, z28.s, z0.s\n"
      "fmax z29.s, p4/M, z29.s, z0.s\n"
      "fmax z30.s, p4/M, z30.s, z0.s\n"
      "fmax z31.s, p4/M, z31.s, z0.s\n"
      "71:"  // Height 6: No activation
      "st1w { z8.s }, p3, [x9]\n"
      "st1w { z9.s }, p2, [x9, #1, MUL VL]\n"
      "st1w { z10.s }, p1, [x9, #2, MUL VL]\n"
      "st1w { z11.s }, p0, [x9, #3, MUL VL]\n"
      "addvl x9, x9, #4\n"
      "st1w { z12.s }, p3, [x25]\n"
      "st1w { z13.s }, p2, [x25, #1, MUL VL]\n"
      "st1w { z14.s }, p1, [x25, #2, MUL VL]\n"
      "st1w { z15.s }, p0, [x25, #3, MUL VL]\n"
      "st1w { z16.s }, p3, [x24]\n"
      "st1w { z17.s }, p2, [x24, #1, MUL VL]\n"
      "st1w { z18.s }, p1, [x24, #2, MUL VL]\n"
      "st1w { z19.s }, p0, [x24, #3, MUL VL]\n"
      "st1w { z20.s }, p3, [x23]\n"
      "st1w { z21.s }, p2, [x23, #1, MUL VL]\n"
      "st1w { z22.s }, p1, [x23, #2, MUL VL]\n"
      "st1w { z23.s }, p0, [x23, #3, MUL VL]\n"
      "st1w { z24.s }, p3, [x22]\n"
      "st1w { z25.s }, p2, [x22, #1, MUL VL]\n"
      "st1w { z26.s }, p1, [x22, #2, MUL VL]\n"
      "st1w { z27.s }, p0, [x22, #3, MUL VL]\n"
      "st1w { z28.s }, p3, [x21]\n"
      "st1w { z29.s }, p2, [x21, #1, MUL VL]\n"
      "st1w { z30.s }, p1, [x21, #2, MUL VL]\n"
      "st1w { z31.s }, p0, [x21, #3, MUL VL]\n"
      "72:"  // Height 6: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 62b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 74f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 73f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "73:"  // Update direct input
      "mov x20, #0x18\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "74:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr), [output_ptr] "+&r" (output_ptr)
      : [args_ptr] "r" (&ka), [bias] "r" (bias), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "x9", "x10", "x11", "x12", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
