//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Instrument/AlphaScanEditor.cpp
//! @brief     Defines class AlphaScanEditor
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Instrument/AlphaScanEditor.h"
#include "GUI/Model/Beam/GrazingScanItem.h"
#include "GUI/Model/Descriptor/DistributionItems.h"
#include "GUI/View/Device/SphericalAxisForm.h"
#include "GUI/View/Instrument/DistributionEditor.h"
#include "GUI/View/Instrument/DistributionPlot.h"
#include "GUI/View/Tool/GroupBoxCollapser.h"

AlphaScanEditor::AlphaScanEditor(QWidget* parent, GrazingScanItem* item)
    : QGroupBox("Grazing angles", parent)
    , m_item(item)
{
    auto* hLayout = new QHBoxLayout(this);
    hLayout->setSpacing(50);

    auto* gform = new QFormLayout;
    gform->setSpacing(6);
    hLayout->addLayout(gform);

    //... axis type combo
    auto* typeComboBox = new QComboBox(this);
    typeComboBox->addItem("Uniform axis");
    typeComboBox->addItem("Non-uniform axis"); // for use with experimental data ?
    const int idx = m_item->pointwiseAlphaAxisSelected() ? 1 : 0;
    gform->addRow("Axis type:", typeComboBox);

    //... axis parameters
    m_form = new SphericalAxisForm(gform, this);
    connect(m_form, &SphericalAxisForm::dataChanged, this, &AlphaScanEditor::dataChanged);

    //... beam distribution
    m_selector =
        new DistributionSelector(std::nullopt, GUI::ID::Distributions::Symmetric, this, m_item);
    connect(m_selector, &DistributionSelector::distributionChanged, this,
            &AlphaScanEditor::dataChanged);
    connect(m_selector, &DistributionSelector::distributionChanged, this,
            &AlphaScanEditor::updatePlot);
    gform->addRow(m_selector);

    //... update axis type combo (needs m_form)
    typeComboBox->setCurrentIndex(idx);
    onAxisTypeSelected(idx); // enable currently selected axis

    typeComboBox->setEnabled(m_item->pointwiseAlphaAxisDefined());
    connect(typeComboBox, qOverload<int>(&QComboBox::currentIndexChanged), this,
            &AlphaScanEditor::onAxisTypeSelected);

    //... distribution plot
    m_plot = new DistributionPlot(this);
    m_plot->setFixedSize(280, 170);
    m_plot->setShowMouseCoords(false);
    hLayout->addWidget(m_plot);
    hLayout->setAlignment(m_plot, Qt::AlignTop);
    hLayout->addStretch(1);

    auto* collapser = GroupBoxCollapser::installIntoGroupBox(this);
    collapser->setExpanded(item->isExpandGroupBox());
    connect(collapser, &GroupBoxCollapser::toggled, this,
            [item](bool b) { item->setExpandGroupBox(b); });

    updatePlot();
}

void AlphaScanEditor::updateIndicators()
{
    m_form->updateData();
}

void AlphaScanEditor::onAxisTypeSelected(int index)
{
    if (m_item) {
        if (index == 0 && m_item->pointwiseAlphaAxisSelected()) {
            m_item->selectUniformAxis();
            emit dataChanged();
        } else if (index == 1 && !m_item->pointwiseAlphaAxisSelected()) {
            m_item->selectListScan();
            emit dataChanged();
        }

        m_form->setAxisItem(m_item->alphaAxisItem());
        m_form->setEnabled(index == 0);
    }
}

void AlphaScanEditor::updatePlot()
{
    auto* d = m_selector->item()->distributionItem();
    m_plot->setVisible(!dynamic_cast<const DistributionNoneItem*>(d));
    m_plot->setDistItem(d);
    m_plot->plotItem();
}
