#include "GUI/Model/Sample/MaterialItem.h"
#include "GUI/Model/Sample/MaterialModel.h"
#include "GUI/Support/XML/Backup.h"
#include "Tests/GTestWrapper/google_test.h"
#include <memory>

TEST(TestMaterialModel, addRefractiveMaterial)
{
    std::unique_ptr<MaterialModel> model(new MaterialModel);

    EXPECT_TRUE(model->materialItems().isEmpty());

    const double delta(0.2), beta(0.1);
    const QString name("MaterialName");
    MaterialItem* material = model->addRefractiveMaterialItem(name, delta, beta);

    EXPECT_EQ(model->materialItems().size(), 1);

    EXPECT_EQ(material->matItemName(), name);
    EXPECT_TRUE(material->hasRefractiveIndex());
    EXPECT_EQ(material->delta(), delta);
    EXPECT_EQ(material->beta(), beta);
}

TEST(TestMaterialModel, addSLDMaterial)
{
    std::unique_ptr<MaterialModel> model(new MaterialModel);

    EXPECT_TRUE(model->materialItems().isEmpty());

    const double sld_real(0.2), sld_imag(0.1);
    const QString name("MaterialName");
    MaterialItem* material = model->addSLDMaterialItem(name, sld_real, sld_imag);

    EXPECT_EQ(model->materialItems().size(), 1);

    EXPECT_EQ(material->matItemName(), name);
    EXPECT_FALSE(material->hasRefractiveIndex());
    EXPECT_EQ(material->sldRe(), sld_real);
    EXPECT_EQ(material->sldIm(), sld_imag);
}

TEST(TestMaterialModel, cloneMaterialRefractive)
{
    std::unique_ptr<MaterialModel> model(new MaterialModel);

    EXPECT_TRUE(model->materialItems().isEmpty());

    const double delta(0.2), beta(0.1);
    const QString name("MaterialName");
    MaterialItem* material = model->addRefractiveMaterialItem(name, delta, beta);
    const QString origIdentifier = material->identifier();

    MaterialItem* clonedMaterial = model->copyMaterialItem(*material);
    EXPECT_EQ(model->materialItems().size(), 2);

    // clone should not change identifier of original material (as it once happened)
    EXPECT_EQ(material->identifier(), origIdentifier);

    // cloned material should have different identifier
    EXPECT_NE(clonedMaterial->identifier(), material->identifier());

    // checking name of cloned material
    EXPECT_EQ(material->matItemName() + " (copy)", clonedMaterial->matItemName());
    EXPECT_EQ(material->hasRefractiveIndex(), clonedMaterial->hasRefractiveIndex());
    EXPECT_EQ(material->delta(), delta);
    EXPECT_EQ(material->beta(), beta);
}

TEST(TestMaterialModel, cloneMaterialSLD)
{
    std::unique_ptr<MaterialModel> model(new MaterialModel);

    EXPECT_TRUE(model->materialItems().isEmpty());

    const double real(0.7), imag(0.5);
    const QString name("MaterialName");
    MaterialItem* material = model->addSLDMaterialItem(name, real, imag);
    const QString origIdentifier = material->identifier();

    MaterialItem* clonedMaterial = model->copyMaterialItem(*material);
    EXPECT_EQ(model->materialItems().size(), 2);

    // clone should not change identifier of original material (as it once happened)
    EXPECT_EQ(material->identifier(), origIdentifier);

    // cloned material should have different identifier
    EXPECT_NE(clonedMaterial->identifier(), material->identifier());

    // checking name of cloned material
    EXPECT_EQ(material->matItemName() + " (copy)", clonedMaterial->matItemName());
    EXPECT_EQ(material->hasRefractiveIndex(), clonedMaterial->hasRefractiveIndex());
    EXPECT_EQ(material->sldRe(), real);
    EXPECT_EQ(material->sldIm(), imag);
}

//! Checks the method which returns MaterialItem from known identifier.

TEST(TestMaterialModel, materialFromIdentifier)
{
    MaterialModel model;
    MaterialItem* material1 = model.addRefractiveMaterialItem("aaa", 1.0, 2.0);
    MaterialItem* material2 = model.addRefractiveMaterialItem("bbb", 3.0, 4.0);
    EXPECT_EQ(material1, model.materialItemFromIdentifier(material1->identifier()));
    EXPECT_EQ(material2, model.materialItemFromIdentifier(material2->identifier()));
    EXPECT_EQ(nullptr, model.materialItemFromIdentifier("non-existing-identifier"));
}

//! Checks the method which returns MaterialItem from material name.

TEST(TestMaterialModel, materialFromName)
{
    MaterialModel model;
    MaterialItem* material1 = model.addRefractiveMaterialItem("aaa", 1.0, 2.0);
    MaterialItem* material2 = model.addRefractiveMaterialItem("bbb", 3.0, 4.0);
    EXPECT_EQ(material1, model.materialItemFromName(material1->matItemName()));
    EXPECT_EQ(material2, model.materialItemFromName(material2->matItemName()));
    EXPECT_EQ(nullptr, model.materialItemFromName("non-existing-name"));
}

//! Default MaterialProperty construction.

TEST(TestMaterialModel, defaultMaterialProperty)
{
    MaterialModel model;

    // should throw in the absence of any materials
    // EXPECT_FAILED_ASSERT(model.defaultMaterialItem()); <--- TODO: restore

    // adding materials to the model, default property should refer to first material in a model
    MaterialItem* mat1 = model.addRefractiveMaterialItem("Something1", 1.0, 2.0);
    model.addRefractiveMaterialItem("Something2", 3.0, 4.0);
    EXPECT_EQ(model.defaultMaterialItem(), mat1);
}

TEST(TestMaterialModel, serializeRefractiveMaterial)
{
    MaterialItem material;
    material.setRefractiveIndex(11, 12);
    material.setMatItemName("name");
    material.setMagnetization(R3(1, 2, 3));
    const auto a1 = GUI::Util::createBackup(&material);

    MaterialItem target;
    GUI::Util::restoreBackup(&target, a1);

    // XML comparison
    const auto a2 = GUI::Util::createBackup(&target);
    EXPECT_EQ(a1, a2);

    // Content comparison
    EXPECT_EQ(target.matItemName(), "name");
    EXPECT_TRUE(target.hasRefractiveIndex());
    EXPECT_EQ(target.delta(), 11);
    EXPECT_EQ(target.beta(), 12);
    EXPECT_EQ(target.magnetization().r3(), R3(1, 2, 3));
}

TEST(TestMaterialModel, serializeSLDMaterial)
{
    MaterialItem material;
    material.setScatteringLengthDensity(complex_t(11, 12));
    material.setMatItemName("name");
    material.setMagnetization(R3(1, 2, 3));
    const auto a1 = GUI::Util::createBackup(&material);

    MaterialItem target;
    GUI::Util::restoreBackup(&target, a1);

    // XML comparison
    const auto a2 = GUI::Util::createBackup(&target);
    EXPECT_EQ(a1, a2);

    // Content comparison
    EXPECT_EQ(target.matItemName(), "name");
    EXPECT_FALSE(target.hasRefractiveIndex());
    EXPECT_EQ(target.sldRe(), 11);
    EXPECT_EQ(target.sldIm(), 12);
    EXPECT_EQ(target.magnetization().r3(), R3(1, 2, 3));
}
