/*****************************************************************************
 * $CAMITK_LICENCE_BEGIN$
 *
 * CamiTK - Computer Assisted Medical Intervention ToolKit
 * (c) 2001-2016 Univ. Grenoble Alpes, CNRS, TIMC-IMAG UMR 5525 (GMCAO)
 *
 * Visit http://camitk.imag.fr for more information
 *
 * This file is part of CamiTK.
 *
 * CamiTK is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * CamiTK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with CamiTK.  If not, see <http://www.gnu.org/licenses/>.
 *
 * $CAMITK_LICENCE_END$
 ****************************************************************************/

#ifndef StructuralComponentDC_H
#define StructuralComponentDC_H

#include <pml/StructuralComponent.h>

#include "ComponentDC.h"
#include "PMComponentAPI.h"
#include <algorithm>

#include <vtkSmartPointer.h>

class StructuralComponentDCPopup;
class QPixmap;

class QMenu;

class PMManagerDC;

class Atom;

class AtomDC;

class Cell;

class vtkUnstructuredGrid;

class StructuralComponentDC;

class StructuralComponentDCProperties;

class vtkDoubleArray;

namespace std {
/** definition of a couple (=STL pair) [Atom *, unsigned int]
  * this associates an atom to its index in the vtkPoints array.
  */
typedef std::pair<Atom *, unsigned int> AtomVtkPointsIndexPair;
/** definition of the association map (=STL map) AtomVtkPointsIndexMap.
  * AtomVtkPointsIndexMap associate the atoms with their index in the VtkPoints array.
  * The key is the atom, so that it is simple to retrieve the index from the atom
  * (which is supposed to be the most often used functionnality).
  */
typedef std::map <Atom *, unsigned int> AtomVtkPointsIndexMap;
/// the iterator corresponding to the AtomVtkPointsMap
typedef std::map <Atom *, unsigned int>::iterator AtomVtkPointsIndexMapIterator;
}

/**
 *  @ingroup group_cepmodeling_components_physicalmodel
 *
 *  @brief
 *  The manager of the Structural component data.
 * Maintains a map to tell at what position is a give atom in the vtkPoints list.
 * This class gives also a conveniant method to save a StructuralComponent in a vtk unstructured grid.
 *
 *
 **/
class PHYSICALMODEL_COMPONENT_API StructuralComponentDC : public ComponentDC {
    Q_OBJECT
public:
    /** Default constructor, set the 3rd parameter to false to delay the construction of the Geometry
      * (represention of the DC in 3D).
      * delayRepresentationInit should always be set to true when the constructor is called
      * by a sub-class (see CellDC constructor).
      * If delayRepresentationInit is false (default) and if the default mode is visible,
      * then the SCDC constructor will build an Geometry considering a SC not a sub-class (i.e. Cell).
      */
    StructuralComponentDC(camitk::Component *parent, PMManagerDC * pmManagerDC, StructuralComponent *sc, bool delayRepresentationInit = false);

    /// destructor
    virtual ~StructuralComponentDC();

    /**
    * @name camitk::Component rewritten
    * This methods are overloaded from camitk::Component to give the StructuralComponentDC and
    * its sub classe CellDC a different behaviour.
    */
    ///@{
    virtual QPixmap getIcon();

    /// if the vtkPointSet is modified externally (e.g. by an operator), change the PM structure
    virtual void setPointSet(vtkSmartPointer<vtkPointSet>);

    /// select a specific sub DC using the cell id (do nothing if the SC is not composed by cells)
    virtual void cellPicked(vtkIdType, bool);

    /// select a specific sub DC using the point id (do nothing if the SC is not composed by atoms)
    virtual void pointPicked(vtkIdType, bool);

    /// do not select sub DC if they are AtomDC (AtomDC can only be selected individually)
    virtual void setSelected(const bool, const bool recursive=false);

    /// do not set enhanced modes if children are atomDC
    virtual void setEnhancedModes(const EnhancedModes);

    /// set the rendering mode of the StructuralComponent as well as the DC
    virtual void setRenderingModes(const RenderingModes);

    virtual void setActorColor(const RenderingModes, const double, const double, const double);

    virtual void setActorColor(const RenderingModes, double[4]);

    virtual void setColor(const double, const double, const double);

    virtual void setColor(const double, const double, const double, const double);

    /// add a sub DC: overloaded method to allow change in the physical model data representation
    virtual void addChild(InterfaceNode *);

    /// remove a sub DC: overloaded method to allow change in the physical model data representation
    virtual void removeChild(InterfaceNode *);

    /// Overriden method so that we actually can build a popup menu with different actions
    virtual QMenu * getPopupMenu(QWidget* parent);

    /// reimplement this method to give the property object to the property editor
    virtual QObject * getPropertyObject();
    ///@}

    /** This atom position has been changed, the 3D representation has to be updated.
     *  For optimization reason: if you know the atom order nr in this SCDC, please give it!
     * An atom in the cell position has been changed this method starts the update cascade process, consisting in
     *  updating all the 3D representation of the all the components containing this component, then of
     *  all the components containing all the components containing this component... etc... up
     * to the PM void updatePosition(AtomDC *, CellDC *);
     */
    void updatePosition(AtomDC *, int atomOrderNr = -1);

    /** create all the point data to display specific information
     *  (only works for SCs that have a surface representation).
     *  This will automatically fills up the AtomDC point data pointers.
     */
    virtual void createPointData();

    /// destroy all the point data
    virtual void destroyPointData();

    /** Build an unstructured grid from a Structural Component, and build the correct AtomVtkPointsIndexMap
        */
    static vtkSmartPointer<vtkUnstructuredGrid> structuralComponentToVtk(StructuralComponent *, std::AtomVtkPointsIndexMap *);

    /** Build an unstructured grid from a Structural Component.
        */
    static vtkSmartPointer<vtkUnstructuredGrid> structuralComponentToVtk(StructuralComponent *);

    /// convenient methods
    StructuralComponent *getSC();

    /** return if the structural component is a list of cells or of atoms    */
    StructuralComponent::ComposedBy composedBy();

    /** create an Geometry (composed by vtk unstructured grid of n cells) from the managed SC which is a list of cells */
    camitk::Geometry * cellsToGeometry();

    /// the method to update both atomDCs vector and atomVtkPointsIndexMap
    void updateAtoms();

protected:

    /// nr of atoms
    unsigned int nrOfAtoms;

    /// the point data array
    vtkSmartPointer<vtkDoubleArray> pointData;

private:
    /** create an Geometry (a vtk unstructured grid of 1 poly vertex) from the managed SC which is a list of atoms */
    camitk::Geometry * atomsToGeometry();

    /// the concrete building of the geometry
    virtual void initRepresentation();

    /// the popup menu
    StructuralComponentDCPopup * myPopupMenu;

    /** list of atom DCs.
        * This list is used only when the structural component is made of cells.
        * This list is here to speed up the application:
        * When you need atom's DC you could always call the PMManager::getDC method, but that is at
        * a high cost. So why not maintain the list of the Atom DCs of the structural component.
        * This list is build/update in the updateAtoms method.
        */
    std::vector <AtomDC *> atomDCs;

    /** cumulative nr of cell and points.
        * Example: if sc=[c0,c1] , c0=[a0,a2,a3] and c1=[a1,a2,a4,a0]
        * then nrOfParts = 1 + 3 + 1 + 4 = 9.
        * Updated in updateAtoms()
        */
    unsigned int nrOfParts;

    /// cell ids in the vtk structures to be removed
    std::vector <unsigned int> cellIdToBeRemoved;

    /// the StructuralComponentDC pixmap
    static QPixmap * myPixmap;

    /// the SC DC properties
    StructuralComponentDCProperties * myProp;

};

inline StructuralComponent * StructuralComponentDC::getSC() {
    return dynamic_cast<StructuralComponent *>(myComponent);
}
#endif
