!
!   cif2hkl: convert a CIF or CFL crystal structure file into a PowderN reflection list.
!
!   cif2hkl 1.4.2 (Sept 21st 2022) by [emmanuel.farhi@synchrotron-soleil.fr]
!     Farhi E. using crysFML <https://code.ill.fr/scientific-software/crysfml/>
!   Copyright (C) 2009-2019 Institut Laue Langevin, EUPL
!   Copyright (C) 2020-     Synchrotron Soleil,     GPL2.
!   This is free software; see the source for copying conditions.
!   There is NO warranty; not even for MERCHANTABILITY or FITNESS
!   FOR A PARTICULAR PURPOSE.
!
! Usage: ./cif2hkl [options][-o outfile] file1 file2 ...
! Action: Read a CIF/CFL/SHX/PCR crystallographic description 
!         and generates a HKL F^2 reflection list.
! Input:
!   file1...          Input file in CIF, PCR, CFL, SHX, INS, RES format.
!                       The file format is determined from its extension
!                         .CIF           Crystallographic Information File
!                         .PCR/.CFL      FullProf file
!                         .SHX/.INS/.RES ShelX file
! Output:
!   a file with readable header, and reflection list with columns
!     [ H K L Multiplicity Sin(Theta/Lambda) d_spacing |F|^2 ]
! Options:
! --help     or -h    Show this help
! --version  or -v    Display program version
! --out FILE          Specify the name of the next output file.
!    -o FILE            Default is to add .hkl to the initial file name.
! --lambda LAMBDA     Set the incoming probe wavelength [Angs].
!    -l    LAMBDA       Default is 0.5
! --powder   or -p    Generate a list of unique HKL reflections (for powders). Default.
! --xtal     or -x    Generate a list of all HKL reflections (for single crystals).
! --verbose           Display processing details.
! --no-output-files   Just read the CIF/CFL/ShelX file (for checking).
! Example: ./cif2hkl -o CaF2.laz CaF2.cfl

! Compile with:  
!   gfortran -O2 CFML_GlobalDeps_Linux.f90 CFML_Math_General.f90 CFML_String_Utilities_gf.f90 CFML_Math_3D.f90 CFML_Symmetry_Tables.f90 CFML_Scattering_Chemical_Tables.f90 CFML_Crystallographic_Symmetry.f90 CFML_Crystal_Metrics.f90 CFML_Reflections_Utilities.f90 CFML_Atom_TypeDef.f90 CFML_Geometry_Calc.f90 CFML_Molecular_Crystals.f90 CFML_IO_Formats.f90 CFML_Structure_Factors.f90 -o cif2hkl cif2hkl.F90 -lm
!   rm *.mod
!   ./cif2hkl ../CIF/CaF2.cfl
! 
! Create Matlab Mex with:
!     mex -c -O cif2hkl.F90
!     mex -O cif2hkl_mex.c cif2hkl.o -o cif2hkl -lgfortran

! ==============================================================================

! cif2hkl is available under the GPL2 license (see below). It uses CrysFML which
! is a LGPL library. 
 
! ==============================================================================
! Modules needed to compile this program:
! Module CFML_GlobalDeps                     CFML_GlobalDeps_Linux.f90
! Module CFML_Math_General                   CFML_Math_General.f90
! Module CFML_String_Utilities               CFML_String_Utilities_gf.f90
! Module CFML_Math_3D                        CFML_Math_3D.f90
! Module CFML_Symmetry_Tables                CFML_Symmetry_Tables.f90
! Module CFML_Scattering_Chemical_Tables     CFML_Scattering_Chemical_Tables.f90
! Module CFML_Crystallographic_Symmetry      CFML_Crystallographic_Symmetry.f90
! Module CFML_Crystal_Metrics                CFML_Crystal_Metrics.f90
! Module CFML_Reflections_Utilities          CFML_Reflections_Utilities.f90
! Module CFML_Atom_TypeDef                   CFML_Atom_TypeDef.f90
! Module CFML_Structure_Factors              CFML_Structure_Factors.f90
! Module CFML_Geometry_Calc                  CFML_Geometry_Calc.f90
! Module CFML_Molecular_Crystals             CFML_Molecular_Crystals.f90
! Module CFML_IO_Formats                     CFML_IO_Formats.f90
!
! CFML_GlobalDeps_Linux.f90 CFML_Math_General.f90 CFML_String_Utilities_gf.f90 CFML_Math_3D.f90 CFML_Symmetry_Tables.f90 CFML_Scattering_Chemical_Tables.f90 CFML_Crystallographic_Symmetry.f90 CFML_Crystal_Metrics.f90 CFML_Reflections_Utilities.f90 CFML_Atom_TypeDef.f90 CFML_Geometry_Calc.f90 CFML_Molecular_Crystals.f90 CFML_IO_Formats.f90 CFML_Structure_Factors.f90

 
! ==============================================================================
! routine to sort HKL reflections with decreasing d-spacing
subroutine sort_d(hkl)
  use CFML_Reflections_Utilities,      only: Reflection_List_Type, Reflection_Type
  use CFML_Math_General,               only: sort
  use CFML_GlobalDeps,                 only: cp
  
  type (Reflection_List_Type), INTENT(in out) :: hkl
  
  real(kind=cp),  dimension(hkl%Nref) :: S
  integer,        dimension(hkl%Nref) :: ind
  type (Reflection_Type),  dimension(hkl%Nref) :: ref
  
  integer :: i
  type (Reflection_Type) :: temp
  
  do i=1,hkl%Nref
    S(i)   = hkl%ref(i)%S
    ref(i) = hkl%ref(i)
  end do
  call sort(S, hkl%Nref, ind)
  ! now swap reflections: hkl%ref(ind(i))%S is in ascending order
  do i=1,hkl%Nref
    hkl%ref(i) = ref(ind(i))
  end do
  
end subroutine sort_d


! ==============================================================================
! CFML_cif2hkl: convert 'file_in' (CIF/CFL) into a reflection list 'file_out' for PowderN
! INPUT: 
!   file_in:   the CIF/CFL file name (string)
!   lambda:    neutron wavelength, which limits the number of reflections ; 
!              use e.g. 1.0 (float*8)
!   mode:      "p" or "x" for powder and Xtal handling resp.
!   verbose:   0 or 1 for silent and verbose display resp.
! OUPUT:
!   file_out: the reflection list file name (string)
!   message:  a message to display (string)
subroutine CFML_cif2hkl(file_in, file_out, lambda, powxtal, verbose, message, mode)

  use CFML_Crystallographic_Symmetry,  only: Space_Group_Type
  use CFML_Crystal_Metrics,            only: Crystal_Cell_Type
  use CFML_Atom_TypeDef,               only: Atom_List_Type
  use CFML_IO_Formats,                 only: Readn_Set_Xtal_Structure, File_List_Type
  use CFML_Reflections_Utilities,      only: Reflection_List_Type, Hkl_Uni, Hkl_Gen_SXtal,Get_MaxNumRef,WRITE_REFLIST_INFO
  use CFML_Structure_Factors,          only: Structure_Factors,Init_Structure_Factors,Calc_StrFactor
  use CFML_String_Utilities,           only: L_case
  use CFML_GlobalDeps,                 only: pi, dp
   
  implicit none
  
  ! subroutine I/O
  character(len=1024)         :: file_in, file_out     ! Name of the input/output file
  real*8                      :: lambda                ! probe wavelength
  character(len=1024)         :: powxtal               ! 'p' or 'x' or '-'
  integer                     :: verbose               ! 0 or 1 for verbose mode
  character*4096              :: message
  character(len=1024)         :: mode                  ! NUC,XRA,MAG,ELE
  
  ! local variables
  type (File_List_Type)       :: fich_cfl
  type (Space_Group_Type)     :: SpG
  type (Crystal_Cell_Type)    :: Cell
  type (Atom_List_Type)       :: A
  type (Reflection_List_Type) :: hkl
  real                        :: stlmax
  integer                     :: lun=1, MaxNumRef, I, Y
  character(len=1024)         :: ext                   ! file name extension

  integer, parameter                             :: n_elements=423
  character (Len = 8), dimension(n_elements)     :: element
  real ( KIND = dp ), dimension(n_elements)      :: Bcoh, Binc, weight, Sabs
  real                                           :: sigma_coh, sigma_inc, sigma_abs, mass, F
  
  character(len=1024)                            :: formula
  character(len=4096)                            :: s1,s2,s3  ! temporary string for concatenation
  integer*4, dimension(3)                        :: today, now
  character*2   eol
  
  integer dot_pos
  integer sep_pos
  integer end_pos
   
  !values to test CFML_cif2hkl(program)
  !file_in  = 'Na2Ca3Al2F14.cfl'
  !file_out = 'Na2Ca3Al2F14_out'
  !lambda   = 2.4105
  if (lambda > 0) then
    stlmax = 1/lambda
  else
    stlmax= 1.0
  end if
  
  eol=char(13)//char(10)
  message = ''
  
  ! set the element name, cross sections and weight
  ! list of isotopes
  element   = (/'H    ','D    ','2H   ','3H   ','He   ','3He  ','4He  ','Li   ','6Li  ','7Li  ','Be   ',&
  '9Be  ','B    ','10B  ','11B  ','C    ','12C  ','13C  ','14C  ','N    ','14N  ','15N  ','O    ','16O  ',&
  '17O  ','18O  ','F    ','19F  ','Ne   ','20Ne ','21Ne ','22Ne ','Na   ','23Na ','Mg   ','24Mg ','25Mg ',&
  '26Mg ','Al   ','27Al ','Si   ','28Si ','29Si ','30Si ','P    ','31P  ','S    ','32S  ','33S  ','34S  ',&
  '36S  ','Cl   ','35Cl ','37Cl ','Ar   ','36Ar ','38Ar ','40Ar ','K    ','39K  ','40K  ','41K  ','Ca   ',&
  '40Ca ','42Ca ','43Ca ','44Ca ','46Ca ','48Ca ','Sc   ','45Sc ','Ti   ','46Ti ','47Ti ','48Ti ','49Ti ',&
  '50Ti ','V    ','50V  ','51V  ','Cr   ','50Cr ','52Cr ','53Cr ','54Cr ','Mn   ','55Mn ','Fe   ','54Fe ',&
  '56Fe ','57Fe ','58Fe ','Co   ','59Co ','Ni   ','58Ni ','60Ni ','61Ni ','62Ni ','64Ni ','Cu   ','63Cu ',&
  '65Cu ','Zn   ','64Zn ','66Zn ','67Zn ','68Zn ','70Zn ','Ga   ','69Ga ','71Ga ','Ge   ','70Ge ','72Ge ',&
  '73Ge ','74Ge ','76Ge ','As   ','75As ','Se   ','74Se ','76Se ','77Se ','78Se ','80Se ','82Se ','Br   ',&
  '79Br ','81Br ','Kr   ','78Kr ','80Kr ','82Kr ','83Kr ','84Kr ','86Kr ','Rb   ','85Rb ','87Rb ','Sr   ',&
  '84Sr ','86Sr ','87Sr ','88Sr ','Y    ','89Y  ','Zr   ','90Zr ','91Zr ','92Zr ','94Zr ','96Zr ','Nb   ',&
  '93Nb ','Mo   ','92Mo ','94Mo ','95Mo ','96Mo ','97Mo ','98Mo ','100Mo','Tc   ','99Tc ','Ru   ','96Ru ',&
  '98Ru ','99Ru ','100Ru','101Ru','102Ru','104Ru','Rh   ','103Rh','Pd   ','102Pd','104Pd','105Pd','106Pd',&
  '108Pd','110Pd','Ag   ','107Ag','109Ag','Cd   ','106Cd','108Cd','110Cd','111Cd','112Cd','113Cd','114Cd',&
  '116Cd','In   ','113In','115In','Sn   ','112Sn','114Sn','115Sn','116Sn','117Sn','118Sn','119Sn','120Sn',&
  '122Sn','124Sn','Sb   ','121Sb','123Sb','Te   ','120Te','122Te','123Te','124Te','125Te','126Te','128Te',&
  '130Te','I    ','127I ','Xe   ','124Xe','126Xe','128Xe','129Xe','130Xe','131Xe','132Xe','134Xe','136Xe',&
  'Cs   ','133Cs','Ba   ','130Ba','132Ba','134Ba','135Ba','136Ba','137Ba','138Ba','La   ','138La','139La',&
  'Ce   ','136Ce','138Ce','140Ce','142Ce','Pr   ','141Pr','Nd   ','142Nd','143Nd','144Nd','145Nd','146Nd',&
  '148Nd','150Nd','Pm   ','147Pm','Sm   ','144Sm','147Sm','148Sm','149Sm','150Sm','152Sm','154Sm','Eu   ',&
  '151Eu','153Eu','Gd   ','152Gd','154Gd','155Gd','156Gd','157Gd','158Gd','160Gd','Tb   ','159Tb','Dy   ',&
  '156Dy','158Dy','160Dy','161Dy','162Dy','163Dy','164Dy','Ho   ','165Ho','Er   ','162Er','164Er','166Er',&
  '167Er','168Er','170Er','Tm   ','169Tm','Yb   ','168Yb','170Yb','171Yb','172Yb','173Yb','174Yb','176Yb',&
  'Lu   ','175Lu','176Lu','Hf   ','174Hf','176Hf','177Hf','178Hf','179Hf','180Hf','Ta   ','180Ta','181Ta',&
  'W    ','180W ','182W ','183W ','184W ','186W ','Re   ','185Re','187Re','Os   ','184Os','186Os','187Os',&
  '188Os','189Os','190Os','192Os','Ir   ','191Ir','193Ir','Pt   ','190Pt','192Pt','194Pt','195Pt','196Pt',&
  '198Pt','Au   ','196Au','Hg   ','196Hg','198Hg','199Hg','200Hg','201Hg','202Hg','204Hg','Tl   ','203Tl',&
  '205Tl','Pb   ','204Pb','206Pb','207Pb','208Pb','Bi   ','209Bi','Po   ','210Po','At   ','Rn   ','222Rn',&
  'Fr   ','Ra   ','226Ra','Ac   ','Th   ','232Th','Pa   ','231Pa','U    ','233U ','234U ','235U ','238U ',&
  'Np   ','237Np','Pu   ','238Pu','239Pu','240Pu','242Pu','Am   ','243Am','Cm   ','244Cm','246Cm','248Cm',&
  'Bk   ','Cf   ','Es   ','Fm   ','Md   ','No   ','Lr   ','Rf   ','Db   ','Sg   ','Bh   ','Hs   ','Mt   ',&
  'Ds   ','Rg   ','Cn   ','Uut  ','Uuq  ','Uup  ','Uuh  ','Uus  ','Uuo  '/)
  ! list of coherent lengths
  Bcoh      = (/-0.00003739, 0.00006671, 0.00006671, 0.00004792, &
   0.00003260, 0.00000000, 0.00003260, -0.00001900, 0.00000000, -0.00002220,&
    0.00007790, 0.00007790, 0.00000000	, 0.00000000, 0.00006650,&
    0.00006646	, 0.00006651, 0.00006190, 0.00000000, 0.00009360	,&
    0.00009370, 0.00006440, 0.00005803	, 0.00005803, 0.00005780,&
    0.00005840, 0.00005654	, 0.00005654, 0.00004566, 0.00004631,&
    0.00006660, 0.00003870, 0.00003630, 0.00003630, 0.00005375, 0.00005660,&
    0.00003620, 0.00004890, 0.00003449, 0.00003449, 0.00004149, 0.00004107,&
    0.00004700, 0.00004580, 0.00005130	, 0.00005130, 0.00002847	,&
    0.00002804, 0.00004740, 0.00003480, 0.00003000, 0.00009577, 0.00011650,&
    0.00003080, 0.00001909, 0.00024900, 0.00003500, 0.00001830, 0.00003670	, 0.00003740, 0.00003000,&
    0.00002690, 0.00004700, 0.00004800, 0.00003360, -0.00001560, 0.00001420, 0.00003600, 0.00000390,&
    0.00012290, 0.00012290, -0.00003438, 0.00004930, 0.00003630, -0.00006080, 0.00001040, 0.00006180,&
    -0.00000382, 0.00007600, -0.00000402, 0.00003635, -0.00004500, 0.00004920, -0.00004200, 0.00004550,&
    -0.00003730, -0.00003730, 0.00009450, 0.00004200, 0.00009940, 0.00002300, 0.00015000, 0.00002490,&
    0.00002490, 0.00010300, 0.00014400, 0.00002800, 0.00007600, -0.00008700, -0.00000370, 0.00007718,&
    0.00006430, 0.00010610, 0.00005680, 0.00005220, 0.00005970, 0.00007560, 0.00006030, 0.00006000, 0.00007288,&
    0.00007880, 0.00006400, 0.00008185, 0.00010000, 0.00008510, 0.00005020, 0.00007580, 0.00008200, 0.00006580,&
    0.00006580, 0.00007970, 0.00000800, 0.00012200, 0.00008250, 0.00008240, 0.00007480, 0.00006340, 0.00006795,&
    0.00006800, 0.00006790, 0.00007810, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00008100,&
    0.00007090, 0.00007030, 0.00007230, 0.00007020, 0.00007000, 0.00005670, 0.00007400, 0.00007150,&
    0.00007750	, 0.00007750, 0.00007160, 0.00006400, 0.00008700, 0.00007400, 0.00008200, 0.00005500,&
    0.00007054, 0.00007054, 0.00006715, 0.00006910, 0.00006800, 0.00006910, 0.00006200, 0.00007240, 0.00006580,&
    0.00006730, 0.00006800, 0.00006800, 0.00007030, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000,&
    0.00000000, 0.00000000, 0.00005880, 0.00005880, 0.00005910, 0.00007700, 0.00007700, 0.00005500, 0.00006400,&
    0.00004100, 0.00007700, 0.00005922, 0.00007555, 0.00004165, 0.00000000, 0.00005000, 0.00005400, 0.00005900,&
    0.00006500, 0.00006400, 0.00000000, 0.00007500, 0.00006300, 0.00000000, 0.00005390, 0.00000000, 0.00006225,&
    0.00006000, 0.00006200, 0.00006000, 0.00005930, 0.00006480, 0.00006070, 0.00006120, 0.00006490, 0.00005740,&
    0.00005970, 0.00005570, 0.00005710, 0.00005380, 0.00005800, 0.00005300, 0.00003800, 0.00000000, 0.00007960,&
    0.00005020, 0.00005560, 0.00005890, 0.00006020, 0.00005280	, 0.00005280, 0.00004920, 0.00000000,&
    0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00005420,&
    0.00005420, 0.00005070, -0.00003600, 0.00007800, 0.00005700, 0.00004670, 0.00004910, 0.00006830,&
    0.00004840, 0.00008240, 0.00008000, 0.00008240, 0.00004840, 0.00005800, 0.00006700, 0.00004840, 0.00004750,&
    0.00004580, 0.00004580, 0.00007690, 0.00007700, 0.00014000, 0.00002800, 0.00014000, 0.00008700, 0.00005700,&
    0.00005300, 0.00012600, 0.00012600, 0.00000000, -0.00003000, 0.00014000, -0.00003000, 0.00000000,&
    0.00014000, -0.00005000, 0.00009300, 0.00000000, 0.00000000, 0.00008220, 0.00000000, 0.00010000,&
    0.00010000, 0.00000000, 0.00006300, 0.00000000, 0.00009000, 0.00009150, 0.00007380, 0.00007380, 0.00000000,&
    0.00006100, 0.00006000, 0.00006700, 0.00010300, -0.00001400, 0.00005000, 0.00000000, 0.00008010,&
    0.00008010, 0.00007790, 0.00008800, 0.00008200, 0.00010600, 0.00003000, 0.00007400, 0.00009600, 0.00007070,&
    0.00007070, 0.00012430, 0.00000000, 0.00006770, 0.00009660, 0.00009430, 0.00009560, 0.00019300, 0.00008720,&
    0.00007210, 0.00007240, 0.00000000, 0.00007700, 0.00010900, 0.00006610, 0.00000800, 0.00005900, 0.00007460,&
    0.00013200, 0.00006910, 0.00007000, 0.00006910, 0.00004860	, 0.00005000, 0.00006970, 0.00006530,&
    0.00007480, -0.00000720, 0.00009200, 0.00009000, 0.00009300, 0.00010700, 0.00010000, 0.00011600,&
    0.00010000, 0.00007600, 0.00010700, 0.00011000, 0.00011500, 0.00010600, 0.00000000, 0.00000000, 0.00009600,&
    0.00009000, 0.00009900, 0.00010550, 0.00008830, 0.00009890, 0.00007800, 0.00007630, 0.00007630, 0.00012692,&
    0.00030300, 0.00000000, 0.00016900, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00008776, 0.00006990,&
    0.00009520, 0.00009405, 0.00009900, 0.00009220, 0.00009280, 0.00009500, 0.00008532, 0.00008532, 0.00000000,&
    0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00010000, 0.00010000, 0.00000000, 0.00010310,&
    0.00010310, 0.00009100, 0.00009100, 0.00008417	, 0.00010100, 0.00012400, 0.00010470, 0.00008402,&
    0.00010550, 0.00010550, 0.00000000, 0.00014100, 0.00007700, 0.00003500, 0.00008100, 0.00008300, 0.00008300,&
    0.00000000, 0.00009500, 0.00009300, 0.00007700, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000,&
    0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000,&
    0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000 /)
  ! list of incoherent lengths
  Binc      = (/0.00025272	, 0.00004040, 0.00004040	, -0.00001040, 0.00000000	, 0.00000000, 0.00000000,&
    2.70576E-05, 0.00000000, -0.00002490, 0.00000120	, 0.00000120, 3.67807E-05, 0.00000000, -0.00001300,&
    8.92062E-07, 0.00000000, -0.00000520, 0.00000000, 1.99471E-05, 0.00002000, -0.00000020, 7.97885E-07,&
    0.00000000, 0.00000180, 0.00000000, -0.00000082	, -0.00000082, 2.52313E-06, 0.00000000, 0.00000600,&
    0.00000000, 0.00003590	, 0.00003590, 7.97885E-06, 0.00000000, 0.00001480, 0.00000000, 0.00000256	,&
    0.00000256, 1.78412E-06, 0.00000000, 0.00000090, 0.00000000, 0.00000200	, 0.00000200, 2.36017E-06,&
    0.00000000, 0.00001500, 0.00000000, 0.00000000, 6.49431E-05, 0.00006100, 0.00000100, 1.33809E-05,&
    0.00000000, 0.00000000, 0.00000000, 1.46581E-05, 0.00001400, 1.99471E-05, 0.00001500, 6.30783E-06,&
    0.00000000, 0.00000000, 1.99471E-05, 0.00000000, 0.00000000, 0.00000000, -0.00006000, -0.00006000,&
    4.77899E-05, 0.00000000, -0.00003500, 0.00000000, 0.00005100, 0.00000000, 6.35809E-05, 1.99471E-05,&
    0.00006350, 3.81611E-05, 0.00000000, 0.00000000, 0.00006870, 0.00000000, 0.00001790, 0.00001790,&
    1.78412E-05, 0.00000000, 0.00000000, 1.5451E-05, 0.00000000, -0.00006200, -0.00006200, 6.43275E-05,&
    0.00000000, 0.00000000, 0.00003900, 0.00000000, 0.00000000, 2.09207E-05, 0.00000220, 0.00001790,&
    7.82781E-06, 0.00000000, 0.00000000, -0.00001500, 0.00000000, 0.00000000, 1.12838E-05, -0.00000850,&
    -0.00000820, 1.19683E-05, 0.00000000, 0.00000000, 0.00003400, 0.00000000, 0.00000000, -0.00000690,&
    -0.00000690, 1.59577E-05, 0.00000000, 0.00000000, 0.00000600, 0.00000000, 0.00000000, 0.00000000,&
    8.92062E-06, -0.00001100, 0.00000600, 2.82095E-06, 0.00000000, 0.00000000, 0.00000000, 0.00000000,&
    0.00000000, 0.00000000, 1.99471E-05, 1.99471E-05, 1.99471E-05, 6.90988E-06, 0.00000000, 0.00000000,&
    1.99471E-05, 0.00000000, 0.00001100	, 0.00001100, 3.98942E-06, 0.00000000, -0.00001080, 0.00000000,&
    0.00000000, 0.00000000, -0.00000139, -0.00000139, 5.6419E-06, 0.00000000, 0.00000000, 1.99471E-05,&
    0.00000000, 1.99471E-05, 0.00000000, 0.00000000, 1.99471E-05, 1.99471E-05, 1.78412E-05, 0.00000000,&
    0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 1.5451E-05, 1.5451E-05,&
    8.60273E-06, 0.00000000, 0.00000000, -0.00002600, 0.00000000, 0.00000000, 0.00000000, 2.14837E-05,&
    0.00001000, -0.00001600, 5.24727E-05, 0.00000000, 0.00000000, 0.00000000, 1.5451E-05, 0.00000000,&
    1.5451E-05, 0.00000000, 0.00000000, 2.07296E-05, 0.00000017, -0.00002100, 4.18414E-06, 0.00000000,&
    0.00000000, 1.5451E-05, 0.00000000, 1.5451E-05, 0.00000000, 1.5451E-05, 0.00000000, 0.00000000, 0.00000000,&
    2.36017E-06, -0.00000050, -0.00000100, 8.46284E-06, 0.00000000, 0.00000000, -0.00002040, 0.00000000,&
    -0.00000260, 0.00000000, 0.00000000, 0.00000000, 0.00001580	, 0.00001580, 0.00003040	, 0.00000000,&
    0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000,&
    0.00001290	, 0.00001290, 1.09255E-05, 0.00000000, 0.00000000, 0.00000000, 1.99471E-05, 0.00000000,&
    1.99471E-05, 0.00000000, 2.99871E-05, 1.99471E-05, 0.00003000, 8.92062E-07, 0.00000000, 0.00000000,&
    0.00000000, 0.00000000, -0.00000350, -0.00000350, 8.55636E-05, 0.00000000, 0.00021000, 0.00000000,&
    6.30783E-05, 0.00000000, 0.00000000, 0.00000000, 0.00003200	, 0.00003200, 0.00017617	, 0.00000000,&
    0.00011000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 4.46031E-05, 0.00000000,&
    0.00003200, 0.00034664	, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000,&
    0.00000000, -0.00000170, -0.00000170, 0.00020806	, 0.00000000, 0.00000000, 0.00000000, 0.00004900,&
    0.00000000, 0.00001300, 0.00000000, -0.00001700, -0.00001700, 2.95864E-05, 0.00000000, 0.00000000,&
    0.00000000, 0.00001000, 0.00000000, 0.00000000, 0.00000900	, 0.00000900, 5.6419E-05, 0.00000000,&
    0.00000000, -0.00005590, 0.00000000, -0.00005300, 0.00000000, 0.00000000, 2.36017E-05, 0.00002200,&
    0.00000000, 4.54864E-05, 0.00000000, 0.00000000, 0.00000900, 0.00000000, 0.00001060, 0.00000000,&
    2.82095E-06, 1.99471E-05, -0.00000290, 3.60155E-05, 0.00000000, 0.00000000, 1.5451E-05, 0.00000000,&
    0.00000000, 2.67619E-05, 0.00002000, 0.00002800, 1.5451E-05, 0.00000000, 0.00000000, 1.5451E-05,&
    0.00000000, 1.99471E-05, 0.00000000, 0.00000000, 0.00000000	, 0.00000000, 0.00000000, 1.01711E-05,&
    0.00000000, 0.00000000, 0.00000000, -0.00001000, 0.00000000, 0.00000000, -0.00001840, -0.00001840,&
    7.24715E-05, 0.00000000, 0.00000000, 0.00015500, 0.00000000, 0.00000000, 0.00000000, 0.00000000,&
    1.29272E-05, 0.00001060, -0.00000242, 1.5451E-06, 0.00000000, 0.00000000, 0.00000140, 0.00000000,&
    2.58544E-06, 2.58544E-06, 0.00000259	, 0.00000259, 0.00000000	, 0.00000000	, 0.00000000, 0.00000000	,&
    0.00000000	, 0.00000000, 0.00000000	, 0.00000000	, 0.00000000, 8.92062E-06, 8.92062E-06, 1.99471E-06,&
    0.00001000, 0.00000000, 0.00001300, 0.00000000, 1.99471E-05, 1.99471E-05, 0.00000000	, 0.00000000,&
    0.00001300, 0.00000000, 0.00000000, 0.00002000	, 0.00002000, 0.00000000	, 0.00000000, 0.00000000,&
    0.00000000, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	,&
    0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	,&
    0.00000000	, 0.00000000	, 0.00000000	, 0.00000000, 0.00000000, 0.00000000, 0.00000000, 0.00000000,&
    0.00000000/)
  ! list of absorption cross sections
  Sabs = (/0.33260000	, 0.00051900	, 0.00051900	, 0.00000000	, 0.00747000	, 5333.00000000,&
    0.00000000	, 70.50000000, 940.00000000, 0.04540000, 0.00760000	, 0.00760000	, 767.00000000,&
    3835.00000000, 0.00550000, 0.00350000	, 0.00353000	, 0.00137000, 0.00000000	, 1.90000000	, 1.91000000	,&
    0.00002400, 0.00019000	, 0.00010000	, 0.23600000	, 0.00016000	, 0.00960000	, 0.00960000, 0.03900000,&
    0.03600000, 0.67000000, 0.04600000, 0.53000000	, 0.53000000, 0.06300000, 0.05000000, 0.19000000,&
    0.03820000, 0.23100000	, 0.23100000, 0.17100000, 0.17700000, 0.10100000, 0.10700000, 0.17200000	,&
    0.17200000	, 0.53000000	, 0.54000000	, 0.54000000	, 0.22700000	, 0.15000000	, 33.50000000,&
    44.10000000, 0.43300000, 0.67500000, 5.20000000, 0.80000000, 0.66000000, 2.10000000	, 2.10000000	,&
    35.00000000, 1.46000000	, 0.43000000, 0.41000000, 0.68000000, 6.20000000, 0.88000000, 0.74000000,&
    1.09000000, 27.50000000, 27.50000000, 6.09000000, 0.59000000, 1.70000000, 7.84000000, 2.20000000,&
    0.17900000, 5.08000000, 60.00000000, 4.90000000, 3.05000000, 15.80000000, 0.76000000, 18.10000000,&
    0.36000000, 13.30000000, 13.30000000, 2.56000000, 2.25000000, 2.59000000, 2.48000000, 1.28000000,&
    37.18000000, 37.18000000, 4.49000000, 4.60000000, 2.90000000, 2.50000000, 14.50000000, 1.52000000,&
    3.78000000, 4.50000000, 2.17000000, 1.11000000, 0.93000000, 0.62000000, 6.80000000, 1.10000000, 0.09200000,&
    2.75000000, 2.18000000, 3.61000000, 2.20000000, 3.00000000, 0.80000000, 15.10000000, 0.40000000,&
    0.16000000, 4.50000000	, 4.50000000, 11.70000000, 51.80000000, 85.00000000, 42.00000000, 0.43000000,&
    0.61000000, 0.04400000, 6.90000000, 11.00000000, 2.70000000, 25.00000000, 6.40000000, 11.80000000,&
    29.00000000, 185.00000000, 0.11300000, 0.00300000, 0.38000000, 0.48000000, 0.12000000, 1.28000000,&
    0.87000000, 1.04000000, 16.00000000, 0.05800000, 1.28000000	, 1.28000000	, 0.18500000, 0.01100000,&
    1.17000000, 0.22000000, 0.04990000, 0.02290000, 1.15000000	, 1.15000000, 2.48000000	, 0.01900000,&
    0.01500000, 13.10000000, 0.50000000, 2.50000000, 0.12700000, 0.40000000, 20.00000000, 20.00000000,&
    2.56000000, 0.28000000, 8.00000000, 6.90000000, 4.80000000, 3.30000000, 1.17000000, 0.31000000,&
    144.80000000, 144.80000000, 6.90000000, 3.40000000, 0.60000000, 20.00000000, 0.30400000, 8.55000000,&
    0.22600000, 63.30000000, 37.60000000, 91.00000000, 2520.00000000, 1.00000000, 1.10000000, 11.00000000,&
    24.00000000, 2.20000000, 20600.00000000, 0.34000000, 0.07500000, 193.80000000, 12.00000000, 202.00000000,&
    0.62600000, 1.00000000, 0.11400000, 30.00000000, 0.14000000, 2.30000000, 0.22000000, 2.20000000,&
    0.14000000, 0.18000000, 0.13300000, 4.91000000, 5.75000000, 3.80000000, 4.70000000, 2.30000000, 3.40000000,&
    418.00000000, 6.80000000, 1.55000000, 1.04000000, 0.21500000, 0.29000000, 6.15000000	, 6.15000000,&
    23.90000000	, 165.00000000, 3.50000000, 8.00000000, 21.00000000, 26.00000000, 85.00000000, 0.45000000,&
    0.26500000, 0.26000000, 29.00000000	, 29.00000000, 1.10000000, 30.00000000, 7.00000000, 2.00000000,&
    5.80000000, 0.68000000, 3.60000000, 0.27000000, 8.97000000, 57.00000000, 8.93000000, 0.63000000,&
    7.30000000, 1.10000000, 0.57000000, 0.95000000, 11.50000000, 11.50000000, 50.50000000, 18.70000000,&
    337.00000000, 3.60000000, 42.00000000, 1.40000000, 2.50000000, 1.20000000, 168.40000000, 168.40000000,&
    5922.00000000, 0.70000000, 57.00000000, 2.40000000, 42080.00000000, 104.00000000, 206.00000000, 8.40000000,&
    4530.00000000, 9100.00000000, 312.00000000, 49700.00000000, 735.00000000, 85.00000000, 61100.00000000,&
    1.50000000, 259000.00000000, 2.20000000, 0.77000000, 23.40000000, 23.40000000, 994.00000000, 33.00000000,&
    43.00000000, 56.00000000, 600.00000000, 194.00000000, 124.00000000, 2840.00000000, 64.70000000,&
    64.70000000, 159.00000000, 19.00000000, 13.00000000, 19.60000000, 659.00000000, 2.74000000, 5.80000000,&
    100.00000000, 100.00000000, 34.80000000, 2230.00000000, 11.40000000, 48.60000000, 0.80000000, 17.10000000,&
    69.40000000, 2.85000000, 74.00000000, 21.00000000, 2065.00000000, 104.10000000, 561.00000000, 23.50000000,&
    373.00000000, 84.00000000, 41.00000000, 13.04000000, 20.60000000, 563.00000000, 20.50000000, 18.30000000,&
    30.00000000, 20.70000000, 10.10000000, 1.70000000, 37.90000000, 89.70000000, 112.00000000, 76.40000000,&
    16.00000000, 3000.00000000, 80.00000000, 320.00000000, 4.70000000, 25.00000000, 13.10000000, 2.00000000,&
    425.00000000, 954.00000000, 111.00000000, 10.30000000, 152.00000000, 10.00000000, 1.44000000, 27.50000000,&
    0.72000000, 3.66000000, 98.65000000, 98.65000000, 372.30000000, 3080.00000000, 2.00000000, 2150.00000000,&
    60.00000000, 7.80000000, 4.89000000, 0.43000000, 3.43000000, 11.40000000, 0.10400000, 0.17100000	,&
    0.65000000, 0.03000000, 0.69900000, 0.00048000, 0.03380000, 0.03380000, 0.00000000	, 0.00000000,&
    0.00000000	, 0.00000000	, 0.00000000, 0.00000000	, 12.80000000	, 12.80000000, 0.00000000	, 7.37000000	,&
    7.37000000, 200.60000000, 200.60000000, 7.57000000	, 574.70000000, 100.10000000, 680.90000000, 2.68000000,&
    175.90000000, 175.90000000, 0.00000000	, 558.00000000, 1017.30000000, 289.60000000, 18.50000000,&
    75.30000000	, 75.30000000, 0.00000000	, 16.20000000, 1.36000000, 3.00000000, 0.00000000	, 0.00000000	,&
    0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	,&
    0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	,&
    0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	, 0.00000000	/)
  ! list of isotopes weights
  weight = (/1.0079000, 2.0141018, 2.0141018, 3.0160493, 4.0026000, 3.0160293, 4.0026033, 6.9410000, 6.0151223,&
    7.0160040, 9.0121821, 9.0121821, 10.8110000, 10.0129370, 11.0093055, 12.0107000, 12.0000000, 13.0033548,&
    14.0032420, 14.0067000, 14.0030740, 15.0001089, 15.9994000, 15.9949146, 16.9991315, 17.9991604, 18.9984032,&
    18.9984032, 20.1797000, 19.9924402, 20.9938467, 21.9913855, 22.9897697, 22.9897697, 24.3050000, 23.9850419,&
    24.9858370, 25.9825930, 26.9815384, 26.9815384, 28.0855000, 27.9769265, 28.9764947, 29.9737702, 30.9737615,&
    30.9737615, 32.0650000, 31.9720707, 32.9714585, 33.9678668, 35.9670809, 35.4530000, 34.9688527, 36.9659026,&
    39.9480000, 35.9675463, 37.9627322, 39.9623831, 39.0983000, 38.9637069, 39.9639987, 40.96182597,&
    40.0780000, 39.9625912, 41.9586183, 42.9587668, 43.9554811, 45.9536928, 47.9525340, 44.9559102, 44.9559102,&
    47.8670000, 45.9526295, 46.9517638, 47.9479471, 48.9478708, 49.9447921, 50.9415000, 49.9471628, 50.9439637,&
    51.9961000, 49.9460496, 51.9405119, 52.9406538, 53.9388849, 54.9380496, 54.9380496, 55.8450000, 53.9396148,&
    55.9349421, 56.9353987, 57.9332805, 58.9332002, 58.9332002, 58.6934000, 57.9353479, 59.9307906, 60.9310604,&
    61.9283488, 63.9279696, 63.5460000, 62.9296011, 64.9277937, 65.3900000, 63.9291466, 65.9260368, 66.9271309,&
    67.9248476, 69.9253250, 69.7230000, 68.9255810, 70.9247050, 72.6400000, 69.9242504, 71.9220762, 72.9234594,&
    73.9211782, 75.9214027, 74.9215964, 74.9215964, 78.9600000, 73.9224766, 75.9192141, 76.9199146, 77.9173095,&
    79.9165218, 81.9167000, 79.9040000, 78.9183376, 80.9162910, 83.8000000, 77.9203860, 79.9163780, 81.9134846,&
    82.9141360, 83.9115070, 85.9106103, 85.4678000, 84.9117893, 86.9091835, 87.6200000, 83.9134250, 85.9092624,&
    86.9088793, 87.9056143, 88.9058479, 88.9058479, 91.2240000, 89.9047037, 90.9056450, 91.9050401, 93.9063158,&
    95.9082760, 92.9063775, 92.9063775, 95.9400000, 91.9068100, 93.9050876, 94.9058415, 95.9046789, 96.9060210,&
    97.9054078, 99.9074770, 98.0000000, 98.0000000, 101.0700000, 95.9075980, 97.9052870, 98.9059393, 99.9042197,&
    100.9055822, 101.9043495, 103.9054300, 102.9055040, 102.9055040, 106.4200000, 101.9056080, 103.9040350,&
    104.9050840, 105.9034830, 107.9038940, 109.9051520, 107.8682000, 106.9050930, 108.9047560, 112.4110000,&
    105.9064580, 107.9041830, 109.9030060, 110.9041820, 111.9027572, 112.9044009, 113.9033581, 115.9047550,&
    114.8180000, 112.9040610, 114.9038780, 118.7100000, 111.9048210, 113.9027820, 114.9033460, 115.9017440,&
    116.9029540, 117.9016060, 118.9033090, 119.9021966, 121.9034401, 123.9052746, 121.7600000, 120.9038180,&
    122.9042157, 127.6000000, 119.9040200, 121.9030471, 122.9042730, 123.9028195, 124.9044247, 125.9033055,&
    127.9044614, 129.9062228, 126.9044680, 126.9044680, 131.2930000, 123.9058958, 125.9042690, 127.9035304,&
    128.9047795, 129.9035079, 130.9050819, 131.9041545, 133.9053945, 135.9072200, 132.9054470, 132.9054470,&
    137.3270000, 129.9063100, 131.9050560, 133.9045030, 134.9056830, 135.9045700, 136.9058210, 137.9052410,&
    138.9055000, 137.9071070, 138.9063480, 140.1160000, 135.9071400, 137.9059860, 139.9054340, 141.9092400,&
    140.9076480, 140.9076480, 144.2400000, 141.9077190, 142.9098100, 143.9100830, 144.9125690, 145.9131120,&
    147.9168890, 149.9208870, 145.0000000, 145.0000000, 150.3600000, 143.9119950, 146.9148930, 147.9148180,&
    148.9171800, 149.9172710, 151.9197280, 153.9222050, 151.9640000, 150.9198460, 152.9212260, 157.2500000,&
    151.9197880, 153.9208620, 154.9226190, 155.9221200, 156.9239570, 157.9241010, 159.9270510, 158.9253430,&
    158.9253430, 162.5000000, 155.9242780, 157.9244050, 159.9251940, 160.9269300, 161.9267950, 162.9287280,&
    163.9291710, 164.9303190, 164.9303190, 167.2590000, 161.9287750, 163.9291970, 165.9302900, 166.9320450,&
    167.9323680, 169.9354600, 168.9342110, 168.9342110, 173.0400000, 167.9338940, 169.9347590, 170.9363220,&
    171.9363777, 172.9382068, 173.9388581, 175.9425680, 174.9670000, 174.9407679, 175.9426824, 178.4900000,&
    173.9400400, 175.9414018, 176.9432200, 177.9436977, 178.9458151, 179.9465488, 180.9479000, 179.9474660,&
    180.9479960, 183.8400000, 179.9467060, 181.9482060, 182.9502245, 183.9509326, 185.9543620, 186.2070000,&
    184.9529557, 186.9557508, 190.2300000, 183.9524910, 185.9538380, 186.9557479, 187.9558360, 188.9581449,&
    189.9584450, 191.9614790, 192.2170000, 190.9605910, 192.9629240, 195.0780000, 189.9599300, 191.9610350,&
    193.9626640, 194.9647740, 195.9649350, 197.9678760, 196.9665520, 196.9665520, 200.5900000, 195.9658150,&
    197.9667520, 198.9682620, 199.9683090, 200.9702850, 201.9706260, 203.9734760, 204.3833000, 202.9723290,&
    204.9744120, 207.2000000, 203.9730290, 205.9744490, 206.9758810, 207.9766360, 208.9803830, 208.9803830,&
    209.0000000, 209.0000000, 210.0000000, 222.0000000, 222.0000000, 223.0000000, 226.0000000, 226.0000000,&
    227.0000000, 232.0381000, 232.0381000, 231.0359000, 231.0359000, 238.0289000, 233.0396280, 234.0409456,&
    235.0439231, 238.0507826, 237.0000000, 237.0000000, 244.0000000, 238.0495534, 239.0521565, 240.0538075,&
    242.0587368, 243.0000000, 243.0000000, 247.0000000, 244.0627463, 246.0672176, 248.0723420, 247.0000000,&
    251.0000000, 252.0829700, 257.0950990, 258.0000000, 259.1010200, 262.1096900, 261.1087500, 262.1141500,&
    266.1219300, 264.1247300, 277.0000000, 268.1388200, 281.0000000, 272.1534800, 285.0000000, 284.0000000,&
    289.0000000, 288.0000000, 293.0000000, 291.0000000, 294.0000000/)
    
  ! get_filename_ext: get file name extension ================================
  dot_pos = index(file_in, '.', back=.true.)  ! position of last dot
  sep_pos = scan(file_in, '/\\', back=.true.) ! position of last separator
  end_pos = len(file_in)
  if (sep_pos > dot_pos) then
    dot_pos = 0
  end if
  if (dot_pos == 0 .or. dot_pos == end_pos) then
    ext = ""
  else
    dot_pos = dot_pos+1 ! skip '.' char
    ext = l_case(file_in(dot_pos:end_pos))
  end if
  if (ext == "ins" .or. ext == "res") then
    ext = "shx"
  end if
  
  ! Compute reflections: mode could be an option (CFL, CIF, SHX, PCR)
  call ReadN_set_Xtal_Structure(file_in, Cell, SpG, A , Mode=ext, file_list=fich_cfl)
  if (A%Natoms .eq. 0) then
    message = "ERROR: cif2hkl: Could not extract a crystallographic structure from file "//&
      trim(file_in)//eol//&
      "       Check file existence/permissions and type (should be a CFL,CIF,ShelX)."//eol
    return
  end if
  if (verbose .ne. 0) then
    message = "file='"//trim(file_in)//"';"//eol
    s1 = trim(ADJUSTL(message))//eol//&
      "% cell         [    a         b         c        alpha     beta      gamma ]"//eol
    write(s2,fmt="(a,6f10.5,a)") "cell=[ ", &
         Cell%cell(1), Cell%cell(2), Cell%cell(3), &
         Cell%ang(1), Cell%ang(2), Cell%ang(3), "];"//eol
    write(s3,fmt="(a,i4,a)") "Spgr='"//trim(SpG%SPG_Symb)//"'; % space group [Number ",&
         SpG%NumSpg, "]"//eol
    message = trim(s1)//trim(s2)//trim(s3)//&
      "%             [    x/a       y/b       z/c      Biso      Occ       Spin      Charge ]"//eol
  end if
  
  !Compute cross section
  sigma_coh=0
  sigma_inc=0
  sigma_abs=0
  mass     =0
  formula  =""
  
  do I=1, A%Natoms
    do Y = 1,n_elements
      if (trim(L_case(A%atom(i)%ChemSymb)) == trim(L_case(element(Y)))) THEN
        ! an atom from the fomula (i) matches one in the isotopes list (y)
        ! compute total cross sections and weight
        sigma_coh = sigma_coh + 4*pi*bcoh(Y)*bcoh(Y)*A%atom(i)%Mult*1E8
        sigma_inc = sigma_inc + 4*pi*binc(Y)*binc(Y)*A%atom(i)%Mult*1E8
        sigma_abs = sigma_abs + Sabs(Y)*A%atom(i)%Mult
        mass      = mass+weight(Y)*A%atom(i)%Mult
        s1 = adjustl(formula)
        write(unit=s2, fmt='(I3)') A%atom(i)%Mult
        formula   = trim(s1)//" "//trim(element(Y))//trim(adjustl(s2))
        ! display verbose information
        ! Label,x/a, y/b, z/c, B, occ, Spin, Charge
        if (verbose .ne. 0) then
          write(s2,fmt="(a,a,a,7f10.5,a)") trim(message),trim(A%atom(i)%lab),"=[",A%atom(i)%X, &
            A%atom(i)%Biso,A%atom(i)%Occ,A%atom(i)%moment,A%atom(i)%Charge,"];"//eol
          message = s2
        end if
        exit
      end if
    end do
  end do
  if (verbose .ne. 0) then
    s1 = trim(message)//"title='"//trim(formula)//" ["//trim(SpG%SPG_Symb)//&
      ", "//trim(SpG%CrystalSys)//", "//trim(SpG%Centre)//"]';"//eol
!    write(*,*) "% sigma_coh  ",  sigma_coh, " coherent   scattering cross section in [barn]"
!    write(*,*) "% sigma_inc  " , sigma_inc, " incoherent scattering cross section in [barn]"
!    write(*,*) "% sigma_abs  " , sigma_abs, " absorption scattering cross section in [barn]"
!    write(*,*) "% density    ",  mass/cell%cellVol*1e24/6.0221409e+23, " in [g/cm^3]"
!    write(*,*) "% weight     ",  mass,         " in [g/mol]"
!    write(*,*) "% Vc         ",  cell%cellVol, " volume of unit cell in [A^3]" 
    message = s1
  end if
  
  ! suppress Sfac computation and output when has --no-output-files option
  if (powxtal(1:1) .ne. '-') then
    MaxNumRef = Get_MaxNumRef(stlmax,Cell%CellVol,mult=SpG%NumOps)
    if (powxtal(1:1) == "p") then
      ! powder mode:
      !    Hkl_Uni(Crystalcell, Spacegroup,Friedel,Value1,Value2,Code,Num_Ref,Reflex, no_order)
      call Hkl_Uni(Cell,Spg,.true.,0.0,stlmax,"s",MaxNumRef,hkl)
    else
      ! SX mode:
      call Hkl_Gen_SXtal(Cell,Spg,0.0,stlmax,MaxNumRef,hkl)
      ! sort SX Bragg peaks
      call sort_d(hkl)
    end if
    ! exit when no reflection can be computed
    if (hkl%Nref == 0) then
      message = "ERROR: cif2hkl: No reflection list can be set from file "//&
        trim(file_in)//eol//&
        "       Check file existence/permissions and type (should be a CFL,CIF,ShelX)."//eol
      return
    end if
    ! mode="nuc" (neutron), "ele" (electrons), "xra" x-rays
    call Structure_Factors(A,SpG,hkl,mode=mode)

    ! get current date/time
    call idate(today)   ! today(1)=day, (2)=month, (3)=year
    call itime(now)     ! now(1)=hour, (2)=minute, (3)=second
    
    !Write reflection file
    open( unit=lun,file=file_out,status="replace",action="write")
    write(unit=lun,fmt="(a,a,a)")    "# TITLE  ",     trim(formula), " ["//trim(SpG%CrystalSys)//", "//trim(SpG%Centre)//"]"
    write(unit=lun,fmt="(a)") "#        a         b         c       alpha     beta      gamma"
    write(unit=lun,fmt="(a,6f10.5)") "# CELL ", Cell%cell(1), Cell%cell(2), Cell%cell(3), Cell%ang(1), Cell%ang(2), Cell%ang(3)
    write(unit=lun,fmt="(a,a,a,i4,a)") "# SPCGRP  ",    trim(SpG%SPG_Symb)," [Number ",SpG%NumSpg, "]"
    write(unit=lun,fmt="(a)") "#                    X         Y         Z         B         Occ       Spin      Charge"
    do I=1, A%Natoms
      write(unit=lun,fmt="(a,a,7f10.5)") "# Atom  ",A%atom(i)%lab,&
            A%atom(i)%X, &
            A%atom(i)%Biso,A%atom(i)%Occ,A%atom(i)%moment,A%atom(i)%Charge
    end do
    write(unit=lun,fmt="(a)")        "# COMMAND cif2hkl "//trim(file_in)//" --output "//trim(file_out)
    write(unit=lun,fmt="(a)")        "# CIF2HKL (c) Synchrotron Soleil, based on CrysFML"
    write(unit=lun,fmt=1000) today, now
  1000 format ( '# DATE    ', i2.2, '/', i2.2, '/', i4.4, ' at ', i2.2, ':', i2.2, ':', i2.2)
    write(unit=lun,fmt="(a)") "#"  
    write(unit=lun,fmt="(a,f14.5,a)") "# Physical parameters:"
    Select Case (l_case(mode(1:3)))
      Case("nuc")
	    write(unit=lun,fmt="(a,f14.5,a)") "# sigma_coh  ",  sigma_coh, " coherent   scattering cross section in [barn]"
	    write(unit=lun,fmt="(a,f14.5,a)") "# sigma_inc  " , sigma_inc, " incoherent scattering cross section in [barn]"
	    write(unit=lun,fmt="(a,f14.5,a)") "# sigma_abs  " , sigma_abs, " absorption scattering cross section in [barn]"
    End Select
    write(unit=lun,fmt="(a,f14.5,a)") "# density    ",  mass/cell%cellVol*1e24/6.0221409e+23, " in [g/cm^3]"
    write(unit=lun,fmt="(a,f14.5,a)") "# weight     ",  mass,         " in [g/mol]"
    write(unit=lun,fmt="(a,f14.5,a)") "# Vc         ",  cell%cellVol, " volume of unit cell in [A^3]" 
    write(unit=lun,fmt="(a,f14.5,a)") "# lattice_a  ", Cell%cell(1),  " lattice parameter a in [Angs]"
    write(unit=lun,fmt="(a,f14.5,a)") "# lattice_b  ", Cell%cell(2),  " lattice parameter b in [Angs]" 
    write(unit=lun,fmt="(a,f14.5,a)") "# lattice_c  ", Cell%cell(3),  " lattice parameter c in [Angs]"
    write(unit=lun,fmt="(a,f14.5,a)") "# lattice_aa ", Cell%ang(1),   " lattice angle alpha in [deg]"
    write(unit=lun,fmt="(a,f14.5,a)") "# lattice_bb ", Cell%ang(2),   " lattice angle beta in [deg]"
    write(unit=lun,fmt="(a,f14.5,a)") "# lattice_cc ", Cell%ang(3),   " lattice angle gamma in [deg]"
    Select Case (l_case(mode(1:3)))
      Case("nuc")
        write(unit=lun,fmt="(a)") "# STRUCTURE FACTORS(NEUTRONS)"
      Case("xra")
        write(unit=lun,fmt="(a)") "# STRUCTURE FACTORS(X-RAYS)"
      Case("ele")
        write(unit=lun,fmt="(a)") "# STRUCTURE FACTORS(ELECTRONS)"
      Case default
        message = "ERROR: cif2hkl: unknown option: --mode "//trim(mode)//eol
        return
    End Select
    write(unit=lun,fmt="(a)") "#"
    write(unit=lun,fmt="(a)") "# Format parameters: Crystallographica format"
    write(unit=lun,fmt="(a)") "# column_h  1"
    write(unit=lun,fmt="(a)") "# column_k  2"
    write(unit=lun,fmt="(a)") "# column_l  3"
    write(unit=lun,fmt="(a)") "# column_j  4   multiplicity 'j'"
    write(unit=lun,fmt="(a)") "# column_d  5   d-spacing 'd' in [Angs]"
    write(unit=lun,fmt="(a)") "# column_F2 6   norm of scattering factor |F|^2 in [barn]"
    write(unit=lun,fmt="(a)") "#"
    write(unit=lun,fmt="(a,i5,a,f10.4,a)") "# List ",hkl%Nref, " reflections for lambda > ", &
            lambda, " [Angs], decreasing d-spacing."
    write(unit=lun,fmt="(a)") &
            "# H   K   L     Mult    dspc                   |Fc|^2"
    do i=1,hkl%Nref
      F    = hkl%ref(i)%Fc
      if ((F+1.0) .ne. F) then ! except for NaN's
        write(unit=lun,fmt="(3(i4,1x),i5,1x,2(f13.5,1x),f25.5)") &
        hkl%ref(i)%h, hkl%ref(i)%mult, &
        0.5/hkl%ref(i)%S, hkl%ref(i)%Fc*hkl%ref(i)%Fc
      end if
    end do
    
    ! write with Write_RefList_Info(Rfl, Iunit, Mode)
    ! call Write_RefList_Info(hkl, lun, mode)
    
    close (unit=lun)
  end if

End subroutine CFML_cif2hkl


subroutine print_version(pgmname,message)
  ! Show program version. pgmname = argv[0] ================================
  character*1024 pgmname
  character*1024, intent(out) :: message
  
  character*256 AUTHOR, DATE, VERSION
  character*2   eol
  
  eol=char(13)//char(10)
  
  AUTHOR ="Farhi E. [emmanuel.farhi@synchrotron-soleil.fr]"//eol//"  using crysFML <https://code.ill.fr/scientific-software/crysfml/>"
  DATE   ="Sept 21st 2022"
  VERSION="1.4.2"
  
  
  message = trim(pgmname)//" "//trim(VERSION)//" ("//trim(DATE)//") by "//trim(AUTHOR)//eol//&
   "  Copyright (C) 2009-2019 Institut Laue Langevin, EUPL."//eol//&
   "  Copyright (C) 2020-     Synchrotron Soleil,     GPL2."//eol//&
   "  This is free software; see the source for copying conditions."//eol//&
   "  There is NO warranty; not even for MERCHANTABILITY or FITNESS"//eol//&
   "  FOR A PARTICULAR PURPOSE."//eol
end subroutine print_version

subroutine print_usage(pgmname,message)
  ! Show program help. pgmname = argv[0] ===================================
  character*1024 pgmname
  character*4096,intent(out) :: message
  
  character*2   eol
  
  eol=char(13)//char(10)

  message = "Usage: "//trim(pgmname)//" [options][-o outfile] file1 file2 ..."//eol//&
    "Action: Read a CIF/CFL/SHX/PCR crystallographic description"//eol//&
    "        and generates a HKL F^2 reflection list."//eol//&
    "Input:"//eol//&
    "  file1...          Input file in CIF, PCR, CFL, SHX, INS, RES format."//eol//&
    "                      The file format is determined from its extension"//eol//&
    "                        .CIF           Crystallographic Information File"//eol//&
    "                        .PCR/.CFL      FullProf file"//eol//&
    "                        .SHX/.INS/.RES ShelX file"//eol//&
    "Output:"//eol//&
    "  a file with readable header, and reflection list with columns"//eol//&
    "    [ H K L Multiplicity Sin(Theta/Lambda) d_spacing |F|^2 ]"//eol//&
    "Options:"//eol//&
    "--help     or -h    Show this help"//eol//&
    "--version  or -v    Display program version"//eol//&
    "--out FILE          Specify the name of the next output file."//eol//&
    "   -o FILE            Default is to add .hkl to the initial file name."//eol//&
    "--lambda LAMBDA     Set the incoming probe wavelength [Angs]."//eol//&
    "   -l    LAMBDA       Default is 0.5"//eol//&
    "--powder   or -p    Generate a list of unique HKL reflections (for powders). Default."//eol//&
    "--xtal     or -x    Generate a list of all HKL reflections (for single crystals)."//eol//&
    "--mode MODE         Generate structure factors for given probe, where MODE is"//eol//&
    "   -m  MODE           NUC=neutron(default) XRA=xrays ELE=electrons"//eol//&
    "--verbose           Display processing details."//eol//&
    "--no-output-files   Just read the CIF/CFL/ShelX file (for checking)."//eol//&
    "Example: "//trim(pgmname)//" --powder --mode NUC -o CaF2.laz CaF2.cfl"//eol

end subroutine print_usage

!*****************************************************************************
! main: Entry point
!****************************************************************************/
program cif2hkl

  character(len=1024) :: pgmname 
  integer             :: i=0
  integer             :: argc=0
  character(len=1024) :: argv
  character(len=1024) :: outfile
  character(len=1024) :: ext            ! input file name extension
  real*8              :: lambda= 0.5    ! wavelength (determines minimum d)
  character(len=1024) :: powxtal="p"    ! 'p'=powder, 's'=SX output file, '-'=no output
  integer             :: verbose=0      ! verbose mode to display additional information
  character(len=1024) :: mode="NUC"     ! 'NUC','XRA','ELE'
  
  character*4096      :: message

  argc    = iargc()       ! number of arguments
  call getarg(0, pgmname) ! program name
  outfile = ""
  message = ""

  if (argc == 0) then
    call print_usage(pgmname, message)
    write(*,*) trim(message)
  else 
    do while (i < argc)
      i = i + 1
      call getarg(i, argv)
      if (argv(1:2) == "-h" .or. argv(1:6) == "--help") then
        call print_usage(pgmname, message)
        write(*,*) trim(message)
      end if
      if (argv(1:2) == "-v" .or. argv(1:9) == "--version") then
        call print_version(pgmname, message)
        write(*,*) trim(message)
      end if
      if ( (argv(1:8) == "--lambda" .or. argv(1:2) == "-l") .and. i<argc) then
        i=i+1
        call getarg(i, message)
        read(message, *) lambda
      end if
      if ( (argv(1:5) == "--out".or. argv(1:2) == "-o") .and. i < argc) then
        i=i+1
        call getarg(i, outfile)
      end if
      if ( (argv(1:6) == "--mode".or. argv(1:2) == "-m") .and. i < argc) then
        i=i+1
        call getarg(i, mode)
      end if
      if (argv(1:2) == "-p" .or. argv(1:3) == "--p") then
        powxtal = "p"
      end if
      if (argv(1:2) == "-x" .or. argv(1:3) == "--x") then
        powxtal = "x"
      end if
      if (argv(1:8) == "-verbose" .or. argv(1:9) == "--verbose") then
        verbose = 1
      end if
      if (argv(1:4) == "--no") then
        powxtal = "-"
      end if
      if (argv(1:1) .ne. '-') then
        ! convert argv[i]: process conversion

        ! check outfile
        if (len_trim(outfile) == 0) then
          outfile = trim(argv)//".hkl"  ! append .hkl extension
        end if
        call cfml_cif2hkl(argv, outfile, lambda, powxtal, verbose, message, mode)
        write(*,*) trim(message)

        ! revert outfile to default
        outfile = ""
      end if
    end do
  end if
  
end program cif2hkl

