/* File: test_suite.h; Copyright and License: see below */

#ifndef TEST_SUITE_H
#define TEST_SUITE_H

/* public file for the doxygen documentation: */
/*!
 *  \file
 *  \brief Provides a set of unit test functions - in line with the xunit architecture.
 *  see https://en.wikipedia.org/wiki/XUnit
 */

#include <stdbool.h>

/*!
 *  \brief constants of simple test suite
 */
enum test_suite_max_enum {
    TEST_SUITE_MAX_TEST_CASES = 48,  /*!< maximum number test cases per test suite */
};

/*!
 *  \brief attributes of a test suite: fixture-setup, fixture-tear-down, set of test cases
 */
struct test_suite_struct {
    const char *name;
    void (*setup) (void);  /*!< pointer to setup function of test fixture */
    void (*teardown) (void);  /*!< pointer to teardown function of test fixture */
    unsigned int test_case_count;  /*!< number of test cases */
    const char *(test_case_name[TEST_SUITE_MAX_TEST_CASES]);  /*!< array of test case names */
    void (*(test_case[TEST_SUITE_MAX_TEST_CASES])) (void);  /*!< array of pointers to test case functions */
};

typedef struct test_suite_struct test_suite_t;


/*!
 *  \brief initializes the test_suite_t
 *
 *  \param name name of test suite
 *  \param setup function pointer to setup function
 *  \param teardown function pointer to teardown function
 *  \param this_ pointer to own object attributes
 */
static inline void test_suite_init( test_suite_t *this_,
                                    const char *name,
                                    void (*setup) (void),
                                    void (*teardown) (void)
                                  );

/*!
 *  \brief destroys the test_suite_t
 *
 *  \param this_ pointer to own object attributes
 */
static inline void test_suite_destroy( test_suite_t *this_ );

/*!
 *  \brief adds a test case to the test_suite_t
 *
 *  \param this_ pointer to own object attributes
 *  \param name name of test case
 *  \param test_case function pointer to test_case function
 */
static inline void test_suite_add_test_case( test_suite_t *this_,
                                             const char *name,
                                             void (*test_case) (void)
                                           );

/*!
 *  \brief adds a test case to the test_suite_t
 *
 *  \param this_ pointer to own object attributes
 *  \return number of test cases
 */
static inline unsigned int test_suite_get_test_case_count( test_suite_t *this_ );

/*!
 *  \brief executes a test case
 *
 *  \param this_ pointer to own object attributes
 *  \param index index of the test case, value between 0 and (test_case_count-1)
 *  \return true in case of success
 */
static inline bool test_suite_run_test_case( test_suite_t *this_, unsigned int index );

/*!
 *  \brief gets the name of the test_suite_t
 *
 *  \param this_ pointer to own object attributes
 *  \return name of test suite
 */
static inline const char* test_suite_get_name( test_suite_t *this_ );

/*!
 *  \brief gets the name of a test case in the test_suite_t
 *
 *  \param this_ pointer to own object attributes
 *  \param index index of the test case, value between 0 and (test_case_count-1)
 *  \return name of test case
 */
static inline const char* test_suite_get_test_case_name( test_suite_t *this_, unsigned int index );


#include "test_suite.inl"

#endif  /* TEST_SUITE_H */


/*
Copyright 2019-2023 Andreas Warnke

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
