/*
  DB Mixer
  ========
  Description: 
    a DJ Mixer style GUI interface to the DBMix system.

	Copyright (c) 1999, 2000 Robert Michael S Dean

	Author: Robert Michael S Dean
	Version: 1.0


   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public Licensse as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

 */


#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <sys/shm.h>
#include <xmms/plugin.h>
#include <dbchannel.h>
#include <dbaudiolib.h>
#include <dbdebug.h>
#include <dbsoundcard.h>

#include "dbmixer.h"

extern int debug_level;

extern local_channel *local_channels;
extern dbfsd_data * sysdata;

GtkAdjustment * main_volume_adj, * cue_volume_adj, * balance_adj;

int master_mixer_flag, cue_mixer_flag;
int master_mixer_cmd, cue_mixer_cmd;

int master_volume;
int cue_volume;

int Soundcard_Mixer_Init()
{
	int fd1;
	int devs;

	master_mixer_flag = cue_mixer_flag = FALSE;
	master_mixer_cmd = cue_mixer_cmd = 0;


	/* verify the existence of primary mixer device */
	fd1 = open(MASTER_MIXER,O_RDONLY);

	if(fd1 != -1)
	{
		if(ioctl(fd1,SOUND_MIXER_READ_DEVMASK, &devs) != SUCCESS)
		{
		    Error("failed to open primary mixer. %s\n");
			return FAILURE;
		}
	
		Debug("Master mixer is %s",MASTER_MIXER);
	
		if(devs & SOUND_MASK_PCM) 
		{
			master_mixer_flag = TRUE;
			master_mixer_cmd = SOUND_MIXER_WRITE_VOLUME;
		}
		else 
		{
			if (devs & SOUND_MASK_VOLUME)
			{
				master_mixer_flag = TRUE;
				master_mixer_cmd = SOUND_MIXER_WRITE_VOLUME;
			}
			else
			{
				master_mixer_flag = FALSE;
			}
		}
	}

	close(fd1);

	/* verify existence of secondary mixer */

	/* if single output mode, don't need secondary mixer */
	if (!sysdata->single_output && sysdata->cue_enabled)
	{
		fd1 = open(CUE_MIXER,O_RDONLY);

		if(fd1 != -1)
		{
			if(ioctl(fd1,SOUND_MIXER_READ_DEVMASK, &devs) != SUCCESS)
			{
				perror("failed to open cue mixer.\n");
			}
			
			if(devs & SOUND_MASK_PCM) 
			{
				cue_mixer_flag = TRUE;
				cue_mixer_cmd = SOUND_MIXER_WRITE_VOLUME;
			}
			else 
			{
				if (devs & SOUND_MASK_VOLUME)
				{
					cue_mixer_flag = TRUE;
					cue_mixer_cmd = SOUND_MIXER_WRITE_VOLUME;
				}
				else
				{
					cue_mixer_flag = FALSE;
				}
			}
		}
	}

	close(fd1);

	return SUCCESS;
}


int update_soundcards(gpointer data)
{
	int fd, v, i;
	
	if(master_mixer_flag == FALSE) {return FAILURE;}
	
	fd = open(MASTER_MIXER,O_RDONLY);
		
	if (fd != -1)
	{
		if(ioctl(fd, SOUND_MIXER_READ_VOLUME, &v) != 0)
		{
			Error("Could not set master volume.");
		}
		close(fd);
		
		i = 100 - (v & 0x000000ff);
/* 		i = (v & 0x000000ff); */

		
		Debug("setting main volume to be %d",i);
		
		gtk_adjustment_set_value(main_volume_adj,i);
	}
	
	if(Cue_Enabled((&local_channels[0])) && !sysdata->single_output)
	{
		fd = open(CUE_MIXER,O_RDONLY);
		
		if (fd != -1)
		{
			if(ioctl(fd, SOUND_MIXER_READ_VOLUME, &v) != 0)
			{
				Error("Could not set cue volume.");
			}
			close(fd);
			
			i = 100 - (v & 0x000000ff);
			
			Debug("setting cue volume to be %d",i);
			
			gtk_adjustment_set_value(cue_volume_adj,i);
		}
	}
	
	return SUCCESS;
}


int update_balance(gpointer data)
{
	int value;

	if (sysdata->left_balance < 100)
	{
		value = 200 - sysdata->left_balance;
	}
	else
	{
		if (sysdata->right_balance < 100)
		{
		    value = sysdata->right_balance;
		}
		else
		{
			value = 100;				
		}
	}

	gtk_adjustment_set_value(balance_adj, value);

	Debug("balance set to be %d\n",value);

	return SUCCESS;
}



/***********************************************************************/
/*               callback functions for soundcard widgets              */
/***********************************************************************/

void main_level_scale_changed(GtkAdjustment * adj)
{
	int fd, v, i;

	if(master_mixer_flag == FALSE) {return;}

	fd = open(MASTER_MIXER,O_WRONLY);

	if (fd != -1)
	{
		i = 100 - (gint) adj->value;

		master_volume = i;

		v = (i << 8) | i;

		if(ioctl(fd, master_mixer_cmd, &v) != 0)
		{Error("Could not set master volume.");}
		else {Debug("Set master volume to be %d",i);}

		close(fd);
	}

}


void cue_level_scale_changed(GtkAdjustment * adj)
{
	int fd, v, i;

	if (sysdata->single_output)
	{
		cue_volume = 100 - (gint) adj->value;
	}

	if(cue_mixer_flag == FALSE) {return;}
	
	fd = open(CUE_MIXER,O_RDONLY);
	
	if (fd != -1)
	{
		i = 100 - (gint) adj->value;
		v = (i << 8) | i;
		
		cue_volume = i;
		
		if(ioctl(fd, cue_mixer_cmd, &v) != 0)
		{Error("Could not set cue volume.");}
		else {Debug("Set cue volume to be %d",i);}
		
		close(fd);
	}
}


void balance_scale_changed(GtkAdjustment * adj)
{
	if (adj->value < 100)
	{
		sysdata->right_balance = adj->value;
		sysdata->left_balance = 100;
		return;
	}

	if (adj->value > 100)
	{
		sysdata->left_balance = 200 - adj->value;
		sysdata->right_balance = 100;
		return;
	}

	if (adj->value == 100)
	{
		sysdata->right_balance = 100;
		sysdata->left_balance = 100;
		return;
	}
}

void talkover_button_clicked(GtkWidget *w, gchar* data)
{
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		sysdata->talkover_enabled = 1;
	}
	else
	{
		sysdata->talkover_enabled = 0;
	}
}

void cue_split_button_clicked(GtkWidget *w, gchar* data)
{	
	/* single output mode is on, cannot turn off cue split */
	if (sysdata->single_output) return;

	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	{
		sysdata->cue_split = 1;
	}
	else
	{
		sysdata->cue_split = 0;
	}
}



/***********************************************************************/
/*               widget creation functions for soundcards              */
/***********************************************************************/

GtkWidget* Create_Soundcard_Controls()
{
	GtkWidget * soundcard_box;
	GtkWidget * label;
	GtkWidget * level_scale;
	GtkWidget * master_box;
	GtkWidget * cue_box;
	GtkWidget * balance_scale;
	GtkWidget * volume_box;
	GtkWidget * balance_box;
	GtkWidget * cue_split_button;
	GtkWidget * talkover_button;

	/* create master and cue main volume controls */
	soundcard_box = gtk_vbox_new(FALSE,0);
	volume_box = gtk_hbox_new(FALSE,5);
	master_box = gtk_vbox_new(FALSE,5);
	cue_box = gtk_vbox_new(FALSE,5);

	gtk_container_set_border_width(GTK_CONTAINER(soundcard_box),0);

	/* add talkover button */
	{
		talkover_button = (GtkWidget *)gtk_toggle_button_new();
		label = gtk_label_new(TALKOVER_STR);
		gtk_container_add(GTK_CONTAINER(talkover_button),label);
		gtk_box_pack_start(GTK_BOX(soundcard_box),GTK_WIDGET(talkover_button),
						   TRUE,FALSE,0);
	   
		gtk_signal_connect (GTK_OBJECT(talkover_button), "clicked", 
							GTK_SIGNAL_FUNC(talkover_button_clicked),0);
		gtk_widget_show(label);
		gtk_widget_show(talkover_button);
		
	   if(!sysdata->cue_enabled)
	   {
		   gtk_widget_set_sensitive(GTK_WIDGET(talkover_button), FALSE);
	   }

		gtk_toggle_button_set_active((GtkToggleButton*)talkover_button,sysdata->talkover_enabled);
	}


	/* add cue split button */
	{
		cue_split_button = (GtkWidget *)gtk_toggle_button_new();
		label = gtk_label_new(CUE_SPLIT_STR);
		gtk_container_add(GTK_CONTAINER(cue_split_button),label);
		gtk_box_pack_start(GTK_BOX(soundcard_box),GTK_WIDGET(cue_split_button),
						   TRUE,FALSE,0);
	   
		gtk_signal_connect (GTK_OBJECT(cue_split_button), "clicked", 
							GTK_SIGNAL_FUNC(cue_split_button_clicked),0);
		gtk_widget_show(label);
		gtk_widget_show(cue_split_button);
		
	   if(!sysdata->cue_enabled || sysdata->single_output)
	   {
		   gtk_widget_set_sensitive(GTK_WIDGET(cue_split_button), FALSE);
	   }

		gtk_toggle_button_set_active((GtkToggleButton*)cue_split_button,sysdata->cue_split);
	}
	
	/* create master volume level */
	{
		label = gtk_label_new(MASTER_LEVEL_STR);
		gtk_box_pack_start(GTK_BOX(master_box),label,FALSE,FALSE,0);
		
		gtk_widget_show(label);		
		
		main_volume_adj = (GtkAdjustment*) gtk_adjustment_new(0.0,0.0,100.0,1.0,10.0,0.0);
		gtk_signal_connect (GTK_OBJECT(main_volume_adj), "value_changed", 
							GTK_SIGNAL_FUNC(main_level_scale_changed),NULL);
		
		level_scale = gtk_vscale_new(GTK_ADJUSTMENT(main_volume_adj));
		gtk_scale_set_digits((GtkScale*)level_scale,0);
		gtk_scale_set_value_pos((GtkScale*)level_scale,GTK_POS_TOP);	
		
		gtk_box_pack_start(GTK_BOX(master_box),(GtkWidget*)level_scale,FALSE,FALSE,0);

		gtk_widget_set_usize(level_scale,10,100);

		gtk_widget_show((GtkWidget*)level_scale);
	}	

	/* create cue volume level */
	{
		label = gtk_label_new(CUE_LEVEL_STR);
		gtk_box_pack_start(GTK_BOX(cue_box),label,FALSE,FALSE,0);
		
		gtk_widget_show(label);		
		
		cue_volume_adj = (GtkAdjustment*) gtk_adjustment_new(0.0,0.0,100.0,1.0,10.0,0.0);
		gtk_signal_connect (GTK_OBJECT(cue_volume_adj), "value_changed", 
							GTK_SIGNAL_FUNC(cue_level_scale_changed),NULL);
		
		level_scale = gtk_vscale_new(GTK_ADJUSTMENT(cue_volume_adj));
		gtk_scale_set_digits((GtkScale*)level_scale,0);
		gtk_scale_set_value_pos((GtkScale*)level_scale,GTK_POS_TOP);	
		
		gtk_box_pack_start(GTK_BOX(cue_box),(GtkWidget*)level_scale,FALSE,FALSE,0);
		
		if(!sysdata->cue_enabled)
		{
			gtk_widget_set_sensitive(GTK_WIDGET(level_scale),FALSE);
		}
		
		gtk_widget_set_usize(level_scale,10,100);

		gtk_widget_show((GtkWidget*)level_scale);
	}

	update_soundcards(NULL);

	/* create balance scale */
	{
		balance_box = gtk_vbox_new(FALSE,0);
		
		label = (GtkWidget *) gtk_label_new(BALANCE_STR);
		gtk_widget_show(label);
		gtk_box_pack_start(GTK_BOX(soundcard_box),label,FALSE,FALSE,0);
		
		balance_adj = (GtkAdjustment *) gtk_adjustment_new(100.0,0,200,1.0,10.0,0.0);
		balance_scale = (GtkWidget *) gtk_hscale_new(GTK_ADJUSTMENT(balance_adj));
		gtk_scale_set_digits((GtkScale*)balance_scale,1);

		gtk_signal_connect (GTK_OBJECT(balance_adj), "value_changed", 
							GTK_SIGNAL_FUNC(balance_scale_changed),NULL);

		update_balance(NULL);

		gtk_box_pack_start(GTK_BOX(balance_box),balance_scale,FALSE,FALSE,0);		
		gtk_widget_show(balance_scale);
		gtk_widget_show(balance_box);
	}

	gtk_widget_show(master_box);
	gtk_widget_show(cue_box);
	gtk_widget_show(volume_box);
	gtk_box_pack_start(GTK_BOX(volume_box),master_box,FALSE,FALSE,5);
	gtk_box_pack_start(GTK_BOX(volume_box),cue_box,FALSE,FALSE,5);
	gtk_box_pack_start(GTK_BOX(soundcard_box),balance_box,FALSE,FALSE,0);

	gtk_box_pack_start(GTK_BOX(soundcard_box),volume_box,FALSE,FALSE,0);

	return soundcard_box;
}

