#!/usr/bin/python3
################################################################################
# PROJECT: Wavefront Alignments Algorithms 
# LICENCE: MIT License 
# AUTHOR(S): Santiago Marco-Sola <santiagomsola@gmail.com>
# DESCRIPTION: Process debug/report files generated by WFAlib
# USAGE: python3 wfa.debug.py -h
################################################################################

import sys
import argparse
import os
import re

import numpy as np
import matplotlib.pyplot as plt
from scipy.constants.constants import alpha

################################################################################
# Parse inputs
################################################################################
def cigar_compute_differences(cigar):
  # Parse CIGAR
  #   cigar = "10M3D5M1I10M"
  #   cigar_vector = [('10', 'M'), ('3', 'D'), ('5', 'M'), ('1', 'I'), ('10', 'M')]
  cigar_vector = re.findall(r'(\d+)([MXDI])',cigar)
  # Compute edit distance
  score = 0
  for op in cigar_vector:
    if op[1] in "DI": score += int(op[0])
    if op[1] in "X": score += int(op[0])
  return score

################################################################################
# Parse inputs
################################################################################
def parse_input_debug(file_name):
  # Init
  debug_scores = []
  debug_length_pattern = []
  debug_length_text = []
  debug_error_rate = []
  debug_time = []
  debug_memory = []
  # Read/Write files
  input_file = open(file_name,"rt")
  while True:
    # Read line
    line = input_file.readline()
    if not line: break
    # Split
    fields = line.split()
    # Check first field (label)
    if len(fields) == 0 or fields[0] != "[WFA::Debug]": continue
    try:
      # Store fields
      score = int(fields[1])
      plen = int(fields[2])
      tlen = int(fields[3])
      time = float(fields[5])
      mem = int(fields[6][:-2])
      #attr = fields[7]
      cigar = fields[8]
      max_length = max(plen,tlen)
      error_rate = float(cigar_compute_differences(cigar)) / float(max_length)
      debug_scores.append(score)
      debug_length_pattern.append(plen)
      debug_length_text.append(tlen)
      debug_time.append(time)
      debug_memory.append(mem)
      debug_error_rate.append(min(100.0*error_rate,100.0))
    except:
      print("Error parsing: '%s'" % line)
  # Return debug info
  input_debug = {}
  input_debug["SCORES"] = debug_scores
  input_debug["LENGHT_PATTERN"] = debug_length_pattern
  input_debug["LENGHT_TEXT"] = debug_length_text
  input_debug["ERROR_RATE"] = debug_error_rate
  input_debug["TIME"] = debug_time
  input_debug["MEMORY"] = debug_memory
  return input_debug

    
################################################################################
# Plot debug info
################################################################################
def plot_debug_performance(input1_name,input1_debug,input2_name,input2_debug):
  # Init panels
  fig, ((ax1,ax2,ax3),(ax4,ax5,ax6)) = plt.subplots(nrows=2,ncols=3)
  i1_color = "royalblue"
  i2_color = "darkorange"
  
  # Plot time vs error-rate (dots)
  ax1.set_xlabel('Error/Differences (%)')
  ax1.set_ylabel('Time (ms)')
  ax1.set_yscale('log')
  ax1.xaxis.grid(True)
  ax1.yaxis.grid(True)
  ax1.plot(input1_debug["ERROR_RATE"],input1_debug["TIME"],'+',color=i1_color,markersize=0.5,alpha=1)
  if input2_debug is not None:
    ax1.plot(input2_debug["ERROR_RATE"],input2_debug["TIME"],'x',color=i2_color,markersize=0.5,alpha=1)
    
  # Plot time vs error (dots)
  ax2.set_xlabel('Error/Differences (%)')
  ax2.set_ylabel('Length')
  ax2.xaxis.grid(True)
  ax2.yaxis.grid(True)
  ax2.plot(input1_debug["ERROR_RATE"],input1_debug["LENGHT_PATTERN"],'+',color=i1_color,markersize=0.5,alpha=1)
  if input2_debug is not None:
    ax2.plot(input2_debug["ERROR_RATE"],input2_debug["LENGHT_PATTERN"],'x',color=i2_color,markersize=0.5,alpha=1)
    
  # Plot time vs length (dots)
  ax3.set_xlabel('Length')
  ax3.set_ylabel('Time (ms)')
  ax3.xaxis.grid(True)
  ax3.yaxis.grid(True)
  ax3.plot(input1_debug["LENGHT_PATTERN"],input1_debug["TIME"],'+',color=i1_color,markersize=0.5,alpha=1)
  if input2_debug is not None:
    ax3.plot(input2_debug["LENGHT_PATTERN"],input2_debug["TIME"],'x',color=i2_color,markersize=0.5,alpha=1)
    
  # Plot time (histogram)
  ax4.set_xlabel('Time (ms)')
  ax4.set_ylabel('Seq. Count')
  ax4.set_yscale('log')
  ax4.xaxis.grid(True)
  ax4.yaxis.grid(True)
  if input2_debug is None:
    n, bins, patches = ax4.hist(input1_debug["TIME"],50,facecolor=i1_color,alpha=0.5)
  else:
    range_min = min(min(input1_debug["TIME"]),min(input2_debug["TIME"]))
    range_max = max(max(input1_debug["TIME"]),max(input2_debug["TIME"]))
    n, bins, patches = ax4.hist(input1_debug["TIME"],50,range=[range_min,range_max],facecolor=i1_color,alpha=0.5) 
    n, bins, patches = ax4.hist(input2_debug["TIME"],50,range=[range_min,range_max],facecolor=i2_color,alpha=0.5)
    
  # Plot memory (histogram)
  ax5.set_xlabel('Memory (MB)')
  ax5.set_ylabel('Seq. Count')
  ax5.set_yscale('log')
  ax5.xaxis.grid(True)
  ax5.yaxis.grid(True)
  if input2_debug is None:
    n, bins, patches = ax5.hist(input1_debug["MEMORY"],50,facecolor=i1_color,alpha=0.75)
  else:
    range_min = min(min(input1_debug["MEMORY"]),min(input2_debug["MEMORY"]))
    range_max = max(max(input1_debug["MEMORY"]),max(input2_debug["MEMORY"]))
    n, bins, patches = ax5.hist(input1_debug["MEMORY"],50,range=[range_min,range_max],facecolor=i1_color,alpha=0.5)
    n, bins, patches = ax5.hist(input2_debug["MEMORY"],50,range=[range_min,range_max],facecolor=i2_color,alpha=0.5)

#   # Plot time vs length (histogram)
#   ax6.set_xlabel('Length')
#   ax6.set_ylabel('Time (ms)')
#   ax6.xaxis.grid(True)
#   ax6.yaxis.grid(True)
#   if input2_debug is None:
#     n, bins, patches = ax6.hist(input1_debug["MEMORY"],50,facecolor=i1_color,alpha=0.75)
#   else:
#     range_min = min(min(input1_debug["MEMORY"]),min(input2_debug["MEMORY"]))
#     range_max = max(max(input1_debug["MEMORY"]),max(input2_debug["MEMORY"]))
#     n, bins, patches = ax6.hist(input1_debug["MEMORY"],50,range=[range_min,range_max],facecolor=i1_color,alpha=0.5)
#     n, bins, patches = ax6.hist(input2_debug["MEMORY"],50,range=[range_min,range_max],facecolor=i2_color,alpha=0.5)  
    
  # Show or save
  filename1 = os.path.basename(input1_name) if input1_debug is not None else "N/A"
  filename2 = os.path.basename(input2_name) if input2_debug is not None else "N/A"
  fig.suptitle("Performance information (blue='%s',orange='%s')" % (filename1,filename2),fontsize=12)
  plt.show()
  
def plot_debug_workload(input1_name,input1_debug,input2_name,input2_debug):
  # Init panels
  fig, ((ax1,ax2),(ax3,ax4)) = plt.subplots(nrows=2,ncols=2)
  i1_color = "royalblue"
  i2_color = "darkorange"
  # Plot sequence lengths
  ax1.set_xlabel('Pattern Length')
  ax1.set_ylabel('Text Length')
  ax1.xaxis.grid(True)
  ax1.yaxis.grid(True)
  ax1.set_aspect('equal')
  ax1.plot(input1_debug["LENGHT_PATTERN"],input1_debug["LENGHT_TEXT"],'o',color=i1_color,markersize=0.5,alpha=1)
  if input2_debug is not None:
    ax2.set_xlabel('Pattern Length')
    ax2.set_ylabel('Text Length')
    ax2.xaxis.grid(True)
    ax2.yaxis.grid(True)
    ax2.set_aspect('equal')
    ax2.plot(input2_debug["LENGHT_PATTERN"],input2_debug["LENGHT_TEXT"],'x',color=i2_color,markersize=0.5,alpha=1)
  # Plot score histogram
  ax3.set_xlabel('Score')
  ax3.xaxis.grid(True)
  ax3.yaxis.grid(True)
  ax3.set_xlabel('Score reported')
  ax3.set_ylabel('Seq. Count')
  y, bin_length = np.histogram(input1_debug["SCORES"],bins=100)
  bin_centers = 0.5 * (bin_length[1:] + bin_length[:-1])
  ax3.plot(bin_centers,y,linestyle='--',marker='o',markersize=3,c=i1_color)
  if input2_debug is not None:
    y, bin_length = np.histogram(input2_debug["SCORES"],bins=100)
    bin_centers = 0.5 * (bin_length[1:] + bin_length[:-1])
    ax3.plot(bin_centers,y,linestyle='--',marker='o',markersize=3,c=i2_color)
  # Plot time histogram
  ax4.set_xlabel('Error/Differences (%)')
  ax4.set_ylabel('Seq. Count')
  ax4.xaxis.grid(True)
  ax4.yaxis.grid(True)
  y, bin_length = np.histogram(input1_debug["ERROR_RATE"],bins=100)
  bin_centers = 0.5 * (bin_length[1:] + bin_length[:-1])
  ax4.plot(bin_centers,y,linestyle='--',marker='o',markersize=3,c=i1_color)
  if input2_debug is not None:
    y, bin_length = np.histogram(input2_debug["ERROR_RATE"],bins=100)
    bin_centers = 0.5 * (bin_length[1:] + bin_length[:-1])
    ax4.plot(bin_centers,y,linestyle='--',marker='o',markersize=3,c=i2_color)
  # Show or save
  filename1 = os.path.basename(input1_name) if input1_debug is not None else "N/A"
  filename2 = os.path.basename(input2_name) if input2_debug is not None else "N/A"
  fig.suptitle("Workload information (blue='%s',orange='%s')" % (filename1,filename2),fontsize=12)
  plt.show()

################################################################################
# Main
################################################################################
# Configure arguments
parser = argparse.ArgumentParser()
parser.add_argument('-i1','--input1',required=True,help='Input file1 (*.debug)')
parser.add_argument('-i2','--input2',required=False,help='Input file2 (*.debug)')
parser.add_argument('--workload',action='store_true',default=True,help='Display workload summary (default)')
parser.add_argument('--performance',action='store_true',default=False,help='Display performance summary')

# Parse arguments
args = parser.parse_args()
compare_files = args.input2 is not None

# Parse files
input1_debug = parse_input_debug(args.input1)
if compare_files:
  input2_debug = parse_input_debug(args.input2)
else:
  input2_debug = None
  
# Plot debug info
if args.performance:
  plot_debug_performance(args.input1,input1_debug,args.input2,input2_debug)
else:
  plot_debug_workload(args.input1,input1_debug,args.input2,input2_debug)
  



