/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/

/*
 * din irc bot
 * based on LGPLed code by Georgy Yunaev gyunaev@ulduzsoft.com
 * of libircclient.sourceforge.net.
 *
 * copyright (c) 2009-2011 S Jagannathan
 * distributed under gpl 3.0 affero license
 *
*/


#include <cstdio>

#include "bot.h"
#include "console.h"
#include "command.h"
#include "tokenizer.h"
#include "tcl_interp.h"
#include "utils.h"
#include "ansi_color_codes.h"

#include <iostream>
#include <cstdlib>
using namespace std;

extern console cons;
extern cmdlist cmdlst;
extern tcl_interp interpreter;

bot_t::bot_t (const string& ln, const string& sn) : command (ln, sn), sess (0), tid(0), prefixed (0), quitmsg ("i'm done.") {

    memset (&callbacks, 0, sizeof(callbacks));

    callbacks.event_connect = event_connect;
    callbacks.event_join = event_join;
    callbacks.event_nick = dump_event;
    callbacks.event_quit = dump_event;
    callbacks.event_part = dump_event;
    callbacks.event_mode = dump_event;
    callbacks.event_topic = dump_event;
    callbacks.event_kick = dump_event;
    callbacks.event_channel = event_channel;
    callbacks.event_privmsg = event_privmsg;
    callbacks.event_notice = dump_event;
    callbacks.event_invite = dump_event;
    callbacks.event_umode = dump_event;
    callbacks.event_ctcp_rep = dump_event;
    callbacks.event_ctcp_action = dump_event;
    callbacks.event_unknown = dump_event;
    callbacks.event_numeric = event_numeric;
    callbacks.event_dcc_chat_req = irc_event_dcc_chat;
    callbacks.event_dcc_send_req = irc_event_dcc_send;

    sess = irc_create_session (&callbacks);

    if (!sess) {
      cout << FAIL << "!!! could not create IRC session !!!" << ENDL;
      exit(1);
    } else {
      cout << PASS << "+++ created an IRC session +++" << ENDL;
      irc_set_ctx (sess, this);
    }

}

bool bot_t::operator() (tokenizer& tz) {

  string cmd;
  tz >> cmd;
  if ((cmd == "prefix") || (cmd == "p")) {
      string value; tz >> value;
      if (value != "") {
        prefix = value + ": ";
        prefixed = 1;
        cmdlst.result = "will prefix " + prefix;
      } else {
        cmdlst.result = "wont prefix " + prefix;
        prefix = "";
        prefixed = 0;
      }
  } else if ( (cmd == "connect") || (cmd == "c")) {
    tz >> srv >> nick >> chan >> passwd;
    // add # to channel name
    string s = "#"; s += chan; chan = s;

    nickolon = nick + ':';
    len = nickolon.length ();
    kill ();
    make ();
  } else if ( (cmd == "disconnect") || (cmd == "d")) {
    quitmsg = tz.cur2end ();
    if (quitmsg == "") quitmsg = "i'm done.";
    kill ();
    cons << console::red << "disconnected" << eol;
  } else { // send as message to channel
    cmd = cmd + ' ' + tz.cur2end ();
    *this << cmd;
  }
  return true;
}

bot_t& bot_t::operator<< (const std::string& s) {
  if (tid && (s != "")) {
    std::string out (prefix + s.c_str());
    irc_cmd_msg (sess, chan.c_str(), out.c_str());
  }
  return *this;
}

void* irc_monitor (void* pb) {
  bot_t* b = (bot_t*) pb;
  irc_run (b->sess);
  return 0;
}

int bot_t::make () {

  if (sess && !tid) { // no previous bot

    if ( irc_connect (sess, srv.c_str(), 6667, 0, nick.c_str(), 0, 0) ) {
      cmdlst.result = srv + " not found.";
      return 0;
    } else {
      cmdlst.result = "connecting to: " + srv;
      pthread_create (&tid, 0, irc_monitor, this);
      return 1;
    }

  }

  return 0;

}

int bot_t::kill () {
  if (tid) {
    irc_cmd_quit (sess, quitmsg.c_str());
    pthread_join (tid, 0);
    tid = 0;
    if (sess) irc_disconnect (sess);
    return 1;
  }
  return 0;
}

bot_t::~bot_t () {
  kill ();
  if (sess) irc_destroy_session (sess);
}

void dump_event (irc_session_t * session, const char * event, const char * origin, const char ** params, unsigned int count){}

void event_join (irc_session_t * session, const char * event, const char * origin, const char ** params, unsigned int count)
{
  bot_t* pbot = (bot_t *) irc_get_ctx (session);
  pbot->mesgs.push (get_nick (origin) + " has joined " + string(params[0]));
  irc_cmd_channel_mode (session, pbot->chan.c_str(), "");
}

void event_connect (irc_session_t * session, const char * event, const char * origin, const char ** params, unsigned int count) {
  bot_t* pbot = (bot_t *) irc_get_ctx (session);
  pbot->mesgs.push ("connected.");
  irc_cmd_join (session, pbot->chan.c_str(), pbot->passwd.c_str());
}

string get_nick (const char * origin) {

  static const int nickbufsize = 512;
  static char nickbuf [nickbufsize]; // irc_get_target_nick will truncate to this length
  irc_target_get_nick (origin, nickbuf, nickbufsize);
  return string (nickbuf);

}

void event_privmsg (irc_session_t * session, const char * event, const char * origin, const char ** params, unsigned int count) {
  bot_t* pb = (bot_t *) irc_get_ctx (session);
  pb->cmds.push (params[1]);
}

void event_channel (irc_session_t * session, const char * event, const char * origin, const char ** params, unsigned int count) {
  bot_t* pb = (bot_t *) irc_get_ctx (session);
  pb->mesg = params[1];
  string tok (pb->mesg.substr (0, pb->len));
  if (tok == pb->nickolon) {
    pb->mesg = pb->mesg.substr (pb->len, pb->mesg.length());
    pb->cmds.push (pb->mesg);
  } else {
    string s (get_nick (origin) + "@" + pb->srv + ": " + pb->mesg);
    pb->mesgs.push (s);
  }
}

void bot_t::handle_input () {

  if (cmds.size()) {
    string& cmd = cmds.front ();
    into_lines (interpreter (cmd).result, *this);
    cmds.pop ();
  }

  if (mesgs.size()) {
    string& mesg = mesgs.front ();
    cons << mesg << eol;
    mesgs.pop ();
  }

}

void event_numeric (irc_session_t * session, unsigned int event, const char * origin, const char ** params, unsigned int count) {
  if (event == 324) { // channel mode hardcode for building in jaunty
    string mode (params[2]);
    if (mode.find ("k") == string::npos) {
      bot_t* pb = (bot_t *) irc_get_ctx (session);
      pb->cmds.push ("bot disconnect {will connect to password protected channel only}");
    }
  }
}

void irc_event_dcc_chat (irc_session_t * session, const char * nick, const char * addr, irc_dcc_t dccid) {}
void irc_event_dcc_send (irc_session_t * session, const char * nick, const char * addr, const char * filename, unsigned long size, irc_dcc_t dccid) {}
