/*
 * $Id: misc_pci_vhdl_con.c,v 1.46 2012-02-22 09:27:20 siflkres Exp $
 *
 * Copyright (C) 2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* FIXME:
 * currently, the only clock pulses emitted to the pci pins are the ones
 * referring to bus transactions.
 * This of course cannot work if the hardware uses the pci clock as 
 * clock source.
 * Also, the emitted clock cycles are not correlated to simulation time.
 * For this either the global simulation time would need to be 
 * advanced even within callbacks - or the VHDL simulation time would need
 * to be advanced from within the callbacks. Both is ugly.
 *
 * One idea to solve the missing clock cycle problem is to simply 
 * emit a large number of clock pulses every now and then. A VHDL card could
 * then depend on the clock source. However this would still work only for
 * VHDL implementations which are strictly signal triggered, where no timing
 * interferes.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "misc_pci_vhdl_con.h"

#define CHIP_(x) misc_pci_vhdl_con_ ## x
#define CHIP "misc_pci_vhdl_con"

/* set to 1 to debug values of individual signals, produces
 * much output */
#define DEBUG_SIGNAL_VALUES 0

struct cpssp {
	struct sig_std_logic *port_idsel;
	struct sig_std_logic *port_clk;
	struct sig_std_logic *port_rst_n;
	struct sig_std_logic *port_cbe_n_0;
	unsigned int state_cbe_n_0;
	struct sig_std_logic *port_cbe_n_1;
	unsigned int state_cbe_n_1;
	struct sig_std_logic *port_cbe_n_2;
	unsigned int state_cbe_n_2;
	struct sig_std_logic *port_cbe_n_3;
	unsigned int state_cbe_n_3;
	struct sig_std_logic *port_ad_0;
	unsigned int state_ad_0;
	struct sig_std_logic *port_ad_1;
	unsigned int state_ad_1;
	struct sig_std_logic *port_ad_2;
	unsigned int state_ad_2;
	struct sig_std_logic *port_ad_3;
	unsigned int state_ad_3;
	struct sig_std_logic *port_ad_4;
	unsigned int state_ad_4;
	struct sig_std_logic *port_ad_5;
	unsigned int state_ad_5;
	struct sig_std_logic *port_ad_6;
	unsigned int state_ad_6;
	struct sig_std_logic *port_ad_7;
	unsigned int state_ad_7;
	struct sig_std_logic *port_ad_8;
	unsigned int state_ad_8;
	struct sig_std_logic *port_ad_9;
	unsigned int state_ad_9;
	struct sig_std_logic *port_ad_10;
	unsigned int state_ad_10;
	struct sig_std_logic *port_ad_11;
	unsigned int state_ad_11;
	struct sig_std_logic *port_ad_12;
	unsigned int state_ad_12;
	struct sig_std_logic *port_ad_13;
	unsigned int state_ad_13;
	struct sig_std_logic *port_ad_14;
	unsigned int state_ad_14;
	struct sig_std_logic *port_ad_15;
	unsigned int state_ad_15;
	struct sig_std_logic *port_ad_16;
	unsigned int state_ad_16;
	struct sig_std_logic *port_ad_17;
	unsigned int state_ad_17;
	struct sig_std_logic *port_ad_18;
	unsigned int state_ad_18;
	struct sig_std_logic *port_ad_19;
	unsigned int state_ad_19;
	struct sig_std_logic *port_ad_20;
	unsigned int state_ad_20;
	struct sig_std_logic *port_ad_21;
	unsigned int state_ad_21;
	struct sig_std_logic *port_ad_22;
	unsigned int state_ad_22;
	struct sig_std_logic *port_ad_23;
	unsigned int state_ad_23;
	struct sig_std_logic *port_ad_24;
	unsigned int state_ad_24;
	struct sig_std_logic *port_ad_25;
	unsigned int state_ad_25;
	struct sig_std_logic *port_ad_26;
	unsigned int state_ad_26;
	struct sig_std_logic *port_ad_27;
	unsigned int state_ad_27;
	struct sig_std_logic *port_ad_28;
	unsigned int state_ad_28;
	struct sig_std_logic *port_ad_29;
	unsigned int state_ad_29;
	struct sig_std_logic *port_ad_30;
	unsigned int state_ad_30;
	struct sig_std_logic *port_ad_31;
	unsigned int state_ad_31;
	struct sig_std_logic *port_par;
	unsigned int state_par;
	struct sig_std_logic *port_frame_n;
	unsigned int state_frame_n;
	struct sig_std_logic *port_trdy_n;
	unsigned int state_trdy_n;
	struct sig_std_logic *port_irdy_n;
	unsigned int state_irdy_n;
	struct sig_std_logic *port_stop_n;
	unsigned int state_stop_n;
	struct sig_std_logic *port_devsel_n;
	unsigned int state_devsel_n;
	struct sig_std_logic *port_perr_n;
	unsigned int state_perr_n;
	struct sig_std_logic *port_serr_n;
	unsigned int state_serr_n;
	struct sig_pci_bus *port_pci;
};

/** resolve a boolean value to STD_LOGIC 
  * @param bw if zero, resolve to 0, otherwise to 1
  * @return corresponding STD_LOGIC value
  */
static int
CHIP_(bool_2_std)(int bv)
{
	if (bv) {
		return SIG_STD_LOGIC_1;
	}

	return SIG_STD_LOGIC_0;
}

/** resolve a STD_LOGIC value to boolean 
  * @param sl standard logic value
  * @return 0/1.
  */
static int
CHIP_(std_2_bool)(int sl)
{
	switch (sl) {
	case SIG_STD_LOGIC_1:
	case SIG_STD_LOGIC_Z:
	case SIG_STD_LOGIC_H:
	case SIG_STD_LOGIC_U:
		/* pci bus logic to react is inverted. 
		 * map U to 1, denoting that the device doesn't react */
		return 1;
	
	default:
		return 0;
	}
}

/** write val to the A/D lanes. 
  * @param cpssp instance
  * @param val value to put onto the address/data lines.
  */
static void
CHIP_(bus_ad_put)(struct cpssp *cpssp, uint32_t val)
{
	sig_std_logic_set(cpssp->port_ad_0, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 0)));
	sig_std_logic_set(cpssp->port_ad_1, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 1)));
	sig_std_logic_set(cpssp->port_ad_2, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 2)));
	sig_std_logic_set(cpssp->port_ad_3, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 3)));
	sig_std_logic_set(cpssp->port_ad_4, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 4)));
	sig_std_logic_set(cpssp->port_ad_5, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 5)));
	sig_std_logic_set(cpssp->port_ad_6, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 6)));
	sig_std_logic_set(cpssp->port_ad_7, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 7)));
	sig_std_logic_set(cpssp->port_ad_8, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 8)));
	sig_std_logic_set(cpssp->port_ad_9, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 9)));
	sig_std_logic_set(cpssp->port_ad_10, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 10)));
	sig_std_logic_set(cpssp->port_ad_11, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 11)));
	sig_std_logic_set(cpssp->port_ad_12, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 12)));
	sig_std_logic_set(cpssp->port_ad_13, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 13)));
	sig_std_logic_set(cpssp->port_ad_14, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 14)));
	sig_std_logic_set(cpssp->port_ad_15, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 15)));
	sig_std_logic_set(cpssp->port_ad_16, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 16)));
	sig_std_logic_set(cpssp->port_ad_17, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 17)));
	sig_std_logic_set(cpssp->port_ad_18, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 18)));
	sig_std_logic_set(cpssp->port_ad_19, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 19)));
	sig_std_logic_set(cpssp->port_ad_20, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 20)));
	sig_std_logic_set(cpssp->port_ad_21, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 21)));
	sig_std_logic_set(cpssp->port_ad_22, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 22)));
	sig_std_logic_set(cpssp->port_ad_23, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 23)));
	sig_std_logic_set(cpssp->port_ad_24, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 24)));
	sig_std_logic_set(cpssp->port_ad_25, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 25)));
	sig_std_logic_set(cpssp->port_ad_26, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 26)));
	sig_std_logic_set(cpssp->port_ad_27, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 27)));
	sig_std_logic_set(cpssp->port_ad_28, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 28)));
	sig_std_logic_set(cpssp->port_ad_29, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 29)));
	sig_std_logic_set(cpssp->port_ad_30, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 30)));
	sig_std_logic_set(cpssp->port_ad_31, 
			cpssp, 
			CHIP_(bool_2_std)(val & (1 << 31)));
}

/** delay bus transaction for one clock cycle until
 *  the next rising edge.
 *  @param cpssp state instance pointer.
 */
static void
CHIP_(delay_clock)(struct cpssp *cpssp)
{
	/* FIXME trick: do not actually advance simulation time here,
	 *       just toggle the clock signal.
	 */
	sig_std_logic_set(cpssp->port_clk, cpssp, SIG_STD_LOGIC_1);
	/* delay */
	sched_to_scheduler();
	sig_std_logic_set(cpssp->port_clk, cpssp, SIG_STD_LOGIC_0);
	/* delay */
	sched_to_scheduler();
}

static int
CHIP_(bus_busy)(struct cpssp *cpssp)
{
	/* bus is free, if both frame and irdy are deasserted 
	 * bus busy = !(#frame & #irdy) */

	switch (cpssp->state_frame_n) {
	case SIG_STD_LOGIC_0:
	case SIG_STD_LOGIC_L:
	case SIG_STD_LOGIC_U:
		/* bus busy */
		return -1;

	default:
		break;
	}

	switch (cpssp->state_irdy_n) {
	case SIG_STD_LOGIC_0:
	case SIG_STD_LOGIC_L:
	case SIG_STD_LOGIC_U:
		/* bus busy */
		return -1;

	default:
		break;
	}

	return 0;
}

/** resolve pci_type of sig_pci_bus to c/be value.
 *  @param pci_type abstracted bus cycle type from sig_pci_bus.h
 *  @return resolved values for c/be lanes.
 *
 *  cf. pci spec 3.1.1.
 */
static unsigned int
CHIP_(type_to_command)(unsigned int pci_type)
{
	switch (pci_type) {
	case SIG_PCI_BUS_C0R:
		return 10;

	case SIG_PCI_BUS_C0W:
		return 11;

	case SIG_PCI_BUS_C1R:
		/* FIXME */
		return 10;

	case SIG_PCI_BUS_C1W:
		/* FIXME */
		return 11;

	case SIG_PCI_BUS_IOR:
		return 2;

	case SIG_PCI_BUS_IOW:
		return 3;

	case SIG_PCI_BUS_MR:
		return 6;

	case SIG_PCI_BUS_MW:
		return 7;

	case SIG_PCI_BUS_INTA:
		return 0;

	default:
		faum_log(FAUM_LOG_FATAL, CHIP, __func__,
			"pci_type out of range (%x)\n", pci_type);
		assert(0);
	}

	/* not reached */
	assert(0);
	return 0;
}

/** set the cbe_n signals with cbe_n bit field.
 *  @param cpssp state pointer.
 *  @param cbe_n bit field for cbe_n signals (no inversion will
 *         be done, bits must be correct low/high already).
 */
static void
CHIP_(set_cbe_n)(struct cpssp *cpssp, int cbe_n)
{
	sig_std_logic_set(cpssp->port_cbe_n_0, 
			cpssp, 
			CHIP_(bool_2_std)(cbe_n & (1 << 0)));
	sig_std_logic_set(cpssp->port_cbe_n_1, 
			cpssp, 
			CHIP_(bool_2_std)(cbe_n & (1 << 1)));
	sig_std_logic_set(cpssp->port_cbe_n_2, 
			cpssp, 
			CHIP_(bool_2_std)(cbe_n & (1 << 2)));
	sig_std_logic_set(cpssp->port_cbe_n_3, 
			cpssp, 
			CHIP_(bool_2_std)(cbe_n & (1 << 3)));
}

/** wait until devsel is asserted.
 *  @param cpssp state pointer.
 *  @return 0 if the device reacted in time, -1 if not.
 */
static int
CHIP_(wait_for_devsel)(struct cpssp *cpssp)
{
	unsigned int ncycles;

	/* cycles: 0: fast decode
	 *         1: medium decode
	 *         2: slow decode
	 *         3: subtractive decode
	 *         4: no agent reacted.
	 */
	for (ncycles = 0; ncycles < 3; ncycles++) {
		if (! CHIP_(std_2_bool)(cpssp->state_devsel_n)) {
			return 0;
		}
		CHIP_(delay_clock)(cpssp);
	}
	/* start of 4th cycle. */
	if (! CHIP_(std_2_bool)(cpssp->state_devsel_n)) {
		return 0;
	}

	return -1;
}

static void
CHIP_(release_ad)(struct cpssp *cpssp)
{
	sig_std_logic_set(cpssp->port_ad_0, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_1, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_2, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_3, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_4, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_5, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_6, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_7, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_8, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_9, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_10, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_11, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_12, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_13, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_14, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_15, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_16, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_17, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_18, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_19, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_20, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_21, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_22, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_23, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_24, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_25, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_26, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_27, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_28, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_29, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_30, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_ad_31, cpssp, SIG_STD_LOGIC_Z);
}

static int
CHIP_(type_addr_call)(void *_cpssp, unsigned int type, uint32_t addr)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int command;

	/* check if bus busy */
	if (CHIP_(bus_busy)(cpssp)) {
		faum_log(FAUM_LOG_WARNING, CHIP, __func__,
			"Bus busy, terminating.\n");
		return -1;
	}

	/* side note: byte enable is inverted, command is not */
	command = CHIP_(type_to_command)(type);

	switch (type) {
	case SIG_PCI_BUS_C0R:
	case SIG_PCI_BUS_C0W:
	case SIG_PCI_BUS_C1R:
	case SIG_PCI_BUS_C1W:

		sig_std_logic_set(cpssp->port_idsel, cpssp, SIG_STD_LOGIC_1);
		break;
	
	default:
		break;
		
	}

	/* pull down frame for entire command */
	sig_std_logic_set(cpssp->port_frame_n, cpssp, SIG_STD_LOGIC_0);
	/* set command signals */
	CHIP_(set_cbe_n)(cpssp, command);
	/* set address */
	CHIP_(bus_ad_put)(cpssp, addr);

	/* complete address phase */
	CHIP_(delay_clock)(cpssp);

	switch (type) {
	case SIG_PCI_BUS_C0R:
	case SIG_PCI_BUS_C0W:
	case SIG_PCI_BUS_C1R:
	case SIG_PCI_BUS_C1W:

		sig_std_logic_set(cpssp->port_idsel, cpssp, SIG_STD_LOGIC_0);
		break;

	default:

		break;
	}

	return -1;
}

static int
CHIP_(read_data_call)(void *_cpssp, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int ret = 0;

	/* first pull down initiator ready */
	sig_std_logic_set(cpssp->port_irdy_n, cpssp, SIG_STD_LOGIC_0);
	CHIP_(release_ad)(cpssp);

	/* set byte enable lines */
	CHIP_(set_cbe_n)(cpssp, ~bs);

	/* deassert frame, since only 1 word should get read */
	sig_std_logic_set(cpssp->port_frame_n, cpssp, SIG_STD_LOGIC_1);

	/* waiting for devsel / trdy will ensure the turnaround cycle */
	ret = CHIP_(wait_for_devsel)(cpssp);
	if (ret) goto CHIP_(read_data_call_out);

	/* delay until target is ready */
	while (CHIP_(std_2_bool)(cpssp->state_trdy_n)) {
		CHIP_(delay_clock)(cpssp);
	}

	/* pickup data */
	*valp = CHIP_(std_2_bool)(cpssp->state_ad_0) << 0
		| CHIP_(std_2_bool)(cpssp->state_ad_1) << 1
		| CHIP_(std_2_bool)(cpssp->state_ad_2) << 2
		| CHIP_(std_2_bool)(cpssp->state_ad_3) << 3
		| CHIP_(std_2_bool)(cpssp->state_ad_4) << 4
		| CHIP_(std_2_bool)(cpssp->state_ad_5) << 5
		| CHIP_(std_2_bool)(cpssp->state_ad_6) << 6
		| CHIP_(std_2_bool)(cpssp->state_ad_7) << 7
		| CHIP_(std_2_bool)(cpssp->state_ad_8) << 8
		| CHIP_(std_2_bool)(cpssp->state_ad_9) << 9
		| CHIP_(std_2_bool)(cpssp->state_ad_10) << 10
		| CHIP_(std_2_bool)(cpssp->state_ad_11) << 11
		| CHIP_(std_2_bool)(cpssp->state_ad_12) << 12
		| CHIP_(std_2_bool)(cpssp->state_ad_13) << 13
		| CHIP_(std_2_bool)(cpssp->state_ad_14) << 14
		| CHIP_(std_2_bool)(cpssp->state_ad_15) << 15
		| CHIP_(std_2_bool)(cpssp->state_ad_16) << 16
		| CHIP_(std_2_bool)(cpssp->state_ad_17) << 17
		| CHIP_(std_2_bool)(cpssp->state_ad_18) << 18
		| CHIP_(std_2_bool)(cpssp->state_ad_19) << 19
		| CHIP_(std_2_bool)(cpssp->state_ad_20) << 20
		| CHIP_(std_2_bool)(cpssp->state_ad_21) << 21
		| CHIP_(std_2_bool)(cpssp->state_ad_22) << 22
		| CHIP_(std_2_bool)(cpssp->state_ad_23) << 23
		| CHIP_(std_2_bool)(cpssp->state_ad_24) << 24
		| CHIP_(std_2_bool)(cpssp->state_ad_25) << 25
		| CHIP_(std_2_bool)(cpssp->state_ad_26) << 26
		| CHIP_(std_2_bool)(cpssp->state_ad_27) << 27
		| CHIP_(std_2_bool)(cpssp->state_ad_28) << 28
		| CHIP_(std_2_bool)(cpssp->state_ad_29) << 29
		| CHIP_(std_2_bool)(cpssp->state_ad_30) << 30
		| CHIP_(std_2_bool)(cpssp->state_ad_31) << 31;

CHIP_(read_data_call_out):
	/* release bus */
	sig_std_logic_set(cpssp->port_cbe_n_0, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_cbe_n_1, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_cbe_n_2, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_cbe_n_3, cpssp, SIG_STD_LOGIC_Z);

	sig_std_logic_set(cpssp->port_irdy_n, cpssp, SIG_STD_LOGIC_Z);

	/* complete cycle */
	CHIP_(delay_clock)(cpssp);
	CHIP_(delay_clock)(cpssp);

	return ret;
}

static int
CHIP_(write_data_call)(void *_cpssp, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int ret;

	/* assert initiator ready */
	sig_std_logic_set(cpssp->port_irdy_n, cpssp, SIG_STD_LOGIC_0);

	/* set byte enable lanes */
	CHIP_(set_cbe_n)(cpssp, ~bs);
	/* put data on bus */
	CHIP_(bus_ad_put)(cpssp, val);
	/* release frame_n to indicate that this is the last word written */
	sig_std_logic_set(cpssp->port_frame_n, cpssp, SIG_STD_LOGIC_1);

	ret = CHIP_(wait_for_devsel)(cpssp);
	if (ret) goto CHIP_(write_data_call_out);

	while (CHIP_(std_2_bool)(cpssp->state_trdy_n)) {
		CHIP_(delay_clock)(cpssp);
	}

CHIP_(write_data_call_out):
	/* release bus */
	sig_std_logic_set(cpssp->port_cbe_n_0, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_cbe_n_1, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_cbe_n_2, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_set(cpssp->port_cbe_n_3, cpssp, SIG_STD_LOGIC_Z);

	CHIP_(release_ad)(cpssp);

	sig_std_logic_set(cpssp->port_irdy_n, cpssp, SIG_STD_LOGIC_Z);
	/* FIXME parity */

	/* complete cycle */
	CHIP_(delay_clock)(cpssp);
	
	return ret;
}

static int
CHIP_(pci_c0r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int ret;

	ret = CHIP_(type_addr_call)(cpssp, SIG_PCI_BUS_C0R, addr);
	ret = CHIP_(read_data_call)(cpssp, bs, valp);

	return ret;
}

static int
CHIP_(pci_c0w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int ret;
	
	ret = CHIP_(type_addr_call)(cpssp, SIG_PCI_BUS_C0W, addr);
	ret = CHIP_(write_data_call)(cpssp, bs, val);

	return ret;
}

static int
CHIP_(pci_ior)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int ret;
	
	ret = CHIP_(type_addr_call)(cpssp, SIG_PCI_BUS_IOR, addr);
	ret = CHIP_(read_data_call)(cpssp, bs, valp);

	return ret;
}

static int
CHIP_(pci_iow)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;
	int ret;
	
	ret = CHIP_(type_addr_call)(cpssp, SIG_PCI_BUS_IOW, addr);
	ret = CHIP_(write_data_call)(cpssp, bs, val);

	return ret;
}

static void
CHIP_(pci_reset_n_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;

	sig_std_logic_set(cpssp->port_rst_n, cpssp, CHIP_(bool_2_std)(val));
}

#if DEBUG_SIGNAL_VALUES > 0
static char
CHIP_(std_to_char)(unsigned int val)
{
	static char std_logic_vals[] = "UX01ZWLH-";

	assert(val < sizeof(std_logic_vals));
	return std_logic_vals[val];
}

#define DEBUG_PRINT_SIGNAL(val) \
	faum_log(FAUM_LOG_DEBUG, CHIP, __func__, \
		"val=%c\n", CHIP_(std_to_char(val)));
#else /* DEBUG_PRINT_SIGNAL == 0 */

#define DEBUG_PRINT_SIGNAL(val) 
#endif


static void
CHIP_(cbe_n_0_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp*)_cpssp;
	cpssp->state_cbe_n_0 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(cbe_n_1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp*)_cpssp;
	cpssp->state_cbe_n_1 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(cbe_n_2_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp*)_cpssp;
	cpssp->state_cbe_n_2 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(cbe_n_3_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp*)_cpssp;
	cpssp->state_cbe_n_3 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_0_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_0 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_1_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_1 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_2_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_2 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_3_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_3 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_4_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_4 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_5_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_5 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_6_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_6 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_7_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_7 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_8_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_8 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_9_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_9 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_10_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_10 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_11_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_11 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_12_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_12 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_13_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_13 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_14_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_14 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_15_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_15 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_16_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_16 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_17_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_17 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_18_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_18 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_19_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_19 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_20_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_20 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_21_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_21 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_22_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_22 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_23_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_23 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_24_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_24 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_25_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_25 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_26_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_26 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_27_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_27 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_28_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_28 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_29_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_29 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_30_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_30 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(ad_31_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_ad_31 = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(par_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_par = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(frame_n_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_frame_n = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(trdy_n_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_trdy_n = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(irdy_n_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_irdy_n = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(stop_n_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_stop_n = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(devsel_n_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_devsel_n = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(perr_n_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_perr_n = val;
	DEBUG_PRINT_SIGNAL(val);
}

static void
CHIP_(serr_n_set)(void *_cppsp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *)_cppsp;
	cpssp->state_serr_n = val;
	DEBUG_PRINT_SIGNAL(val);
}

#undef DEBUG_PRINT_SIGNAL

static void
CHIP_(reset_1)(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;

	sig_std_logic_set(cpssp->port_rst_n, cpssp, SIG_STD_LOGIC_1);
}


static void
CHIP_(reset_0)(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *)_cpssp;

	sig_std_logic_set(cpssp->port_rst_n, cpssp, SIG_STD_LOGIC_0);
	time_call_after(TIME_HZ / 100000000, CHIP_(reset_1), cpssp);
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_pci_bus *port_pci,
	struct sig_std_logic *port_cbe_n_0,
	struct sig_std_logic *port_cbe_n_1,
	struct sig_std_logic *port_cbe_n_2,
	struct sig_std_logic *port_cbe_n_3,
	struct sig_std_logic *port_ad_0,
	struct sig_std_logic *port_ad_1,
	struct sig_std_logic *port_ad_2,
	struct sig_std_logic *port_ad_3,
	struct sig_std_logic *port_ad_4,
	struct sig_std_logic *port_ad_5,
	struct sig_std_logic *port_ad_6,
	struct sig_std_logic *port_ad_7,
	struct sig_std_logic *port_ad_8,
	struct sig_std_logic *port_ad_9,
	struct sig_std_logic *port_ad_10,
	struct sig_std_logic *port_ad_11,
	struct sig_std_logic *port_ad_12,
	struct sig_std_logic *port_ad_13,
	struct sig_std_logic *port_ad_14,
	struct sig_std_logic *port_ad_15,
	struct sig_std_logic *port_ad_16,
	struct sig_std_logic *port_ad_17,
	struct sig_std_logic *port_ad_18,
	struct sig_std_logic *port_ad_19,
	struct sig_std_logic *port_ad_20,
	struct sig_std_logic *port_ad_21,
	struct sig_std_logic *port_ad_22,
	struct sig_std_logic *port_ad_23,
	struct sig_std_logic *port_ad_24,
	struct sig_std_logic *port_ad_25,
	struct sig_std_logic *port_ad_26,
	struct sig_std_logic *port_ad_27,
	struct sig_std_logic *port_ad_28,
	struct sig_std_logic *port_ad_29,
	struct sig_std_logic *port_ad_30,
	struct sig_std_logic *port_ad_31,
	struct sig_std_logic *port_par,
	struct sig_std_logic *port_frame_n,
	struct sig_std_logic *port_trdy_n,
	struct sig_std_logic *port_irdy_n,
	struct sig_std_logic *port_stop_n,
	struct sig_std_logic *port_devsel_n,
	struct sig_std_logic *port_idsel,
	struct sig_std_logic *port_perr_n,
	struct sig_std_logic *port_serr_n,
	struct sig_std_logic *port_clk,
	struct sig_std_logic *port_rst_n
)
{
	static const struct sig_pci_bus_main_funcs pci_main_funcs = {
		.ior = CHIP_(pci_ior),
		.iow = CHIP_(pci_iow)
	};
	static const struct sig_pci_bus_idsel_funcs pci_idsel_funcs = {
		.c0r = CHIP_(pci_c0r),
		.c0w = CHIP_(pci_c0w)
	};
	static const struct sig_boolean_funcs pci_n_reset_funcs = {
		.set = CHIP_(pci_reset_n_set)
	};
	static const struct sig_std_logic_funcs cbe_n_0_funcs = {
		.set = CHIP_(cbe_n_0_set)
	};
	static const struct sig_std_logic_funcs cbe_n_1_funcs = {
		.set = CHIP_(cbe_n_1_set)
	};
	static const struct sig_std_logic_funcs cbe_n_2_funcs = {
		.set = CHIP_(cbe_n_2_set)
	};
	static const struct sig_std_logic_funcs cbe_n_3_funcs = {
		.set = CHIP_(cbe_n_3_set)
	};
	static const struct sig_std_logic_funcs ad_0_funcs = {
		.set = CHIP_(ad_0_set)
	};
	static const struct sig_std_logic_funcs ad_1_funcs = {
		.set = CHIP_(ad_1_set)
	};
	static const struct sig_std_logic_funcs ad_2_funcs = {
		.set = CHIP_(ad_2_set)
	};
	static const struct sig_std_logic_funcs ad_3_funcs = {
		.set = CHIP_(ad_3_set)
	};
	static const struct sig_std_logic_funcs ad_4_funcs = {
		.set = CHIP_(ad_4_set)
	};
	static const struct sig_std_logic_funcs ad_5_funcs = {
		.set = CHIP_(ad_5_set)
	};
	static const struct sig_std_logic_funcs ad_6_funcs = {
		.set = CHIP_(ad_6_set)
	};
	static const struct sig_std_logic_funcs ad_7_funcs = {
		.set = CHIP_(ad_7_set)
	};
	static const struct sig_std_logic_funcs ad_8_funcs = {
		.set = CHIP_(ad_8_set)
	};
	static const struct sig_std_logic_funcs ad_9_funcs = {
		.set = CHIP_(ad_9_set)
	};
	static const struct sig_std_logic_funcs ad_10_funcs = {
		.set = CHIP_(ad_10_set)
	};
	static const struct sig_std_logic_funcs ad_11_funcs = {
		.set = CHIP_(ad_11_set)
	};
	static const struct sig_std_logic_funcs ad_12_funcs = {
		.set = CHIP_(ad_12_set)
	};
	static const struct sig_std_logic_funcs ad_13_funcs = {
		.set = CHIP_(ad_13_set)
	};
	static const struct sig_std_logic_funcs ad_14_funcs = {
		.set = CHIP_(ad_14_set)
	};
	static const struct sig_std_logic_funcs ad_15_funcs = {
		.set = CHIP_(ad_15_set)
	};
	static const struct sig_std_logic_funcs ad_16_funcs = {
		.set = CHIP_(ad_16_set)
	};
	static const struct sig_std_logic_funcs ad_17_funcs = {
		.set = CHIP_(ad_17_set)
	};
	static const struct sig_std_logic_funcs ad_18_funcs = {
		.set = CHIP_(ad_18_set)
	};
	static const struct sig_std_logic_funcs ad_19_funcs = {
		.set = CHIP_(ad_19_set)
	};
	static const struct sig_std_logic_funcs ad_20_funcs = {
		.set = CHIP_(ad_20_set)
	};
	static const struct sig_std_logic_funcs ad_21_funcs = {
		.set = CHIP_(ad_21_set)
	};
	static const struct sig_std_logic_funcs ad_22_funcs = {
		.set = CHIP_(ad_22_set)
	};
	static const struct sig_std_logic_funcs ad_23_funcs = {
		.set = CHIP_(ad_23_set)
	};
	static const struct sig_std_logic_funcs ad_24_funcs = {
		.set = CHIP_(ad_24_set)
	};
	static const struct sig_std_logic_funcs ad_25_funcs = {
		.set = CHIP_(ad_25_set)
	};
	static const struct sig_std_logic_funcs ad_26_funcs = {
		.set = CHIP_(ad_26_set)
	};
	static const struct sig_std_logic_funcs ad_27_funcs = {
		.set = CHIP_(ad_27_set)
	};
	static const struct sig_std_logic_funcs ad_28_funcs = {
		.set = CHIP_(ad_28_set)
	};
	static const struct sig_std_logic_funcs ad_29_funcs = {
		.set = CHIP_(ad_29_set)
	};
	static const struct sig_std_logic_funcs ad_30_funcs = {
		.set = CHIP_(ad_30_set)
	};
	static const struct sig_std_logic_funcs ad_31_funcs = {
		.set = CHIP_(ad_31_set)
	};
	static const struct sig_std_logic_funcs par_funcs = {
		.set = CHIP_(par_set)
	};
	static const struct sig_std_logic_funcs frame_n_funcs = {
		.set = CHIP_(frame_n_set)
	};
	static const struct sig_std_logic_funcs trdy_n_funcs = {
		.set = CHIP_(trdy_n_set)
	};
	static const struct sig_std_logic_funcs irdy_n_funcs = {
		.set = CHIP_(irdy_n_set)
	};
	static const struct sig_std_logic_funcs stop_n_funcs = {
		.set = CHIP_(stop_n_set)
	};
	static const struct sig_std_logic_funcs devsel_n_funcs = {
		.set = CHIP_(devsel_n_set)
	};
	static const struct sig_std_logic_funcs perr_n_funcs = {
		.set = CHIP_(perr_n_set)
	};
	static const struct sig_std_logic_funcs serr_n_funcs = {
		.set = CHIP_(serr_n_set)
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Call */
	cpssp->port_pci = port_pci;
	sig_pci_bus_main_connect(port_pci->main, cpssp, &pci_main_funcs);
	sig_pci_bus_idsel_connect(port_pci->idsel, cpssp, &pci_idsel_funcs);
	sig_boolean_connect_in(port_pci->n_reset, cpssp, &pci_n_reset_funcs);

	/* Out */
	cpssp->port_cbe_n_0 = port_cbe_n_0;
	sig_std_logic_connect_out(port_cbe_n_0, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_cbe_n_1 = port_cbe_n_1;
	sig_std_logic_connect_out(port_cbe_n_1, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_cbe_n_2 = port_cbe_n_2;
	sig_std_logic_connect_out(port_cbe_n_2, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_cbe_n_3 = port_cbe_n_3;
	sig_std_logic_connect_out(port_cbe_n_3, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_0 = port_ad_0;
	sig_std_logic_connect_out(port_ad_0, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_1 = port_ad_1;
	sig_std_logic_connect_out(port_ad_1, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_2 = port_ad_2;
	sig_std_logic_connect_out(port_ad_2, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_3 = port_ad_3;
	sig_std_logic_connect_out(port_ad_3, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_4 = port_ad_4;
	sig_std_logic_connect_out(port_ad_4, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_5 = port_ad_5;
	sig_std_logic_connect_out(port_ad_5, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_6 = port_ad_6;
	sig_std_logic_connect_out(port_ad_6, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_7 = port_ad_7;
	sig_std_logic_connect_out(port_ad_7, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_8 = port_ad_8;
	sig_std_logic_connect_out(port_ad_8, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_9 = port_ad_9;
	sig_std_logic_connect_out(port_ad_9, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_10 = port_ad_10;
	sig_std_logic_connect_out(port_ad_10, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_11 = port_ad_11;
	sig_std_logic_connect_out(port_ad_11, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_12 = port_ad_12;
	sig_std_logic_connect_out(port_ad_12, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_13 = port_ad_13;
	sig_std_logic_connect_out(port_ad_13, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_14 = port_ad_14;
	sig_std_logic_connect_out(port_ad_14, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_15 = port_ad_15;
	sig_std_logic_connect_out(port_ad_15, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_16 = port_ad_16;
	sig_std_logic_connect_out(port_ad_16, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_17 = port_ad_17;
	sig_std_logic_connect_out(port_ad_17, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_18 = port_ad_18;
	sig_std_logic_connect_out(port_ad_18, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_19 = port_ad_19;
	sig_std_logic_connect_out(port_ad_19, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_20 = port_ad_20;
	sig_std_logic_connect_out(port_ad_20, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_21 = port_ad_21;
	sig_std_logic_connect_out(port_ad_21, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_22 = port_ad_22;
	sig_std_logic_connect_out(port_ad_22, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_23 = port_ad_23;
	sig_std_logic_connect_out(port_ad_23, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_24 = port_ad_24;
	sig_std_logic_connect_out(port_ad_24, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_25 = port_ad_25;
	sig_std_logic_connect_out(port_ad_25, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_26 = port_ad_26;
	sig_std_logic_connect_out(port_ad_26, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_27 = port_ad_27;
	sig_std_logic_connect_out(port_ad_27, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_28 = port_ad_28;
	sig_std_logic_connect_out(port_ad_28, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_29 = port_ad_29;
	sig_std_logic_connect_out(port_ad_29, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_30 = port_ad_30;
	sig_std_logic_connect_out(port_ad_30, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_ad_31 = port_ad_31;
	sig_std_logic_connect_out(port_ad_31, cpssp, SIG_STD_LOGIC_Z);

	/* FIXME parity currently not generated correctly */
	cpssp->port_par = port_par;
	sig_std_logic_connect_out(port_par, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_frame_n = port_frame_n;
	sig_std_logic_connect_out(port_frame_n, cpssp, SIG_STD_LOGIC_1);

	cpssp->port_trdy_n = port_trdy_n;
	sig_std_logic_connect_out(port_trdy_n, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_irdy_n = port_irdy_n;
	sig_std_logic_connect_out(port_irdy_n, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_stop_n = port_stop_n;
	sig_std_logic_connect_out(port_stop_n, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_idsel = port_idsel;
	sig_std_logic_connect_out(port_idsel, cpssp, SIG_STD_LOGIC_0);

	cpssp->port_devsel_n = port_devsel_n;
	sig_std_logic_connect_out(port_devsel_n, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_perr_n = port_perr_n;
	sig_std_logic_connect_out(port_perr_n, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_serr_n = port_serr_n;
	sig_std_logic_connect_out(port_serr_n, cpssp, SIG_STD_LOGIC_Z);

	cpssp->port_clk = port_clk;
	sig_std_logic_connect_out(port_clk, cpssp, SIG_STD_LOGIC_0);

	cpssp->port_rst_n = port_rst_n;
	sig_std_logic_connect_out(port_rst_n, cpssp, SIG_STD_LOGIC_Z);

	/* In */
	cpssp->state_cbe_n_0 = 0;
	sig_std_logic_connect_in(port_cbe_n_0, cpssp, &cbe_n_0_funcs);

	cpssp->state_cbe_n_1 = 0;
	sig_std_logic_connect_in(port_cbe_n_1, cpssp, &cbe_n_1_funcs);

	cpssp->state_cbe_n_2 = 0;
	sig_std_logic_connect_in(port_cbe_n_2, cpssp, &cbe_n_2_funcs);

	cpssp->state_cbe_n_3 = 0;
	sig_std_logic_connect_in(port_cbe_n_3, cpssp, &cbe_n_3_funcs);

	cpssp->state_ad_0 = 0;
	sig_std_logic_connect_in(port_ad_0, cpssp, &ad_0_funcs);

	cpssp->state_ad_1 = 0;
	sig_std_logic_connect_in(port_ad_1, cpssp, &ad_1_funcs);

	cpssp->state_ad_2 = 0;
	sig_std_logic_connect_in(port_ad_2, cpssp, &ad_2_funcs);

	cpssp->state_ad_3 = 0;
	sig_std_logic_connect_in(port_ad_3, cpssp, &ad_3_funcs);

	cpssp->state_ad_4 = 0;
	sig_std_logic_connect_in(port_ad_4, cpssp, &ad_4_funcs);

	cpssp->state_ad_5 = 0;
	sig_std_logic_connect_in(port_ad_5, cpssp, &ad_5_funcs);

	cpssp->state_ad_6 = 0;
	sig_std_logic_connect_in(port_ad_6, cpssp, &ad_6_funcs);

	cpssp->state_ad_7 = 0;
	sig_std_logic_connect_in(port_ad_7, cpssp, &ad_7_funcs);

	cpssp->state_ad_8 = 0;
	sig_std_logic_connect_in(port_ad_8, cpssp, &ad_8_funcs);

	cpssp->state_ad_9 = 0;
	sig_std_logic_connect_in(port_ad_9, cpssp, &ad_9_funcs);

	cpssp->state_ad_10 = 0;
	sig_std_logic_connect_in(port_ad_10, cpssp, &ad_10_funcs);

	cpssp->state_ad_11 = 0;
	sig_std_logic_connect_in(port_ad_11, cpssp, &ad_11_funcs);

	cpssp->state_ad_12 = 0;
	sig_std_logic_connect_in(port_ad_12, cpssp, &ad_12_funcs);

	cpssp->state_ad_13 = 0;
	sig_std_logic_connect_in(port_ad_13, cpssp, &ad_13_funcs);

	cpssp->state_ad_14 = 0;
	sig_std_logic_connect_in(port_ad_14, cpssp, &ad_14_funcs);

	cpssp->state_ad_15 = 0;
	sig_std_logic_connect_in(port_ad_15, cpssp, &ad_15_funcs);

	cpssp->state_ad_16 = 0;
	sig_std_logic_connect_in(port_ad_16, cpssp, &ad_16_funcs);

	cpssp->state_ad_17 = 0;
	sig_std_logic_connect_in(port_ad_17, cpssp, &ad_17_funcs);

	cpssp->state_ad_18 = 0;
	sig_std_logic_connect_in(port_ad_18, cpssp, &ad_18_funcs);

	cpssp->state_ad_19 = 0;
	sig_std_logic_connect_in(port_ad_19, cpssp, &ad_19_funcs);

	cpssp->state_ad_20 = 0;
	sig_std_logic_connect_in(port_ad_20, cpssp, &ad_20_funcs);

	cpssp->state_ad_21 = 0;
	sig_std_logic_connect_in(port_ad_21, cpssp, &ad_21_funcs);

	cpssp->state_ad_22 = 0;
	sig_std_logic_connect_in(port_ad_22, cpssp, &ad_22_funcs);

	cpssp->state_ad_23 = 0;
	sig_std_logic_connect_in(port_ad_23, cpssp, &ad_23_funcs);

	cpssp->state_ad_24 = 0;
	sig_std_logic_connect_in(port_ad_24, cpssp, &ad_24_funcs);

	cpssp->state_ad_25 = 0;
	sig_std_logic_connect_in(port_ad_25, cpssp, &ad_25_funcs);

	cpssp->state_ad_26 = 0;
	sig_std_logic_connect_in(port_ad_26, cpssp, &ad_26_funcs);

	cpssp->state_ad_27 = 0;
	sig_std_logic_connect_in(port_ad_27, cpssp, &ad_27_funcs);

	cpssp->state_ad_28 = 0;
	sig_std_logic_connect_in(port_ad_28, cpssp, &ad_28_funcs);

	cpssp->state_ad_29 = 0;
	sig_std_logic_connect_in(port_ad_29, cpssp, &ad_29_funcs);

	cpssp->state_ad_30 = 0;
	sig_std_logic_connect_in(port_ad_30, cpssp, &ad_30_funcs);

	cpssp->state_ad_31 = 0;
	sig_std_logic_connect_in(port_ad_31, cpssp, &ad_31_funcs);

	cpssp->state_par = 0;
	sig_std_logic_connect_in(port_par, cpssp, &par_funcs);

	cpssp->state_frame_n = 0;
	sig_std_logic_connect_in(port_frame_n, cpssp, &frame_n_funcs);

	cpssp->state_trdy_n = 0;
	sig_std_logic_connect_in(port_trdy_n, cpssp, &trdy_n_funcs);

	cpssp->state_irdy_n = 0;
	sig_std_logic_connect_in(port_irdy_n, cpssp, &irdy_n_funcs);

	cpssp->state_stop_n = 0;
	sig_std_logic_connect_in(port_stop_n, cpssp, &stop_n_funcs);

	cpssp->state_devsel_n = 0;
	sig_std_logic_connect_in(port_devsel_n, cpssp, &devsel_n_funcs);

	cpssp->state_perr_n = 0;
	sig_std_logic_connect_in(port_perr_n, cpssp, &perr_n_funcs);

	cpssp->state_serr_n = 0;
	sig_std_logic_connect_in(port_serr_n, cpssp, &serr_n_funcs);

	time_call_after(0, CHIP_(reset_0), cpssp);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
