// ----------------------------------------------------------------------------
// Copyright (C) 2023
//              David Freese, W1HKJ
//
// This file is part of flrig.
//
// flrig is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
//
// flrig is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
// ----------------------------------------------------------------------------

// Kennwood TM-D710 and similar

#include "other/TMD710.h"

const char TMD710name_[] = "TMD710";

static std::vector<std::string>TMD710modes_;
static const char *vTMD710modes_[] = {
		"FM", "NFM", "AM"};

//gibts nicht
static const char TMD710_mode_type[] =
	{'U', 'U', 'U'};

RIG_TMD710::RIG_TMD710() {
// base class values	
	name_ = TMD710name_;
	modes_ = TMD710modes_;
	bandwidths_ = vNOBWS;
	serial_baudrate = BR38400;
	stopbits = 1;
	serial_retries = 3;

//	serial_write_delay = 0;
	serial_post_write_delay = 50;

	serial_timeout = 1000;
	serial_rtscts = true;
	serial_rtsplus = false;
	serial_dtrplus = false;
	serial_catptt = true;
	serial_rtsptt = false;
	serial_dtrptt = false;
	modeA = 1;
//	bwA = 2;

	has_mode_control =
	has_ptt_control = true;

//	has_attenuator_control =
//	has_preamp_control =
//	has_power_out =
	has_power_control =
	has_sql_control =
/*	has_volume_control =
	has_bandwidth_control =
	has_micgain_control =
	has_notch_control =
	has_ifshift_control =
	has_tune_control =*/
//	has_swr_control = false;
	has_vfoAB = true;
//	has_split =
//	has_split_AB =
	precision = 10;
	ndigits = 7;

}

void RIG_TMD710::initialize()
{
	VECTOR (TMD710modes_, vTMD710modes_);

	modes_ = TMD710modes_;
	bandwidths_ = vNOBWS;
}

bool RIG_TMD710::check ()
{
	cmd = "DL 0\r"; // Enable dual mode
	int ret = wait_char('\r', 5, 100, "check and set Dual Band Mode", ASC);
	if (ret < 1) return false;
	return true;
}

// Works for TM-D710E
unsigned long long RIG_TMD710::get_vfoA ()
{
	cmd = "FO 0\r";
	int ret = wait_char('\r', 49, 100, "get VFO", ASC);
	if (ret < 49) return freqA;

	getcr("get_vfoA");

//	char frequency[11];
//	int p = 5;
//	int n = 0;
//	for (n = 0; n < 10; n++) frequency[n] = (replystr[p + n]);
//	frequency[10] = '\0';
//	sscanf(frequency, "%lld", &freqA);

//	char mode[2];
//	mode[0] = (replystr[47]);
//	mode[1] = '\0';
// 	modeA = atoi(mode);

	replystr[15] = 0;
	sscanf(&replystr[5], "%lld", &freqA);
	modeA = replystr[47] - '0';

	return freqA;

	/*
1 	Band
					0 	Band A
					1 	Band B
					2 	TX A - RX B
					3 	TX B - RX A
2 	Frequency in Hz 10 digit. must be within selected band -> ok
3 	Step size 		0 	5 Khz
					1 	6.25 Khz
					2 	28,33 KHz
					3 	10 Khz
					4 	12.5 Khz
					5 	15 Khz
					6 	20 Khz
					7 	25 KHz
					8 	30 KHz
					9 	50 Khz
					A 	100 Khz
4 	Shift direction
					0 	simplex or split
					1 	up
					2 	down
5 	Reverse
					0 	off
					1 	on
6 	Tone status
					0 	off
					1 	on
7 	CTCSS status
					0 	off
					1 	on
8 	DCS status
					0 	off
					1 	on
9 	Tone frequency
					01 	67
					02 	69,3
					03 	71,9
					04 	74,4
					05 	77
					06 	79,7
					07 	82,5
					08 	85,4
					09 	88,5
					10 	91,5
					11 	94,8
					12 	97,4
					13 	100
					14 	103,5
					15 	107,2
					16 	110,9
					17 	114,8
					18 	118,8
					19 	123
					20 	127,3
					21 	131,8
					22 	136,5
					23 	141,3
					24 	146,2
					25 	151,4
					26 	156,7
					27 	162,2
					28 	167,9
					29 	173,8
					30 	179,9
					31 	186,2
					32 	192,8
					33 	203,5
					34 	240,7
					35 	210,7
					36 	218,1
					37 	225,7
					38 	229,1
					39 	233,6
					40 	241,8
					41 	250,3
					42 	254,1
10 	CTCSS frequency
					01 	67
					02 	69,3
					03 	71,9
					04 	74,4
					05 	77
					06 	79,7
					07 	82,5
					08 	85,4
					09 	88,5
					10 	91,5
					11 	94,8
					12 	97,4
					13 	100
					14 	103,5
					15 	107,2
					16 	110,9
					17 	114,8
					18 	118,8
					19 	123
					20 	127,3
					21 	131,8
					22 	136,5
					23 	141,3
					24 	146,2
					25 	151,4
					26 	156,7
					27 	162,2
					28 	167,9
					29 	173,8
					30 	179,9
					31 	186,2
					32 	192,8
					33 	203,5
					34 	240,7
					35 	210,7
					36 	218,1
					37 	225,7
					38 	229,1
					39 	233,6
					40 	241,8
					41 	250,3
					42 	254,1
11 	DCS frequency  -> seite down
12 	Offset frequency in Hz 8 digit
13 	Mode
					0 	FM
					1 	NFM
					2 	AM

	*/

}

unsigned long long RIG_TMD710::get_vfoB ()
{
	cmd = "FO 1\r";
	int ret = wait_char('\r', 49, 100, "get VFO", ASC);
	getcr("get_vfoB");

	if (ret < 49) return freqB;

//	char frequency[11];
//	int p = 5;
//	int n = 0;
//	for (n = 0; n < 10; n++) frequency[n] = (replystr[p + n]);
//	frequency[10] = '\0';
//	sscanf(frequency, "%lld", &freqB);

//	char mode[2];
//	mode[0] = (replystr[47]);
//	mode[1] = '\0';
// 	modeB = atoi(mode);

	replystr[15] = 0;
	sscanf(&replystr[5], "%lld", &freqB);
	modeB = replystr[47] - '0';

	return freqB;
}

void RIG_TMD710::set_vfoA (unsigned long long freq)
{
	cmd = "FO 0\r";
	int ret = wait_char('\r', 49, 200, "get VFO A", ASC);
	getcr("read info vfoA");
	if (ret < 49) return;
	cmd = replystr;
	char frequency[11];
	snprintf(frequency,11, "%010llu", freq);
	int n = 0;
	int p = 5;
	for (n = 0; n < 10; n++) (cmd[p + n])= frequency[n];
	wait_char('\r', 49, 200, "set frequency A", ASC);
	setcr("set vfoA");
	return;
}

void RIG_TMD710::set_vfoB (unsigned long long freq)
{
	cmd = "FO 1\r";
	int ret = wait_char('\r', 49, 200, "get VFO B", ASC);
	getcr("read info vfoB");
	if (ret < 49) return;
	cmd = replystr;
	char frequency[11];
	snprintf(frequency,11, "%010llu", freq);
	int n = 0;
	int p = 5;
	for (n = 0; n < 10; n++) (cmd[p + n])= frequency[n];
	wait_char('\r', 49, 200, "set frequency B", ASC);
	setcr("set vfoB");
	return;
}

int RIG_TMD710::get_modetype(int n)
{
	return TMD710_mode_type[n];
}

void RIG_TMD710::set_modeA(int val)
{
	// AM is not available at 430 MHz
	cmd = "FO 0\r";
	int ret = wait_char('\r', 49, 200, "get VFO A", ASC);
	getcr("read info vfoA");
	if (ret < 49) return;

	cmd = replystr;
	cmd[47] = val +'0';
	wait_char('\r', 49, 200, "set mode A", ASC);
	setcr("set modeA");
	return;
}

void RIG_TMD710::set_modeB(int val)
{
	cmd = "FO 1\r";
	int ret = wait_char('\r', 49, 200, "get VFO B", ASC);
	getcr("read info vfoB");
	if (ret < 49) return;

	cmd = replystr;
	cmd[47] = val +'0';
	wait_char('\r', 49, 200, "set mode B", ASC);
	setcr("set mode B");
	return;
}

int RIG_TMD710::get_modeA()
{
	//information is available with cyclic VFO readings
	return modeA;
}

int RIG_TMD710::get_modeB()
{
	//information is available with cyclic VFO readings
	return modeB;
}

void RIG_TMD710::selectA()
{
	cmd = "BC 0,0\r";
	wait_char('\r', 3, 100, "set select A PTT/CTRL band", ASC);
	setcr("selectA");
	inuse = onA;
}

void RIG_TMD710::selectB()
{
	cmd = "BC 1,1\r";
	wait_char('\r', 3, 100, "set select B PTT/CTRL band", ASC);
	setcr("selectB");
	inuse = onB;
}

// Tranceiver PTT on/off
void RIG_TMD710::set_PTT_control(int val)
{
	if (val) sendCommand("TX\r");
	else	 sendCommand("RX\r");
	ptt_ = val;
}

int RIG_TMD710::get_PTT()
{
	return ptt_;
}

// Transceiver power level
double RIG_TMD710::get_power_control()
{
	char seite[2];
	cmd = "PC ";
	snprintf(seite,2, "%d", inuse);
	cmd += seite;
	cmd += '\r';
	int ret = wait_char('\r', 8, 100, "get power", ASC);
	getcr("get power control");
	if (ret < 1) return false;

	getcr("get_power_control");
	int wert;
	int mtr = 30;
	wert = replystr[5] - '0';
	switch (wert) {
			case 0: mtr = 50; break;
			case 1: mtr = 10; break;
			case 2: mtr = 5; break;
		}
	return mtr;
}

void RIG_TMD710::set_power_control(double val)
{
	char mtrs = '2';
	if (val >= 35) mtrs = '0';
	if (val < 35) mtrs = '1';
	if (val < 10) mtrs = '2';
	char seite[2];
	cmd = "PC ";
	snprintf(seite,2, "%d", inuse);
	cmd += seite;
	cmd += ',';
	cmd += mtrs;
	cmd += '\r';
	wait_char('\r', 8, 100, "set power", ASC);
	setcr("set power control");
}

void RIG_TMD710::set_squelch(int val)
{
	char hexstr[3] = "00";
	char seite[2];
	cmd = "SQ ";
	snprintf(seite,2, "%d", inuse);
	cmd += seite;
	cmd += ',';
	snprintf(hexstr,3,"%02X", val);
	cmd += hexstr;
	cmd += '\r';
	wait_char('\r', 9, 100, "set power", ASC);
	setcr("set squelch");
}

// needed for mingw compile which does not support the C++11 stoi template
namespace stoi_fix
{
	int stoi( const std::string& str, std::size_t* pos = 0, int base = 10 )
	{
		const char* begin = str.c_str() ;
		char* end = 0 ;
		long value = std::strtol( begin, &end, base ) ;
		if (pos) *pos = end - begin ;
		return value ;
	}
}

int  RIG_TMD710::get_squelch()
{
	char seite[2];
		cmd = "SQ ";
		snprintf(seite,2, "%d", inuse);
		cmd += seite;
		cmd += '\r';
		int ret = wait_char('\r', 7, 100, "get squelch", ASC);
		if (ret < 1) return false;
		getcr("get_squelch_control");

		char hexstr[3];
		hexstr[0]= replystr[3];
		hexstr[1]= replystr[4];
		hexstr[2]= '\0';

		return stoi_fix::stoi (hexstr, 0, 16);
}
