/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_io.h>
#include "globus_rls_client.h"
#include "globus_rls_rpc.h"
#include "misc.h"
#include "lock.h"
#include "bloom.h"
#include "lrc.h"
#include "db.h"
#include "update.h"
#include <syslog.h>
#include <string.h>
#include <fnmatch.h>

#define METHOD_RLI_UPDATE		"rli_update"
#define METHOD_RLI_UPDATE_LEN		sizeof(METHOD_RLI_UPDATE)

#define METHOD_RLI_UPDATE2		"rli_update2"
#define METHOD_RLI_UPDATE_LEN2		sizeof(METHOD_RLI_UPDATE2)

#define METHOD_RLI_UPDATEBF		"rli_updatebf"
#define METHOD_RLI_UPDATEBF_LEN		sizeof(METHOD_RLI_UPDATEBF)

#define METHOD_RLI_UPDATEBF2		"rli_updatebf2"
#define METHOD_RLI_UPDATEBF_LEN2	sizeof(METHOD_RLI_UPDATEBF2)

#define LS(s)		((s) ? (s) : "null")

extern int		loglevel;
extern int		update_buftime;
extern int		update_immediate;
extern int		update_bf_int;
extern int		update_ll_int;
extern int		update_factor;
extern int		update_retry;
extern int		rli_bloomfilter;
extern char		*myurl;
extern int		myurllen;

static int		rlilistcb(void *a, char *url, char *aflags,
				  char *pattern);
static void		*process_update(void *vrli);
static void		write_update(RLI *rli);
static void		flush_pending(RLI *rli, int queuelocked);
static void		free_update(UPDATE *u);
static char		*cachestr(char *str);

static UPDATE		*freelist_update = NULL;
static UPDATE_LIST	*freelist_updatelist = NULL;
static globus_mutex_t	update_mtx;

#ifdef COUNTUPDATE
int			updatebytes = 0;
#endif

/*
 * Free rlilist, kills update threads and closes open connections.
 * rlilist should already be write locked.
 */
void
update_freelist(RLI **rlilistp)

{
  RLI		*rli;
  UPDATEPAT	*p;

  if (loglevel > 1)
    logit(LOG_DEBUG, "update_freelist:");
  while ((rli = *rlilistp)) {
    if (!(rli->flags & FR_BLOOMFILTER)) {
      update_queue(rli, RU_EXIT);	/* Kill update thread		*/
      globus_mutex_lock(&rli->mtx);	/* Wait for it to exit		*/
      while (rli->flags & FR_UPDATETHREAD)
	globus_cond_wait(&rli->cond, &rli->mtx);
      globus_mutex_unlock(&rli->mtx);
    }

    *rlilistp = (*rlilistp)->nxt;
    while ((p = rli->patternlist)) {
      rli->patternlist = rli->patternlist->nxt;
      globus_libc_free(p->pattern);
      globus_libc_free(p);
    }
    globus_cond_destroy(&rli->cond);
    globus_mutex_destroy(&rli->mtx);
    globus_libc_free(rli->url);
    globus_libc_free(rli);
  }
}

void
update_init()

{
  int	rc;

  if (loglevel)
    logit(LOG_DEBUG, "update_init:");

  if ((rc = globus_mutex_init(&update_mtx, GLOBUS_NULL)) != 0)
    logit(LOG_WARNING, "update_init: globus_mutex_init: %d", rc);
}

/*
 * Allocate entry to be updated on RLIs.  Note space for lfn must be
 * allocated, but lrc string is stored in string cache to avoid
 * duplicating the same string over and over (see cachestr()).
 */
UPDATE *
update_new(char *lfn, char *lrc, RLIOP op)

{
  UPDATE	*u;

  if (loglevel > 2)
    logit(LOG_DEBUG, "update_new: %s %s %d", LS(lfn), LS(lrc), op);

  if (freelist_update) {
    globus_mutex_lock(&update_mtx);
    u = freelist_update;
    freelist_update = freelist_update->nxt;
    globus_mutex_unlock(&update_mtx);
  } else {
    if ((u = (UPDATE *) globus_libc_malloc(sizeof(UPDATE))) == NULL) {
      logit(LOG_WARNING, "update_new: No memory for %s\n", lfn);
      return NULL;
    }
    globus_mutex_init(&u->mtx, GLOBUS_NULL);
  }
  u->lfnlen = strlen(lfn) + 2;		/* Cmd byte plus trailing null	*/
  if ((u->lfn = globus_libc_malloc(u->lfnlen)) == NULL) {
    globus_mutex_lock(&update_mtx);
    u->nxt = freelist_update;
    freelist_update = u;
    globus_mutex_unlock(&update_mtx);
    logit(LOG_WARNING, "update_new: No memory for %s\n", lfn);
    return NULL;
  }
  *u->lfn = (op == rliop_add) ? RLI_ADD : RLI_DELETE;
  strcpy(&u->lfn[1], lfn);
  u->lrc = lrc ? cachestr(lrc) : NULL;
  u->refcnt = 0;
  return u;
}

/*
 * Return true if lfn matches any of the patterns for rli, also return
 * true if rli has no patterns.
 */ 
globus_bool_t
update_patternmatch(RLI *rli, char *lfn)

{
  UPDATEPAT	*p;

  if (loglevel > 2)
    logit(LOG_DEBUG, "update_patternmatch: %s %s", rli->url, lfn);

  if (!rli->patternlist)
    return GLOBUS_TRUE;
  for (p = rli->patternlist; p; p = p->nxt)
    if (fnmatch(p->pattern, lfn, FNM_PATHNAME) == 0)
      return GLOBUS_TRUE;
  return GLOBUS_FALSE;
}

/*
 * Queue update for RLI.
 */
void
update_queue(RLI *rli, UPDATE *u)

{
  UPDATE_LIST	*ul;

  if (loglevel > 2)
    logit(LOG_DEBUG, "update_queue: %s", rli->url);

  if (freelist_updatelist) {
    globus_mutex_lock(&update_mtx);
    ul = freelist_updatelist;
    freelist_updatelist = freelist_updatelist->nxt;
    globus_mutex_unlock(&update_mtx);
  } else if ((ul = (UPDATE_LIST *) globus_libc_malloc(sizeof(UPDATE_LIST))) == NULL) {
    logit(LOG_WARNING, "update_queue: No memory for %s", rli->url);
    return;
  }

  ul->u = u;
  ul->nxt = NULL;
  if (u == RU_EXIT) {
    globus_mutex_lock(&rli->mtx);
    rli->flags |= FR_EXITING;
    globus_mutex_unlock(&rli->mtx);
  } else if (u != RU_SOFTSTATEEND) {
    globus_mutex_lock(&u->mtx);
    u->refcnt++;
    globus_mutex_unlock(&u->mtx);
  }
  globus_mutex_lock(&rli->queue.mtx);
  if (rli->queue.last)
    rli->queue.last->nxt = ul;
  else
    rli->queue.first = ul;
  rli->queue.last = ul;
  globus_cond_signal(&rli->queue.cond);
  globus_mutex_unlock(&rli->queue.mtx);
}

/*
 * Free all updates queued for RLI.  Called at start of full softstate update
 * to free requests waiting to be processed by process_update().  Normally
 * won't have anything to do as updates will be moved in a timely fashion to
 * the pending queue and written to the RLI.  However if the RLI is down
 * or slow want to avoid queueing the same updates over and over.  Note
 * this is distinct from freeing entries on the pending queue (see
 * flush_pending()) which holds entries in current buffer to be written to
 * an RLI.
 */
void
update_flushqueue(RLI *rli)

{
  UPDATE_LIST	*ul;

  if (loglevel > 1)
    logit(LOG_DEBUG, "update_flushqueue: %s", rli->url);

  if (!rli->queue.first)
    return;
  globus_mutex_lock(&rli->queue.mtx);
  globus_mutex_lock(&update_mtx);
  for (ul = rli->queue.first; ul; ul = ul->nxt)
    free_update(ul->u);
  rli->queue.last->nxt = freelist_updatelist;
  freelist_updatelist = rli->queue.first;
  rli->queue.first = NULL;
  rli->queue.last = NULL;
  globus_mutex_unlock(&update_mtx);
  globus_mutex_unlock(&rli->queue.mtx);
}

/*
 * Get list of RLIs updated by this server.
 */
void
update_readrli(void *dbh, LOCK *listlock, RLI **rlilistp)

{
  int		rc;
  char		errbuf[MAXERRMSG];
  char		buf[MAXERRMSG];

  rls_lock_get(listlock, writelock);
  update_freelist(rlilistp);
  if ((rc = db_update_get_part(dbh, NULL, NULL, rlilistcb, rlilistp,
			       errbuf)) != GLOBUS_RLS_SUCCESS)
    if (loglevel || rc != GLOBUS_RLS_RLI_NEXIST)
      logit(LOG_WARNING, "update_readrli: %s",
	    globus_rls_errmsg(rc, errbuf, buf, MAXERRMSG));
  rls_lock_release(listlock, writelock);
}

/*
 * Open connection to RLI server and send bloom filter.  bloomfilter
 * should be read locked by caller (by locking xxx_rlilistlock).
 */
void
update_sendbf(RLI *rli, bloomfilter_t *bloomfilter, char *lrcurl)

{
  int			rc;
  globus_result_t	r;
  char			errbuf[MAXERRMSG];
  char			errbuf2[MAXERRMSG];
  char			buf1[100];
  char			buf2[100];
  char			buf3[100];
  BUFFER		b;
  globus_size_t		nb;
  int			ui;
  int			n;

  if (loglevel)
    logit(LOG_DEBUG, "update_sendbf: Sending bloomfilter to %s", rli->url);

  globus_mutex_lock(&rli->mtx);

  /*
   * If immediate mode and we're synced (softstatestart == lastupdate)
   * then the update interval is multiplied by update_factor.
   */
  if (update_immediate && rli->softstatestart == rli->lastupdate)
    ui = update_bf_int * update_factor;
  else
    ui = update_bf_int;
  rli->softstatestart = time(0);
  if ((r = globus_rls_client_connect(rli->url, &rli->h)) != GLOBUS_SUCCESS) {
    globus_rls_client_error_info(r, NULL, errbuf, MAXERRMSG, GLOBUS_FALSE);
    logit(LOG_WARNING, "updatebf(%s): %s", rli->url, errbuf);
    globus_mutex_unlock(&rli->mtx);
    return;
  }

  sprintf(buf1, "%d", bloomfilter->bfsize);
  sprintf(buf2, "%d", bloomfilter->numhash);
  sprintf(buf3, "%d", ui);
  if (lrcurl) {
    rli->iov[0].iov_base = METHOD_RLI_UPDATEBF2;
    rli->iov[0].iov_len = METHOD_RLI_UPDATEBF_LEN2;
    rli->iov[1].iov_base = lrcurl;
    rli->iov[1].iov_len = strlen(lrcurl) + 1;
    n = 2;
  } else {
    rli->iov[0].iov_base = METHOD_RLI_UPDATEBF;
    rli->iov[0].iov_len = METHOD_RLI_UPDATEBF_LEN;
    n = 1;
  }
  rli->iov[n].iov_base = myurl;
  rli->iov[n++].iov_len = myurllen;
  rli->iov[n].iov_base = buf1;
  rli->iov[n++].iov_len = strlen(buf1) + 1;
  rli->iov[n].iov_base = buf2;
  rli->iov[n++].iov_len = strlen(buf2) + 1;
  rli->iov[n].iov_base = buf3;
  rli->iov[n++].iov_len = strlen(buf3) + 1;
  rli->iov[n].iov_base = (char *) bloomfilter->bits;
  rli->iov[n++].iov_len = BITS2BYTES(bloomfilter->bfsize);
  if ((rc = rrpc_writev(&rli->h->handle, rli->iov, n, &nb,
			errbuf)) == GLOBUS_RLS_SUCCESS) {
    rc = rrpc_getresult(rli->h, &b, errbuf);
#ifdef COUNTUPDATE
    updatebytes += nb;
#endif
  }
  if (rc != GLOBUS_RLS_SUCCESS)
    logit(LOG_WARNING, "updatebf(%s): %s", rli->url,
	  globus_rls_errmsg(rc, errbuf, errbuf2, MAXERRMSG));
  else {
    rli->lastupdate = rli->softstatestart;
    if (loglevel)
      logit(LOG_DEBUG, "updatebf: Update %s took %d (size %u)",
	    rli->url, time(0) - rli->softstatestart, bloomfilter->bfsize);
  }
	  
  globus_rls_client_close(rli->h);
  globus_mutex_unlock(&rli->mtx);
}

/*
 * Callback to add rli/rlipartition table entry from database to
 * rlilist.  rlilistlock should already be write locked.
 */
static int
rlilistcb(void *a, char *url, char *aflags, char *pattern)

{
  RLI		**rlilistp = (RLI **) a;
  RLI		*rli = NULL;
  UPDATEPAT	*p = NULL;
  pthread_t	thr;
  int		rc;
  int		dbflags;

  if (*pattern) {
    if ((p = globus_libc_malloc(sizeof(UPDATEPAT))) == NULL)
      goto error;
    if ((p->pattern = globus_libc_strdup(pattern)) == NULL)
      goto error;
    p->nxt = NULL;
  }

  for (rli = *rlilistp; rli; rli = rli->nxt)
    if (strcasecmp(url, rli->url) == 0)
      break;
  if (rli) {
    if (p) {
      p->nxt = rli->patternlist;
      rli->patternlist = p;
    }
    return GLOBUS_TRUE;
  }

  if ((rli = (RLI *) globus_libc_malloc(sizeof(RLI))) == NULL)
    goto error;
  if ((rli->url = (char *) globus_libc_strdup(url)) == NULL)
    goto error;
  globus_mutex_init(&rli->mtx, GLOBUS_NULL);
  globus_cond_init(&rli->cond, GLOBUS_NULL);
  rli->patternlist = p;
  globus_mutex_init(&rli->queue.mtx, GLOBUS_NULL);
  globus_cond_init(&rli->queue.cond, GLOBUS_NULL);
  rli->queue.first = NULL;
  rli->queue.last = NULL;
  rli->queue.pending = NULL;
  rli->h = NULL;
  rli->idx = 0;
  rli->retrytime = 0;
  rli->softstatestart = 0;
  rli->lastupdate = 0;
  rli->lastlrc = NULL;
  rli->flags = 0;
  dbflags = atoi(aflags);
  if (dbflags & FRLI_RLI)
    if (rli_bloomfilter)
      dbflags |= FRLI_BLOOMFILTER;
  if (dbflags & FRLI_BLOOMFILTER)
    rli->flags |= FR_BLOOMFILTER;
  else {
    if ((rc = pthread_create(&thr, GLOBUS_NULL, process_update, rli)) != 0){
      logit(LOG_WARNING, "rlilistcb: pthread_create: %s",
	    globus_libc_system_error_string(rc));
      goto error;
    }
    if ((rc = pthread_detach(thr)) != 0)
      logit(LOG_INFO, "rlilistcb: pthread_detach: %s",
	    globus_libc_system_error_string(rc));
  }
  rli->nxt = *rlilistp;
  *rlilistp = rli;
  return GLOBUS_TRUE;

 error:
  logit(LOG_WARNING, "rlilistcb(%s,%s): No memory", rli, pattern);
  if (p) {
    if (p->pattern)
      globus_libc_free(p->pattern);
    globus_libc_free(p);
  }
  if (rli) {
    if (rli->url)
      globus_libc_free(rli->url);
    globus_libc_free(rli);
  }
  return GLOBUS_FALSE;
}

/*
 * Process RLI updates.  rlilistlock is not necessarily locked, however
 * threads that modify rlilist must first call update_freelist()
 * which will kill threads executing this function (by queuing a
 * RU_EXIT update which will cause pending updates to be flushed
 * before the thread exits).
 */
static void *
process_update(void *vrli)

{
  RLI			*rli = (RLI *) vrli;
  UPDATE_LIST		*ul;
  struct timespec	t;
  time_t		now;

  if (loglevel)
    logit(LOG_DEBUG, "process_update(%s): Start thread", rli->url);
  globus_mutex_lock(&rli->mtx);
  rli->flags |= FR_UPDATETHREAD;
  globus_mutex_unlock(&rli->mtx);

  t.tv_nsec = 0;
  while (1) {
    /*
     * rli->retrytime is non-zero when an RLI server can't be updated, and
     * is set to the time at which the connect should be retried.  Following
     * loop waits until rli->retrytime has arrived, then attempts to send
     * pending updates unless a new full update has started, in which case
     * it flushes queue.
     */
    while (rli->retrytime && !(rli->flags & FR_EXITING)) {
      now = time(0);
      t.tv_sec = rli->retrytime;
      globus_mutex_lock(&rli->queue.mtx);
      while (rli->retrytime > now && !(rli->flags & FR_EXITING)) {
        globus_cond_timedwait(&rli->queue.cond, &rli->queue.mtx, &t);
	now = time(0);
      }
      rli->retrytime = 0;
      /*
       * If a new full update has started, we want to flush pending updates
       * to prevent queue from growing infinitely if an RLI is dead.
       * In this case terminate this loop and process new update normally.
       * If RLI is still dead retrytime will get set again and we'll be
       * back here.
       */
      if (rli->flags & FR_FLUSHPENDING) {
	flush_pending(rli, 1);
	globus_mutex_unlock(&rli->queue.mtx);
	globus_mutex_lock(&rli->mtx);
	rli->flags &= ~FR_FLUSHPENDING;
	globus_mutex_unlock(&rli->mtx);
      } else {
	globus_mutex_unlock(&rli->queue.mtx);
	write_update(rli);	/* Write pending updates	*/
      }
    }

    /*
     * Wait for an update to be queued, or if there's pending output wake
     * up if its been pending more than update_buftime seconds and write it.
     */
    globus_mutex_lock(&rli->queue.mtx);
    while (!(ul = rli->queue.first))
      if (rli->idx) {
	t.tv_sec = rli->bufferage + update_buftime;
        if (t.tv_sec <= time(0))
	  break;
	globus_cond_timedwait(&rli->queue.cond, &rli->queue.mtx, &t);
      } else
	globus_cond_wait(&rli->queue.cond, &rli->queue.mtx);

    /*
     * It's possible we started a new full update while we had old
     * updates waiting to be written to a dead RLI.  Since we're
     * sending a full update now there's no point in keeping the
     * old queued updates, so flush them.  This prevents the
     * queue from growing infinitely when an RLI is permanently dead.
     */
    if (rli->flags & FR_FLUSHPENDING) {
      flush_pending(rli, 1);
      globus_mutex_lock(&rli->mtx);
      rli->flags &= ~FR_FLUSHPENDING;
      globus_mutex_unlock(&rli->mtx);
    }

    /*
     * If we woke up because a new update was queued, remove update
     * from input queue.
     */
    if (ul)
      if (!(rli->queue.first = rli->queue.first->nxt))
	rli->queue.last = NULL;
    globus_mutex_unlock(&rli->queue.mtx);

    if (!ul) {		/* Timer expired, flush output queue	*/
      write_update(rli);
      continue;
    }

    if (ul->u != RU_EXIT && ul->u != RU_SOFTSTATEEND) {
      /*
       * If an RLI is updating an RLI then we need to include the LRC
       * where the LFN originally came from, as well as our URL.  If we're
       * an LRC these are the same.  For backwards compatibility (before
       * RLIs could update RLIs) we have 2 update methods.  First we write
       * current buffer if the lrc has changed so a new one will be started.
       */
      if (rli->idx && (rli->lastlrc != ul->u->lrc)) {
	write_update(rli);
	rli->lastlrc = ul->u->lrc;
      }
      if (!rli->idx) {
	if (ul->u->lrc) {	/* Send URL of originating LRC if set	*/
	  rli->iov[0].iov_base = METHOD_RLI_UPDATE2;
	  rli->iov[0].iov_len = METHOD_RLI_UPDATE_LEN2;
	  rli->iov[1].iov_base = ul->u->lrc;
	  rli->iov[1].iov_len = strlen(ul->u->lrc) + 1;
	  rli->idx = 3;
	} else {
	  rli->iov[0].iov_base = METHOD_RLI_UPDATE;
	  rli->iov[0].iov_len = METHOD_RLI_UPDATE_LEN;
	  rli->idx = 2;
	}
	rli->iov[rli->idx - 1].iov_base = myurl;
	rli->iov[rli->idx - 1].iov_len = myurllen;
	rli->bufferage = time(0);
      }
      rli->iov[rli->idx].iov_base = ul->u->lfn;
      rli->iov[rli->idx++].iov_len = ul->u->lfnlen;
    }

    /*
     * Add update to pending list so will get freed when buffer is written.
     */
    globus_mutex_lock(&rli->queue.mtx);
    ul->nxt = rli->queue.pending;
    rli->queue.pending = ul;
    globus_mutex_unlock(&rli->queue.mtx);

    if (ul->u == RU_EXIT || ul->u == RU_SOFTSTATEEND || rli->idx >= RLIIOV - 2)
      write_update(rli);

    if (ul->u == RU_EXIT) {
      globus_mutex_lock(&rli->mtx);
      if (rli->flags & FR_OPEN)
	globus_rls_client_close(rli->h);
      rli->flags &= ~(FR_UPDATETHREAD|FR_OPEN);
      globus_cond_signal(&rli->cond);
      globus_mutex_unlock(&rli->mtx);
      if (loglevel)
        logit(LOG_DEBUG, "process_update(%s): End thread", rli->url);
      pthread_exit(0);
    }
  }
}

/*
 * If there's pending updates for this RLI add trailing null byte to output
 * buffer and write to RLI server.
 * Decrements UPDATE refcnts referenced by rli->iov[] and frees if
 * no more references.  If we didn't open the rli connection this call
 * and an error occurs, close the connection and retry.  It's possible
 * the rli server timed out it's side of the connection, or some other
 * recoverable error occured.  Frees rli->pending list.  If an error occurs
 * leave data in output buffer, and set rli->retrytime to the time at which
 * to attempt reconnect.  Some errors are immediately retryable (DBERROR,
 * GLOBUSERR).
 */
static void
write_update(RLI *rli)

{
  globus_size_t		nb;
  BUFFER		b;
  char			errbuf[MAXERRMSG];
  char			errbuf2[MAXERRMSG];
  int			rc;
  globus_result_t	r;
  int			immediateretry = 1;

  if (loglevel > 1)
    logit(LOG_DEBUG, "write_update: %s %d pending", rli->url, rli->idx);

  if (rli->idx) {
    rli->iov[rli->idx].iov_base = "";
    rli->iov[rli->idx++].iov_len = 1;

   retry:
    if (!(rli->flags & FR_OPEN)) {
      if (rli->flags & FR_EXITING) {
	flush_pending(rli, 0);
	return;
      }
      rli->h = NULL;
      if ((r = globus_rls_client_connect(rli->url,&rli->h)) != GLOBUS_SUCCESS){
	globus_rls_client_error_info(r, NULL, errbuf, MAXERRMSG, GLOBUS_FALSE);
	logit(LOG_WARNING, "write_update(%s): %s", rli->url, errbuf);
	if (rli->flags & FR_EXITING)	/* Discard pending output?	*/
	  flush_pending(rli, 0);
        else {
	  rli->idx--;
	  rli->retrytime = time(0) + update_retry;
	}
	return;
      }
      globus_mutex_lock(&rli->mtx);
      rli->flags |= FR_OPEN;
      globus_mutex_unlock(&rli->mtx);
      immediateretry = 0;
    }

    if ((rc = rrpc_writev(&rli->h->handle, rli->iov, rli->idx,
			  &nb, errbuf)) == GLOBUS_RLS_SUCCESS) {
      rc = rrpc_getresult(rli->h, &b, errbuf);
#ifdef COUNTUPDATE
      updatebytes += nb;
#endif
    }

    /*
     * On some errors we want to try closing and reopening connection
     * to RLI server.  This allows us to recover automatically if an
     * RLI is restarted or drops our connection for some reason.
     */
    if (rc == GLOBUS_RLS_DBERROR || rc == GLOBUS_RLS_GLOBUSERR ||
        rc == GLOBUS_RLS_TIMEOUT || rc == GLOBUS_RLS_INVHANDLE) {
      globus_rls_client_close(rli->h);
      globus_mutex_lock(&rli->mtx);
      rli->flags &= ~FR_OPEN;
      globus_mutex_unlock(&rli->mtx);
      logit(LOG_DEBUG, "write_update(%s): %s: retrying", rli->url,
	    globus_rls_errmsg(rc, errbuf, errbuf2, MAXERRMSG));
      if (immediateretry)
	goto retry;
      if (rli->flags & FR_EXITING)	/* Discard pending output?	*/
	flush_pending(rli, 0);
      else {
	rli->idx--;
	rli->retrytime = time(0) + update_retry;
      }
      return;
    }
    if (rc != GLOBUS_RLS_SUCCESS) {
      logit(LOG_WARNING, "write_update(%s): %s", rli->url,
	    globus_rls_errmsg(rc, errbuf, errbuf2, MAXERRMSG));
      if (rli->flags & FR_SYNCED) {
	globus_mutex_lock(&rli->mtx);
	rli->flags &= ~FR_SYNCED;
	globus_mutex_unlock(&rli->mtx);
      }      
    }
  }

  flush_pending(rli, 0);
}

/*
 * Free list of pending updates to rli.  If we see a RU_SOFTSTATEEND
 * (queued at end of softstate update) then log time update took.
 */
static void
flush_pending(RLI *rli, int queuelocked)

{
  UPDATE_LIST	*ul;
  UPDATE_LIST	*lul;
  time_t	ut;

  if (loglevel > 1)
    logit(LOG_DEBUG, "flush_pending(%s): queue %X", rli->url,
	  rli->queue.pending);

  if (!rli->queue.pending)
    return;
    
  if (!queuelocked)
    globus_mutex_lock(&rli->queue.mtx);

  /* Free UPDATEs on pending list, then free pending list	*/
  globus_mutex_lock(&update_mtx);
  for (ul = rli->queue.pending; ul; ul = ul->nxt) {
    if (ul->u == RU_SOFTSTATEEND) {
      if (rli->flags & FR_SYNCED)
	rli->lastupdate = time(0);
      if (loglevel) {
	ut = time(0) - rli->softstatestart;
	logit(LOG_DEBUG, "write_update: update %s took %02u:%02u:%02u",
	      rli->url, ut / 3600, (ut / 60 % 60), ut % 60);
      }
    } else
      free_update(ul->u);
    lul = ul;
  }
  lul->nxt = freelist_updatelist;
  freelist_updatelist = rli->queue.pending;
  globus_mutex_unlock(&update_mtx);
  rli->queue.pending = NULL;
  rli->idx = 0;

  if (!queuelocked)
    globus_mutex_unlock(&rli->queue.mtx);
}

/*
 * Decrement refcnt on UPDATE structure, if 0 then add to freelist.
 * Note update_mtx, which protects freelist_update, should already
 * be locked by caller.
 */
static void
free_update(UPDATE *u)

{
  if (u == RU_EXIT || u == RU_SOFTSTATEEND)
    return;
  globus_mutex_lock(&u->mtx);
  u->refcnt--;
  globus_mutex_unlock(&u->mtx);
  if (u->refcnt == 0) {
    globus_libc_free(u->lfn);
    u->nxt = freelist_update;
    freelist_update = u;
  }
}

/*
 * cachestr - Add a string to cache and return
 *   pointer to it.  Avoids allocating space for duplicate strings (eg
 *   URLs queued for softstate updates).
 */
static char *
cachestr(char *str)

{
  static int			inited = 0;
  static globus_hashtable_t	ht;
  char				*s;

  if (!str)
    return NULL;

  if (!inited) {
    globus_hashtable_init(&ht, 511, globus_hashtable_string_hash,
			  globus_hashtable_string_keyeq);
    inited = 1;
  }
  if (!(s = globus_hashtable_lookup(&ht, str))) {
    s = globus_libc_strdup(str);
    globus_hashtable_insert(&ht, s, s);
  }
  return s;
}
