------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov, Maxim Reznik                 --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-notifications.adb,v 1.12 2007/04/25 10:36:29 vagul Exp $

with Ada.Exceptions;
with Ada.Text_IO;

with OCI.Lib;
with OCI.Thread;
with OCI.Environments;
with Interfaces.C;
with System.Address_To_Access_Conversions;

package body OCI.Thick.Notifications is

   use type System.Address;

   To_UB4 : constant array (Namespace_Enum) of Ub4
     := (AQ        => OCI_SUBSCR_NAMESPACE_AQ,
         Anonymous => OCI_SUBSCR_NAMESPACE_ANONYMOUS);

   function OCISubscriptionNotify
      (pCtx        : in DVoid;
       pSubscrHp   : in OCISubscription;
       pPayload    : in DVoid;
       iPayloadLen : in Ub4;
       pDescriptor : in DVoid;
       iMode       : in Ub4)
       return      SWord;

   pragma Convention (C, OCISubscriptionNotify);

   ------------
   -- Create --
   ------------

   procedure Create
     (Item      : in out Subscription;
      Name      : in     String;
      Namespace : in     Namespace_Enum) is
   begin
      Create (Item, To_UB4 (Namespace));

      Set_Attr
        (OCIHandle (Item.Handle),
         OCI_HTYPE_SUBSCRIPTION,
         OCI_ATTR_SUBSCR_NAME,
         Name);
   end Create;

   procedure Create
     (Item      : in out Base_Subscription;
      Namespace : in     OCI.Lib.Ub4) is
   begin
      if Item.Handle /= OCISubscription (Empty_Handle) then
         --  Avoid memory and handles leak.

         raise Already_Registered;
      end if;

      Item.Handle := OCISubscription
                       (Alloc_Handle (Thread.Environment,
                                      OCI_HTYPE_SUBSCRIPTION));

      Set_Attr
        (OCIHandle (Item.Handle),
         OCI_HTYPE_SUBSCRIPTION,
         OCI_ATTR_SUBSCR_NAMESPACE,
         Value => Namespace'Address);
   end Create;

   -------------
   -- Disable --
   -------------

   procedure Disable (Item : in Base_Subscription) is
   begin
      Check_Error
        (OCISubscriptionDisable (Item.Handle, Thread.Error, OCI_DEFAULT));
   end Disable;

   ------------
   -- Enable --
   ------------

   procedure Enable (Item : in Base_Subscription)is
   begin
      Check_Error
        (OCISubscriptionEnable (Item.Handle, Thread.Error, OCI_DEFAULT));
   end Enable;

   --------------
   -- Finalize --
   --------------

   procedure Finalize (Object : in out Subscription) is
   begin
      if Object.Handle /= OCISubscription (Empty_Handle) then
         Free
           (H => OCIHandle (Object.Handle), HType => OCI_HTYPE_SUBSCRIPTION);
      end if;
   end Finalize;

   ---------------------
   -- Internal_Notify --
   ---------------------

   procedure Internal_Notify
     (Item       : in out Base_Subscription;
      Descriptor : in     OCI.Lib.DVoid;
      Buffer     : in     Buffer_Type)
   is
      pragma Unreferenced (Item);
      pragma Unreferenced (Buffer);
      pragma Unreferenced (Descriptor);
   begin
      --  To be overloaded in child types
      null;
   end Internal_Notify;

   procedure Internal_Notify
     (Item       : in out Subscription;
      Descriptor : in     OCI.Lib.DVoid;
      Buffer     : in     Buffer_Type)
   is
      pragma Unreferenced (Descriptor);
   begin
      Notify (Subscription'Class (Item), Buffer);
   end Internal_Notify;

   ------------
   -- Notify --
   ------------

   procedure Notify
     (Item   : in out Subscription;
      Buffer : in     Buffer_Type)
   is
      pragma Unreferenced (Item);
      pragma Unreferenced (Buffer);
   begin
      null;
   end Notify;

   ---------------------------
   -- OCISubscriptionNotify --
   ---------------------------

   function OCISubscriptionNotify
      (pCtx        : in DVoid;
       pSubscrHp   : in OCISubscription;
       pPayload    : in DVoid;
       iPayloadLen : in Ub4;
       pDescriptor : in DVoid;
       iMode       : in Ub4)
       return      SWord
   is
      pragma Unreferenced (pSubscrHp);
      pragma Unreferenced (iMode);

      package Converter is new
         System.Address_To_Access_Conversions (Subscription'Class);

      use Converter;
      use Ada.Text_IO;

      Subscr : constant Object_Pointer := To_Pointer (pCtx);
      --  Queue_Name : String := Get_Attr (pDescriptor,
      --     OCI_DTYPE_AQNFY_DESCRIPTOR, OCI_ATTR_QUEUE_NAME);
      --  CONSUMER_Name : String := Get_Attr (pDescriptor,
      --     OCI_DTYPE_AQNFY_DESCRIPTOR, OCI_ATTR_CONSUMER_NAME);
      Buffer : constant Buffer_Type := (Ptr    => pPayload,
                               Length => Natural (iPayloadLen));
   begin
      Internal_Notify (Subscr.all, pDescriptor, Buffer);
      return OCI_CONTINUE;

   exception
      when E : others =>
         Put_Line (Current_Error, Ada.Exceptions.Exception_Information (E));

      return OCI_ERROR;
   end OCISubscriptionNotify;

   ----------
   -- Post --
   ----------

   procedure Post
     (Item    : in out Subscription;
      Connect : in     Connection;
      Data    : in     String) is
   begin
      Set_Attr
        (H     => OCIHandle (Item.Handle),
         HType => OCI_HTYPE_SUBSCRIPTION,
         Attr  => OCI_ATTR_SUBSCR_PAYLOAD,
         Value => Data);

      Check_Error (OCISubscriptionPost
        (svchp     => OCISvcCtx (Handle (Connect)),
         subscrhpp => Item.Handle'Access,
         count     => 1,
         errhp     => Thread.Error,
         mode      => OCI_DEFAULT));

      Item.Connect := Connect;

      Set_Attr
        (H     => OCIHandle (Item.Handle),
         HType => OCI_HTYPE_SUBSCRIPTION,
         Attr  => OCI_ATTR_SUBSCR_PAYLOAD,
         Value => System.Null_Address);
   end Post;

   --------------
   -- Register --
   --------------

   procedure Register
     (Item    : in out Base_Subscription;
      Connect : in     Connection)
   is
      RC : SWord;
      Connect_Handle : constant OCIHandle := Handle (Connect);
   begin
      Set_Attr
        (OCIHandle (Item.Handle),
         OCI_HTYPE_SUBSCRIPTION,
         OCI_ATTR_SUBSCR_CALLBACK,
         Value => OCISubscriptionNotify'Address);

      Set_Attr
        (OCIHandle (Item.Handle),
         OCI_HTYPE_SUBSCRIPTION,
         OCI_ATTR_SUBSCR_CTX,
         Value => Item'Address);

      RC := OCISubscriptionRegister
            (svchp     => OCISvcCtx (Connect_Handle),
             subscrhpp => Item.Handle'Access,
             count     => 1,
             errhp     => Thread.Error,
             mode      => OCI_DEFAULT);

      Check_Error (RC);
      Item.Connect := Connect;
   end Register;

   ---------------
   -- To_String --
   ---------------

   function To_String (Buffer : in Buffer_Type) return String is
      use Interfaces.C;

      type Data_Type is new char_array (0 .. size_t (Buffer.Length) - 1);

      package Converter is
        new System.Address_To_Access_Conversions (Data_Type);
      use Converter;

      Ptr : constant Object_Pointer := To_Pointer (Buffer.Ptr);
   begin
      if Ptr = null then
         return "";
      else
         return To_Ada (Ptr.all, Trim_Nul => False);
      end if;
   end To_String;

   ----------------
   -- Unregister --
   ----------------

   procedure Unregister (Item : in out Base_Subscription) is
   begin
      Check_Error (OCISubscriptionUnRegister
                     (svchp    => OCISvcCtx (Handle (Item.Connect)),
                      subscrhp => Item.Handle,
                      errhp    => Thread.Error,
                      mode     => OCI_DEFAULT));
   end Unregister;

begin
   OCI.Environments.Set_Create_Mode_Flag (OCI_EVENTS);
end OCI.Thick.Notifications;
