module Main(main) where

import Data.HMAC
import Test.HUnit
import Codec.Utils (Octet)
import Data.Char(ord)

-- HMAC-SHA1 tests from
-- <http://csrc.nist.gov/publications/fips/fips198/fips-198a.pdf>

-- HMAC-MD5 tests from
-- <http://www.ietf.org/rfc/rfc2104.txt>

-- Test #1: SHA-1 64-byte key (single block)

hmacTest1 =
   TestCase (
      assertEqual "Test HMAC-SHA1 #1 with 64-byte key"
                  hmac_1 (hmac_sha1 key_1 message_1)
   )

message_1 = string2words "Sample #1"

key_1 :: [Octet]
key_1 =
    [0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
     0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
     0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
     0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
     0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
     0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
     0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
     0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f]

hmac_1 :: [Octet]
hmac_1 =
    [0x4f, 0x4c, 0xa3, 0xd5, 0xd6, 0x8b, 0xa7, 0xcc,
     0x0a, 0x12, 0x08, 0xc9, 0xc6, 0x1e, 0x9c, 0x5d,
     0xa0, 0x40, 0x3c, 0x0a]


-- Test #2: SHA 20-byte key (short block)

hmacTest2 =
   TestCase (
      assertEqual "Test HMAC-SHA1 #2 with 20-byte key"
                  hmac_2 (hmac_sha1 key_2 message_2)
   )

message_2 = string2words "Sample #2"

key_2 :: [Octet]
key_2 =
    [0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
     0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
     0x40, 0x41, 0x42, 0x43]

hmac_2 :: [Octet]
hmac_2 =
    [0x09, 0x22, 0xd3, 0x40, 0x5f, 0xaa, 0x3d, 0x19,
     0x4f, 0x82, 0xa4, 0x58, 0x30, 0x73, 0x7d, 0x5c,
     0xc6, 0xc7, 0x5d, 0x24]

-- Test #3: SHA 100-byte key (multi-block)

hmacTest3 =
   TestCase (
      assertEqual "Test HMAC-SHA1 #3 with 100-byte key"
                  hmac_3 (hmac_sha1 key_3 message_3)
   )

message_3 = string2words "Sample #3"

key_3 :: [Octet]
key_3 =
    [0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
     0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
     0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
     0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
     0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
     0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
     0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
     0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
     0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
     0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
     0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
     0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
     0xb0, 0xb1, 0xb2, 0xb3]

hmac_3 :: [Octet]
hmac_3 =
    [0xbc, 0xf4, 0x1e, 0xab, 0x8b, 0xb2, 0xd8, 0x02,
     0xf3, 0xd0, 0x5c, 0xaf, 0x7c, 0xb0, 0x92, 0xec,
     0xf8, 0xd1, 0xa3, 0xaa]

-- Test #4: SHA 49-byte key (truncated to 12-byte HMAC)

hmacTest4 =
   TestCase (
      assertEqual "Test HMAC-SHA1 #4 with 49-byte key, truncated to 12-bytes"
                  hmac_4 (take 12 (hmac_sha1 key_4 message_4))
   )

message_4 = string2words "Sample #4"

key_4 :: [Octet]
key_4 =
    [0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
     0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
     0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
     0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
     0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
     0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
     0xa0]

hmac_4 :: [Octet]
hmac_4 =
    [0x9e, 0xa8, 0x86, 0xef, 0xe2, 0x68, 0xdb, 0xec,
     0xce, 0x42, 0x0c, 0x75]

-- Test #5: MD5 Test from rfc 2104 (16-byte key)

hmacTest5 =
   TestCase (
      assertEqual "Test HMAC-MD5 with 16-byte key"
                  hmac_5 (hmac_md5 key_5 message_5)
   )

message_5 = string2words "Hi There"

key_5 :: [Octet]
key_5 = take 16 (repeat 0xb)

hmac_5 :: [Octet]
hmac_5 = [0x92, 0x94, 0x72, 0x7a, 0x36, 0x38, 0xbb, 0x1c,
          0x13, 0xf4, 0x8e, 0xf8, 0x15, 0x8b, 0xfc, 0x9d]

-- Test #6: MD5 Test from rfc 2104 (28-byte key)

hmacTest6 =
   TestCase (
      assertEqual "Test HMAC-MD5 with 28-byte key"
                  hmac_6 (hmac_md5 key_6 message_6)
   )

message_6 = string2words "what do ya want for nothing?"

key_6 :: [Octet]
key_6 = string2words "Jefe"

hmac_6 :: [Octet]
hmac_6 = [0x75, 0x0c, 0x78, 0x3e, 0x6a, 0xb0, 0xb5, 0x03,
          0xea, 0xa8, 0x6e, 0x31, 0x0a, 0x5d, 0xb7, 0x38]

-- Test #7: MD5 Test from rfc 2104 (16-byte key)

hmacTest7 =
   TestCase (
      assertEqual "Test HMAC-MD5 with 16-byte key"
                  hmac_7 (hmac_md5 key_7 message_7)
   )

message_7 = take 50 (repeat 0xdd)

key_7 :: [Octet]
key_7 = take 16 (repeat 0xaa)

hmac_7 :: [Octet]
hmac_7 = [0x56, 0xbe, 0x34, 0x52, 0x1d, 0x14, 0x4c, 0x88,
          0xdb, 0xb8, 0xc7, 0x33, 0xf0, 0xe8, 0xb3, 0xf6]


string2words :: String -> [Octet]
string2words = map (fromIntegral . ord)

-- Run the tests

tests =
   TestList [hmacTest1, hmacTest2, hmacTest3, hmacTest4,
            hmacTest5, hmacTest6, hmacTest7]

main = runTestTT tests

