/* Copyright Joyent, Inc. and other Node contributors. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include "uv.h"
#include "internal.h"

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <fcntl.h>
#include <poll.h>

#if defined(__APPLE__) && !TARGET_OS_IPHONE
# include <crt_externs.h>
# define environ (*_NSGetEnviron())
#else
extern char **environ;
#endif


static QUEUE* uv__process_queue(uv_loop_t* loop, int pid) {
  assert(pid > 0);
  return loop->process_handles + pid % ARRAY_SIZE(loop->process_handles);
}


static uv_process_t* uv__process_find(uv_loop_t* loop, int pid) {
  uv_process_t* handle;
  QUEUE* h;
  QUEUE* q;

  h = uv__process_queue(loop, pid);

  QUEUE_FOREACH(q, h) {
    handle = QUEUE_DATA(q, uv_process_t, queue);
    if (handle->pid == pid) return handle;
  }

  return NULL;
}


static void uv__chld(uv_signal_t* handle, int signum) {
  uv_process_t* process;
  int exit_status;
  int term_signal;
  int status;
  pid_t pid;

  assert(signum == SIGCHLD);

  for (;;) {
    pid = waitpid(-1, &status, WNOHANG);

    if (pid == 0)
      return;

    if (pid == -1) {
      if (errno == ECHILD)
        return; /* XXX stop signal watcher? */
      else
        abort();
    }

    process = uv__process_find(handle->loop, pid);
    if (process == NULL)
      continue; /* XXX bug? abort? */

    uv__handle_stop(process);

    if (process->exit_cb == NULL)
      continue;

    exit_status = 0;
    term_signal = 0;

    if (WIFEXITED(status))
      exit_status = WEXITSTATUS(status);

    if (WIFSIGNALED(status))
      term_signal = WTERMSIG(status);

    if (process->errorno)
      exit_status = process->errorno; /* execve() failed */

    process->exit_cb(process, exit_status, term_signal);
  }
}


int uv__make_pipe(int fds[2], int flags) {
#if defined(__linux__)
  static int no_pipe2;

  if (no_pipe2)
    goto skip;

  if (uv__pipe2(fds, flags | UV__O_CLOEXEC) == 0)
    return 0;

  if (errno != ENOSYS)
    return -errno;

  no_pipe2 = 1;

skip:
#endif

  if (pipe(fds))
    return -errno;

  uv__cloexec(fds[0], 1);
  uv__cloexec(fds[1], 1);

  if (flags & UV__F_NONBLOCK) {
    uv__nonblock(fds[0], 1);
    uv__nonblock(fds[1], 1);
  }

  return 0;
}


/*
 * Used for initializing stdio streams like options.stdin_stream. Returns
 * zero on success.
 */
static int uv__process_init_stdio(uv_stdio_container_t* container, int *fd) {
  switch (container->type) {
    case UV_STREAM:
      if (container->data.stream == NULL) {
        *fd = -1;
        return 0;
      } else {
        *fd = container->data.stream->io_watcher.fd;
      }
      break;
    case UV_RAW_FD:
    case UV_RAW_HANDLE:
      *fd = container->data.fd;
      break;
    default:
      assert (0 && "Unexpected flags");
      *fd = -1;
    return -EINVAL;
      errno = EINVAL;
  }
  if (*fd == -1) {
    errno = EINVAL;
    return -1;
  } else {
    return 0;
  }
}

static void uv__write_int(int fd, int val) {
  ssize_t n;

  do
    n = write(fd, &val, sizeof(val));
  while (n == -1 && errno == EINTR);

  if (n == -1 && errno == EPIPE)
    return; /* parent process has quit */

  assert(n == sizeof(val));
}


static void uv__process_child_init(uv_process_options_t options,
                                   int stdio_count,
                                   int *pipes,
                                   int error_fd) {
  int use_fd;
  int fd;

  if (options.flags & UV_PROCESS_DETACHED)
    setsid();

  for (fd = 0; fd < stdio_count; fd++) {
    use_fd = pipes[fd];

    if (use_fd >= 0) {
    } else if (fd >= 3) {
      continue;
    } else {
      /* redirect stdin, stdout and stderr to /dev/null even if UV_IGNORE is
       * set
       */
      use_fd = open("/dev/null", fd == 0 ? O_RDONLY : O_RDWR);

      if (use_fd == -1) {
        uv__write_int(error_fd, -errno);
        /* perror("failed to open stdio"); */
        _exit(127);
      }
    }

    if (fd == use_fd)
      uv__cloexec(use_fd, 0);
    else {
      dup2(use_fd, fd);
      close(use_fd);
    }

    if (fd <= 2)
      uv__nonblock(fd, 0);
  }

  if (options.cwd && chdir(options.cwd)) {
    uv__write_int(error_fd, -errno);
    /* perror("chdir()"); */
    _exit(127);
  }

  if ((options.flags & UV_PROCESS_SETGID) && setgid(options.gid)) {
    uv__write_int(error_fd, -errno);
    /* perror("setgid()"); */
    _exit(127);
  }

  if ((options.flags & UV_PROCESS_SETUID) && setuid(options.uid)) {
    uv__write_int(error_fd, -errno);
    /* perror("setuid()"); */
    _exit(127);
  }

  if ((options.flags & UV_PROCESS_RESET_SIGPIPE) && signal(SIGPIPE,SIG_DFL) == SIG_ERR)
  {
    uv__write_int(error_fd, errno);
    /* perror("signal()"); */
    _exit(127);
  }

  if (options.env) {
    environ = options.env;
  }

  execvp(options.file, options.args);
  uv__write_int(error_fd, -errno);
  /* perror("execvp()"); */
  _exit(127);
}

#include <signal.h>
int uv_spawn(uv_loop_t* loop,
             uv_process_t* process,
             const uv_process_options_t options) {
  int signal_pipe[2] = { -1, -1 };
  int *pipes;
  int stdio_count;
  QUEUE* q;
  ssize_t r;
  pid_t pid;
  int err;
  int i;

  assert(options.file != NULL);
  assert(!(options.flags & ~(UV_PROCESS_DETACHED |
                             UV_PROCESS_SETGID |
                             UV_PROCESS_SETUID |
                             UV_PROCESS_WINDOWS_HIDE |
                             UV_PROCESS_WINDOWS_VERBATIM_ARGUMENTS |
                             UV_PROCESS_RESET_SIGPIPE)));

  sigset_t sigset, sigoset;
  sigfillset(&sigset);
  sigprocmask(SIG_SETMASK, &sigset, &sigoset);

  uv__handle_init(loop, (uv_handle_t*)process, UV_PROCESS);
  QUEUE_INIT(&process->queue);

  stdio_count = options.stdio_count;
  if (stdio_count < 3)
    stdio_count = 3;

  err = -ENOMEM;
  pipes = malloc(stdio_count * sizeof(*pipes));
  if (pipes == NULL)
    goto error;

  for (i = 0; i < stdio_count; i++) {
    pipes[i] = -1;
  }

  for (i = 0; i < options.stdio_count; i++) {
    err = uv__process_init_stdio(options.stdio + i, &pipes[i]);
    if (err)
      goto error;
  }

  /* This pipe is used by the parent to wait until
   * the child has called `execve()`. We need this
   * to avoid the following race condition:
   *
   *    if ((pid = fork()) > 0) {
   *      kill(pid, SIGTERM);
   *    }
   *    else if (pid == 0) {
   *      execve("/bin/cat", argp, envp);
   *    }
   *
   * The parent sends a signal immediately after forking.
   * Since the child may not have called `execve()` yet,
   * there is no telling what process receives the signal,
   * our fork or /bin/cat.
   *
   * To avoid ambiguity, we create a pipe with both ends
   * marked close-on-exec. Then, after the call to `fork()`,
   * the parent polls the read end until it EOFs or errors with EPIPE.
   */
  err = uv__make_pipe(signal_pipe, 0);
  if (err)
    goto error;

  uv_signal_start(&loop->child_watcher, uv__chld, SIGCHLD);
  
  pid = fork();

  if (pid == -1) {
    err = -errno;
    close(signal_pipe[0]);
    close(signal_pipe[1]);
    goto error;
  }

  if (pid == 0) {
    sigprocmask(SIG_SETMASK, &sigoset, NULL);
    uv__process_child_init(options, stdio_count, pipes, signal_pipe[1]);
    abort();
  }

  close(signal_pipe[1]);

  process->errorno = 0;
  do
    r = read(signal_pipe[0], &process->errorno, sizeof(process->errorno));
  while (r == -1 && errno == EINTR);

  if (r == 0)
    ; /* okay, EOF */
  else if (r == sizeof(process->errorno))
    ; /* okay, read errorno */
  else if (r == -1 && errno == EPIPE)
    ; /* okay, got EPIPE */
  else
    abort();

  close(signal_pipe[0]);

  q = uv__process_queue(loop, pid);
  QUEUE_INSERT_TAIL(q, &process->queue);

  process->pid = pid;
  process->exit_cb = options.exit_cb;
  uv__handle_start(process);

  free(pipes);
  sigprocmask(SIG_SETMASK, &sigoset, NULL);
  return 0;

error:
  for (i = 0; i < stdio_count; i++) {
    if (options.stdio[i].type == UV_STREAM && options.stdio[i].data.stream == NULL)
      close(pipes[i]);
  }
  free(pipes);
  sigprocmask(SIG_SETMASK, &sigoset, NULL);

  return err;
}


int uv_process_kill(uv_process_t* process, int signum) {
  return uv_kill(process->pid, signum);
}


int uv_kill(int pid, int signum) {
  if (kill(pid, signum))
    return -errno;
  else
    return 0;
}


void uv__process_close(uv_process_t* handle) {
  /* TODO stop signal watcher when this is the last handle */
  QUEUE_REMOVE(&handle->queue);
  uv__handle_stop(handle);
}
