/*
 * Copyright (C) 2001-2003 FhG Fokus
 *
 * This file is part of Kamailio, a free SIP server.
 *
 * Kamailio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version
 *
 * Kamailio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

/*! \file
 *  \brief USRLOC - List of registered domains
 *  \ingroup usrloc
 *
 * - Module: \ref usrloc
 */


#include "dlist.h"
#include <stdlib.h> /* abort */
#include <string.h> /* strlen, memcmp */
#include <stdio.h>	/* printf */
#include "../../core/ut.h"
#include "../../lib/srdb1/db_ut.h"
#include "../../core/mem/shm_mem.h"
#include "../../core/dprint.h"
#include "../../core/ip_addr.h"
#include "../../core/socket_info.h"
#include "udomain.h" /* new_udomain, free_udomain */
#include "usrloc.h"
#include "utime.h"
#include "ul_keepalive.h"
#include "usrloc_mod.h"


extern int ul_version_table;
extern int ul_ka_mode;
extern int ul_ka_filter;

/*! \brief Global list of all registered domains */
dlist_t *_ksr_ul_root = 0;

unsigned int _ul_max_partition = 0;

void ul_set_max_partition(unsigned int m)
{
	_ul_max_partition = m;
}

/*!
 * \brief Find domain with the given name
 * \param _n domain name
 * \param _d pointer to domain
 * \return 0 if the domain was found and 1 of not
 */
static inline int find_dlist(str *_n, dlist_t **_d)
{
	dlist_t *ptr;

	ptr = _ksr_ul_root;
	while(ptr) {
		if((_n->len == ptr->name.len) && !memcmp(_n->s, ptr->name.s, _n->len)) {
			*_d = ptr;
			return 0;
		}

		ptr = ptr->next;
	}

	return 1;
}

/*!
 * \brief Keepalive routine for all contacts from the database
 */
int ul_ka_db_records(int partidx)
{
	db1_res_t *res = NULL;
	db_row_t *row = NULL;
	db_key_t keys1[4]; /* where */
	db_val_t vals1[4];
	db_op_t ops1[4];
	db_key_t keys2[9]; /* select */
	int n[2] = {2, 9}; /* number of dynamic values used on key1/key2 */
	dlist_t *dom = NULL;
	urecord_t ur;
	ucontact_t uc;
	int port = 0;
	int proto = 0;
	str host = STR_NULL;
	char *p = NULL;
#define ULKA_AORBUF_SIZE 1024
	char aorbuf[ULKA_AORBUF_SIZE];
	int i = 0;

	/* select fields */
	keys2[0] = &ul_received_col;
	keys2[1] = &ul_contact_col;
	keys2[2] = &ul_sock_col;
	keys2[3] = &ul_cflags_col;
	keys2[4] = &ul_path_col;
	keys2[5] = &ul_ruid_col;
	keys2[6] = &ul_user_col;
	keys2[7] = &ul_domain_col;
	keys2[8] = &ul_con_id_col;

	/* where fields */
	keys1[0] = &ul_expires_col;
	ops1[0] = OP_GT;
	vals1[0].nul = 0;
	UL_DB_EXPIRES_SET(&vals1[0], time(0));

	keys1[1] = &ul_partition_col;
	ops1[1] = OP_EQ;
	vals1[1].type = DB1_INT;
	vals1[1].nul = 0;
	if(_ul_max_partition > 0)
		vals1[1].val.int_val = partidx;
	else
		vals1[1].val.int_val = 0;

	if(ul_ka_mode & ULKA_NAT) {
		keys1[n[0]] = &ul_keepalive_col;
		ops1[n[0]] = OP_EQ;
		vals1[n[0]].type = DB1_INT;
		vals1[n[0]].nul = 0;
		vals1[n[0]].val.int_val = 1;
		n[0]++;
	}
	if(ul_ka_filter & GAU_OPT_SERVER_ID) {
		keys1[n[0]] = &ul_srv_id_col;
		ops1[n[0]] = OP_EQ;
		vals1[n[0]].type = DB1_INT;
		vals1[n[0]].nul = 0;
		vals1[n[0]].val.int_val = server_id;
		n[0]++;
	}

	for(dom = _ksr_ul_root; dom != NULL; dom = dom->next) {
		if(ul_dbf.use_table(ul_dbh, dom->d->name) < 0) {
			LM_ERR("sql use_table failed\n");
			return -1;
		}
		if(ul_dbf.query(
				   ul_dbh, keys1, ops1, vals1, keys2, n[0], n[1], NULL, &res)
				< 0) {
			LM_ERR("query error\n");
			return -1;
		}
		if(RES_ROW_N(res) == 0) {
			ul_dbf.free_result(ul_dbh, res);
			continue;
		}

		for(i = 0; i < RES_ROW_N(res); i++) {
			row = RES_ROWS(res) + i;
			memset(&ur, 0, sizeof(urecord_t));
			memset(&uc, 0, sizeof(ucontact_t));

			/* received */
			uc.received.s = (char *)VAL_STRING(ROW_VALUES(row));
			if(VAL_NULL(ROW_VALUES(row)) || uc.received.s == 0
					|| uc.received.s[0] == 0) {
				uc.received.s = NULL;
				uc.received.len = 0;
			} else {
				uc.received.len = strlen(uc.received.s);
			}

			/* contact */
			uc.c.s = (char *)VAL_STRING(ROW_VALUES(row) + 1);
			if(VAL_NULL(ROW_VALUES(row) + 1) || uc.c.s == 0 || uc.c.s[0] == 0) {
				LM_ERR("empty contact -> skipping\n");
				continue;
			} else {
				uc.c.len = strlen(uc.c.s);
			}

			/* path */
			uc.path.s = (char *)VAL_STRING(ROW_VALUES(row) + 4);
			if(VAL_NULL(ROW_VALUES(row) + 4) || uc.path.s == 0
					|| uc.path.s[0] == 0) {
				uc.path.s = NULL;
				uc.path.len = 0;
			} else {
				uc.path.len = strlen(uc.path.s);
			}

			/* ruid */
			uc.ruid.s = (char *)VAL_STRING(ROW_VALUES(row) + 5);
			if(VAL_NULL(ROW_VALUES(row) + 5) || uc.ruid.s == 0
					|| uc.ruid.s[0] == 0) {
				uc.ruid.s = NULL;
				uc.ruid.len = 0;
			} else {
				uc.ruid.len = strlen(uc.ruid.s);
			}
			/* sock */
			p = (char *)VAL_STRING(ROW_VALUES(row) + 2);
			if(VAL_NULL(ROW_VALUES(row) + 2) || p == 0 || p[0] == 0) {
				uc.sock = 0;
			} else {
				if(parse_phostport(p, &host.s, &host.len, &port, &proto) != 0) {
					LM_ERR("bad socket <%s>...set to 0\n", p);
					uc.sock = 0;
				} else {
					uc.sock =
							grep_sock_info(&host, (unsigned short)port, proto);
					if(uc.sock == 0) {
						LM_DBG("non-local socket <%s>...set to 0\n", p);
					}
				}
			}

			/* flags */
			uc.cflags = VAL_BITMAP(ROW_VALUES(row) + 3);

			/* server id */
			uc.server_id = server_id;

			/* aor from username and domain */
			ur.aor.s = aorbuf;
			ur.domain = &dom->name;

			/* user */
			p = (char *)VAL_STRING(ROW_VALUES(row) + 6);
			if(VAL_NULL(ROW_VALUES(row) + 6) || p == 0 || p[0] == 0) {
				LM_ERR("empty username -> skipping\n");
				continue;
			}
			ur.aor.len = strlen(p);
			if(ur.aor.len >= ULKA_AORBUF_SIZE - 1) {
				LM_DBG("long username ->skipping\n");
				continue;
			}
			strcpy(aorbuf, p);

			/* domain */
			p = (char *)VAL_STRING(ROW_VALUES(row) + 7);
			if(!(VAL_NULL(ROW_VALUES(row) + 7) || p == 0 || p[0] == 0)) {
				if(ur.aor.len + strlen(p) >= ULKA_AORBUF_SIZE - 2) {
					LM_DBG("long aor ->skipping\n");
					continue;
				}
				aorbuf[ur.aor.len] = '@';
				strcpy(aorbuf + ur.aor.len + 1, p);
				ur.aor.len = strlen(aorbuf);
			}
			ur.aorhash = ul_get_aorhash(&ur.aor);
			ur.contacts = &uc;

			/* tcpconn_id */
			uc.tcpconn_id = VAL_INT(ROW_VALUES(row) + 8);

			ul_ka_urecord(&ur);
		} /* row cycle */

		ul_dbf.free_result(ul_dbh, res);
	} /* domain cycle */

	return 0;
}

/*!
 * \brief Get all contacts from the database, in partitions if wanted
 * \see get_all_ucontacts
 * \param buf target buffer
 * \param len length of buffer
 * \param flags contact flags
 * \param part_idx part index
 * \param part_max maximal part
 * \param options options
 * \return 0 on success, positive if buffer size was not sufficient, negative on failure
 */
static inline int get_all_db_ucontacts(void *buf, int len, unsigned int flags,
		unsigned int part_idx, unsigned int part_max, int options)
{
	struct socket_info *sock;
	unsigned int dbflags;
	db1_res_t *res = NULL;
	db_row_t *row;
	dlist_t *dom;
	int port, proto;
	char *p;
	str addr;
	str recv;
	str path;
	str ruid;
	str host;
	unsigned int aorhash;
	int i;
	void *cp;
	int shortage, needed;
	db_key_t keys1[4]; /* where */
	db_val_t vals1[4];
	db_op_t ops1[4];
	db_key_t keys2[6]; /* select */
	int n[2] = {2, 6}; /* number of dynamic values used on key1/key2 */

	cp = buf;
	shortage = 0;
	/* Reserve space for terminating 0000 */
	len -= sizeof(addr.len);

	aorhash = 0;

	/* select fields */
	keys2[0] = &ul_received_col;
	keys2[1] = &ul_contact_col;
	keys2[2] = &ul_sock_col;
	keys2[3] = &ul_cflags_col;
	keys2[4] = &ul_path_col;
	keys2[5] = &ul_ruid_col;

	/* where fields */
	keys1[0] = &ul_expires_col;
	ops1[0] = OP_GT;
	vals1[0].nul = 0;
	UL_DB_EXPIRES_SET(&vals1[0], time(0));

	keys1[1] = &ul_partition_col;
	ops1[1] = OP_EQ;
	vals1[1].type = DB1_INT;
	vals1[1].nul = 0;
	if(_ul_max_partition > 0)
		vals1[1].val.int_val = part_idx;
	else
		vals1[1].val.int_val = 0;

	if(flags & ul_nat_bflag) {
		keys1[n[0]] = &ul_keepalive_col;
		ops1[n[0]] = OP_EQ;
		vals1[n[0]].type = DB1_INT;
		vals1[n[0]].nul = 0;
		vals1[n[0]].val.int_val = 1;
		n[0]++;
	}
	if(options & GAU_OPT_SERVER_ID) {
		keys1[n[0]] = &ul_srv_id_col;
		ops1[n[0]] = OP_EQ;
		vals1[n[0]].type = DB1_INT;
		vals1[n[0]].nul = 0;
		vals1[n[0]].val.int_val = server_id;
		n[0]++;
	}

	/* TODO: use part_idx and part_max on keys1 */

	for(dom = _ksr_ul_root; dom != NULL; dom = dom->next) {
		if(ul_dbf.use_table(ul_dbh, dom->d->name) < 0) {
			LM_ERR("sql use_table failed\n");
			return -1;
		}
		if(ul_dbf.query(
				   ul_dbh, keys1, ops1, vals1, keys2, n[0], n[1], NULL, &res)
				< 0) {
			LM_ERR("query error\n");
			return -1;
		}
		if(RES_ROW_N(res) == 0) {
			ul_dbf.free_result(ul_dbh, res);
			continue;
		}

		for(i = 0; i < RES_ROW_N(res); i++) {
			row = RES_ROWS(res) + i;


			/* received */
			recv.s = (char *)VAL_STRING(ROW_VALUES(row));
			if(VAL_NULL(ROW_VALUES(row)) || recv.s == 0 || recv.s[0] == 0) {
				recv.s = NULL;
				recv.len = 0;
			} else {
				recv.len = strlen(recv.s);
			}

			/* contact */
			addr.s = (char *)VAL_STRING(ROW_VALUES(row) + 1);
			if(VAL_NULL(ROW_VALUES(row) + 1) || addr.s == 0 || addr.s[0] == 0) {
				LM_ERR("empty contact -> skipping\n");
				continue;
			} else {
				addr.len = strlen(addr.s);
			}

			/* path */
			path.s = (char *)VAL_STRING(ROW_VALUES(row) + 4);
			if(VAL_NULL(ROW_VALUES(row) + 4) || path.s == 0 || path.s[0] == 0) {
				path.s = NULL;
				path.len = 0;
			} else {
				path.len = strlen(path.s);
			}

			/* ruid */
			ruid.s = (char *)VAL_STRING(ROW_VALUES(row) + 5);
			if(VAL_NULL(ROW_VALUES(row) + 5) || ruid.s == 0 || ruid.s[0] == 0) {
				ruid.s = NULL;
				ruid.len = 0;
			} else {
				ruid.len = strlen(ruid.s);
			}

			needed = (int)(sizeof(addr.len) + addr.len + sizeof(recv.len)
						   + recv.len + sizeof(sock) + sizeof(dbflags)
						   + sizeof(path.len) + path.len + sizeof(ruid.len)
						   + ruid.len + sizeof(aorhash));
			if(len < needed) {
				shortage += needed;
				continue;
			}

			/* write contact */
			memcpy(cp, &addr.len, sizeof(addr.len));
			cp = (char *)cp + sizeof(addr.len);
			memcpy(cp, addr.s, addr.len);
			cp = (char *)cp + addr.len;

			/* write received */
			memcpy(cp, &recv.len, sizeof(recv.len));
			cp = (char *)cp + sizeof(recv.len);
			/* copy received only if exist */
			if(recv.len) {
				memcpy(cp, recv.s, recv.len);
				cp = (char *)cp + recv.len;
			}

			/* sock */
			p = (char *)VAL_STRING(ROW_VALUES(row) + 2);
			if(VAL_NULL(ROW_VALUES(row) + 2) || p == 0 || p[0] == 0) {
				sock = 0;
			} else {
				if(parse_phostport(p, &host.s, &host.len, &port, &proto) != 0) {
					LM_ERR("bad socket <%s>...set to 0\n", p);
					sock = 0;
				} else {
					sock = grep_sock_info(&host, (unsigned short)port, proto);
					if(sock == 0) {
						LM_DBG("non-local socket <%s>...set to 0\n", p);
					}
				}
			}

			/* flags */
			dbflags = VAL_BITMAP(ROW_VALUES(row) + 3);

			/* write sock and flags */
			memcpy(cp, &sock, sizeof(sock));
			cp = (char *)cp + sizeof(sock);
			memcpy(cp, &dbflags, sizeof(dbflags));
			cp = (char *)cp + sizeof(dbflags);

			/* write path */
			memcpy(cp, &path.len, sizeof(path.len));
			cp = (char *)cp + sizeof(path.len);
			/* copy path only if exist */
			if(path.len) {
				memcpy(cp, path.s, path.len);
				cp = (char *)cp + path.len;
			}

			/* write ruid */
			memcpy(cp, &ruid.len, sizeof(ruid.len));
			cp = (char *)cp + sizeof(ruid.len);
			/* copy ruid only if exist */
			if(ruid.len) {
				memcpy(cp, ruid.s, ruid.len);
				cp = (char *)cp + ruid.len;
			}
			/* aorhash not used for db-only records, but it is added
			 * (as 0) to match the struct used for mem records */
			memcpy(cp, &aorhash, sizeof(aorhash));
			cp = (char *)cp + sizeof(aorhash);

			len -= needed;
		} /* row cycle */

		ul_dbf.free_result(ul_dbh, res);
	} /* domain cycle */

	/* len < 0 is possible, if size of the buffer < sizeof(c->c.len) */
	if(len >= 0)
		memset(cp, 0, sizeof(addr.len));

	/* Shouldn't happen */
	if(shortage > 0 && len > shortage) {
		abort();
	}

	shortage -= len;

	return shortage > 0 ? shortage : 0;
}


/*!
 * \brief Get all contacts from the memory, in partitions if wanted
 * \see get_all_ucontacts
 * \param buf target buffer
 * \param len length of buffer
 * \param flags contact flags
 * \param part_idx part index
 * \param part_max maximal part
 * \param options options
 * \return 0 on success, positive if buffer size was not sufficient, negative on failure
 */
static inline int get_all_mem_ucontacts(void *buf, int len, unsigned int flags,
		unsigned int part_idx, unsigned int part_max, int options)
{
	dlist_t *p;
	urecord_t *r;
	ucontact_t *c;
	void *cp;
	int shortage;
	int needed;
	int i = 0;
	cp = buf;
	shortage = 0;
	time_t tnow = 0;


	if(ul_keepalive_timeout > 0)
		tnow = time(NULL);

	/* Reserve space for terminating 0000 */
	len -= sizeof(c->c.len);

	for(p = _ksr_ul_root; p != NULL; p = p->next) {

		for(i = 0; i < p->d->size; i++) {

			if((i % part_max) != part_idx)
				continue;

			lock_ulslot(p->d, i);
			if(p->d->table[i].n <= 0) {
				unlock_ulslot(p->d, i);
				continue;
			}
			for(r = p->d->table[i].first; r != NULL; r = r->next) {
				for(c = r->contacts; c != NULL; c = c->next) {
					if(c->c.len <= 0)
						continue;
					/*
					 * List only contacts that have all requested
					 * flags set
					 */
					if((c->cflags & flags) != flags)
						continue;

					if(options & GAU_OPT_SERVER_ID && server_id != c->server_id)
						continue;

					if(ul_keepalive_timeout > 0 && c->last_keepalive > 0) {
						if(c->sock != NULL && c->sock->proto == PROTO_UDP) {
							if(c->last_keepalive + ul_keepalive_timeout
									< tnow) {
								/* set contact as expired in 10s */
								if(c->expires > tnow + 10)
									c->expires = tnow + 10;
								continue;
							}
						}
					}

					needed = (int)(sizeof(c->c.len) + c->c.len
								   + sizeof(c->received.len) + c->received.len
								   + sizeof(c->sock) + sizeof(c->cflags)
								   + sizeof(c->path.len) + c->path.len
								   + sizeof(c->ruid.len) + c->ruid.len
								   + sizeof(r->aorhash));
					if(len >= needed) {
						memcpy(cp, &c->c.len, sizeof(c->c.len));
						cp = (char *)cp + sizeof(c->c.len);
						memcpy(cp, c->c.s, c->c.len);
						cp = (char *)cp + c->c.len;
						memcpy(cp, &c->received.len, sizeof(c->received.len));
						cp = (char *)cp + sizeof(c->received.len);
						memcpy(cp, c->received.s, c->received.len);
						cp = (char *)cp + c->received.len;
						memcpy(cp, &c->sock, sizeof(c->sock));
						cp = (char *)cp + sizeof(c->sock);
						memcpy(cp, &c->cflags, sizeof(c->cflags));
						cp = (char *)cp + sizeof(c->cflags);
						memcpy(cp, &c->path.len, sizeof(c->path.len));
						cp = (char *)cp + sizeof(c->path.len);
						memcpy(cp, c->path.s, c->path.len);
						cp = (char *)cp + c->path.len;
						memcpy(cp, &c->ruid.len, sizeof(c->ruid.len));
						cp = (char *)cp + sizeof(c->ruid.len);
						memcpy(cp, c->ruid.s, c->ruid.len);
						cp = (char *)cp + c->ruid.len;
						memcpy(cp, &r->aorhash, sizeof(r->aorhash));
						cp = (char *)cp + sizeof(r->aorhash);
						len -= needed;
					} else {
						shortage += needed;
					}
				}
			}
			unlock_ulslot(p->d, i);
		}
	}
	/* len < 0 is possible, if size of the buffer < sizeof(c->c.len) */
	if(len >= 0)
		memset(cp, 0, sizeof(c->c.len));

	/* Shouldn't happen */
	if(shortage > 0 && len > shortage) {
		abort();
	}

	shortage -= len;

	return shortage > 0 ? shortage : 0;
}


/*!
 * \brief Get all contacts from the usrloc, in partitions if wanted
 *
 * Return list of all contacts for all currently registered
 * users in all domains. The caller must provide buffer of
 * sufficient length for fitting all those contacts. In the
 * case when buffer was exhausted, the function returns
 * estimated amount of additional space needed, in this
 * case the caller is expected to repeat the call using
 * this value as the hint.
 *
 * Information is packed into the buffer as follows:
 *
 * +------------+----------+-----+------+-----+
 * |contact1.len|contact1.s|sock1|flags1|path1|
 * +------------+----------+-----+------+-----+
 * |contact2.len|contact2.s|sock2|flags2|path1|
 * +------------+----------+-----+------+-----+
 * |..........................................|
 * +------------+----------+-----+------+-----+
 * |contactN.len|contactN.s|sockN|flagsN|pathN|
 * +------------+----------+-----+------+-----+
 * |000000000000|
 * +------------+
 *
 * \param buf target buffer
 * \param len length of buffer
 * \param flags contact flags
 * \param part_idx part index
 * \param part_max maximal part
 * \param options options
 * \return 0 on success, positive if buffer size was not sufficient, negative on failure
 */
int get_all_ucontacts(void *buf, int len, unsigned int flags,
		unsigned int part_idx, unsigned int part_max, int options)
{
	if(ul_db_mode == DB_ONLY)
		return get_all_db_ucontacts(
				buf, len, flags, part_idx, part_max, options);
	else
		return get_all_mem_ucontacts(
				buf, len, flags, part_idx, part_max, options);
}


/**
 *
 */
int ul_update_keepalive(
		unsigned int _aorhash, str *_ruid, time_t tval, unsigned int rtrip)
{
	dlist_t *p;
	urecord_t *r;
	ucontact_t *c;
	int i;

	/* todo: get location domain via param */

	for(p = _ksr_ul_root; p != NULL; p = p->next) {
		i = _aorhash & (p->d->size - 1);
		lock_ulslot(p->d, i);
		if(p->d->table[i].n <= 0) {
			unlock_ulslot(p->d, i);
			continue;
		}
		for(r = p->d->table[i].first; r != NULL; r = r->next) {
			if(r->aorhash == _aorhash) {
				for(c = r->contacts; c != NULL; c = c->next) {
					if(c->c.len <= 0 || c->ruid.len <= 0)
						continue;
					if(c->ruid.len == _ruid->len
							&& !memcmp(c->ruid.s, _ruid->s, _ruid->len)) {
						/* found */
						c->last_keepalive = tval;
						c->ka_roundtrip = rtrip;
						LM_DBG("updated keepalive for [%.*s:%u] to %u (rtrip: "
							   "%u)\n",
								_ruid->len, _ruid->s, _aorhash,
								(unsigned int)c->last_keepalive,
								c->ka_roundtrip);
						unlock_ulslot(p->d, i);
						return 0;
					}
				}
			}
		}
		unlock_ulslot(p->d, i);
	}

	return 0;
}

/**
 *
 */
int ul_refresh_keepalive(unsigned int _aorhash, str *_ruid)
{
	return ul_update_keepalive(_aorhash, _ruid, time(NULL), 0);
}

/*!
 * \brief Create a new domain structure
 * \return 0 if everything went OK, otherwise value < 0 is returned
 *
 * \note The structure is NOT created in shared memory so the
 * function must be called before the server forks if it should
 * be available to all processes
 */
static inline int new_dlist(str *_n, dlist_t **_d)
{
	dlist_t *ptr;

	/* Domains are created before ser forks,
	 * so we can create them using pkg_malloc
	 */
	ptr = (dlist_t *)shm_malloc(sizeof(dlist_t));
	if(ptr == 0) {
		SHM_MEM_ERROR;
		return -1;
	}
	memset(ptr, 0, sizeof(dlist_t));

	/* copy domain name as null terminated string */
	ptr->name.s = (char *)shm_malloc(_n->len + 1);
	if(ptr->name.s == 0) {
		SHM_MEM_ERROR;
		shm_free(ptr);
		return -2;
	}

	memcpy(ptr->name.s, _n->s, _n->len);
	ptr->name.len = _n->len;
	ptr->name.s[ptr->name.len] = 0;

	if(new_udomain(&(ptr->name), ul_hash_size, &(ptr->d)) < 0) {
		LM_ERR("creating domain structure failed\n");
		shm_free(ptr->name.s);
		shm_free(ptr);
		return -3;
	}

	*_d = ptr;
	return 0;
}

/*!
 * \brief Registers a new domain with usrloc
 *
 * Find and return a usrloc domain (location table)
 * \param _n domain name
 * \param _d usrloc domain
 * \return 0 on success, -1 on failure
 */
int get_udomain(const char *_n, udomain_t **_d)
{
	dlist_t *d;
	str s;

	if(_n == NULL) {
		LM_ERR("null location table name\n");
		goto notfound;
	}

	s.s = (char *)_n;
	s.len = strlen(_n);
	if(s.len <= 0) {
		LM_ERR("empty location table name\n");
		goto notfound;
	}

	if(find_dlist(&s, &d) == 0) {
		*_d = d->d;
		return 0;
	}

notfound:
	*_d = NULL;
	return -1;
}

/*!
 * \brief Registers a new domain with usrloc
 *
 * Registers a new domain with usrloc. If the domain exists,
 * a pointer to existing structure will be returned, otherwise
 * a new domain will be created
 * \param _n domain name
 * \param _d new created domain
 * \return 0 on success, -1 on failure
 */
int register_udomain(const char *_n, udomain_t **_d)
{
	dlist_t *d;
	str s;
	db1_con_t *con;

	s.s = (char *)_n;
	s.len = strlen(_n);

	if(find_dlist(&s, &d) == 0) {
		*_d = d->d;
		return 0;
	}

	if(new_dlist(&s, &d) < 0) {
		LM_ERR("failed to create new domain\n");
		return -1;
	}

	/* Test tables from database if we are gonna
	 * to use database
	 */
	if(ul_db_mode != NO_DB) {
		con = ul_dbf.init(&ul_db_url);
		if(!con) {
			LM_ERR("failed to open database connection\n");
			goto dberror;
		}

		if(ul_version_table != 0
				&& db_check_table_version(&ul_dbf, con, &s, UL_TABLE_VERSION)
						   < 0) {
			DB_TABLE_VERSION_ERROR(s);
			goto dberror;
		}
		/* test if DB really exists */
		if(testdb_udomain(con, d->d) < 0) {
			LM_ERR("testing domain '%.*s' failed\n", s.len, ZSW(s.s));
			goto dberror;
		}

		ul_dbf.close(con);
		con = 0;
	}

	d->next = _ksr_ul_root;
	_ksr_ul_root = d;

	*_d = d->d;
	return 0;

dberror:
	if(con)
		ul_dbf.close(con);
	con = 0;
	free_udomain(d->d);
	shm_free(d->name.s);
	shm_free(d);
	return -1;
}


/*!
 * \brief Free all allocated memory for domains
 */
void free_all_udomains(void)
{
	dlist_t *ptr;

	while(_ksr_ul_root) {
		ptr = _ksr_ul_root;
		_ksr_ul_root = _ksr_ul_root->next;

		free_udomain(ptr->d);
		shm_free(ptr->name.s);
		shm_free(ptr);
	}
}


/*!
 * \brief Print all domains, just for debugging
 * \param _f output file
 */
void print_all_udomains(FILE *_f)
{
	dlist_t *ptr;

	ptr = _ksr_ul_root;

	fprintf(_f, "===Domain list===\n");
	while(ptr) {
		print_udomain(_f, ptr->d);
		ptr = ptr->next;
	}
	fprintf(_f, "===/Domain list===\n");
}


/*!
 * \brief Loops through all domains summing up the number of users
 * \return the number of users, could be zero
 */
unsigned long get_number_of_users(void)
{
	long numberOfUsers = 0;

	dlist_t *current_dlist;

	current_dlist = _ksr_ul_root;

	while(current_dlist) {
		numberOfUsers += get_stat_val(current_dlist->d->users);
		current_dlist = current_dlist->next;
	}

	return numberOfUsers;
}


/*!
 * \brief Run timer handler of all domains
 * \return 0 if all timer return 0, != 0 otherwise
 */
int synchronize_all_udomains(int istart, int istep)
{
	int res = 0;
	dlist_t *ptr;

	ul_get_act_time(); /* Get and save actual time */

	if(ul_db_mode == DB_ONLY) {
		if(istart == 0) {
			for(ptr = _ksr_ul_root; ptr; ptr = ptr->next) {
				res |= db_timer_udomain(ptr->d);
			}
		}
		if(ul_ka_mode != ULKA_NONE) {
			ul_ka_db_records((unsigned int)istart);
		}
	} else {
		for(ptr = _ksr_ul_root; ptr; ptr = ptr->next) {
			mem_timer_udomain(ptr->d, istart, istep);
		}
	}

	return res;
}

/*!
 * \brief Run timer handler to clean all domains in db
 * \return 0 if all timer return 0, != 0 otherwise
 */
int ul_db_clean_udomains(void)
{
	int res = 0;
	dlist_t *ptr;

	ul_get_act_time(); /* Get and save actual time */

	for(ptr = _ksr_ul_root; ptr; ptr = ptr->next)
		res |= db_timer_udomain(ptr->d);

	return res;
}

/*!
 * \brief Find a particular domain, small wrapper around find_dlist
 * \param _d domain name
 * \param _p pointer to domain if found
 * \return 1 if domain was found, 0 otherwise
 */
int find_domain(str *_d, udomain_t **_p)
{
	dlist_t *d;

	if(find_dlist(_d, &d) == 0) {
		*_p = d->d;
		return 0;
	}

	return 1;
}
