/*
 *  Copyright 1994-2013 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "oscillo.h"


u_long id = 946482112;
u_long options = BE_SFX2D;
u_long mode = OVERLAY;
char desc[] = "Oscilloscope based on a sine-wave";


static Porteuse_t *P = NULL;

const float sin1_phi_inc = 0.0101667; /* do not ask me why --oliv3 */
const float sin1_freq_min = 2, sin1_freq_max = 10;

static float sin1_phi = 0;
static float sin1_freq;        /* initialized on create */
static float sin1_target_freq; /* initialized on create */
static float sin1_freq_inc;    /* initialized on create */
static int   connect = 1;


static inline float
rnd_freq()
{
  /* drand48() a l'arrache --oliv3 */
  return drand48() * (sin1_freq_max - sin1_freq_min) + sin1_freq_min;
}


static void
change_params()
{
  if (sin1_freq_inc > 0) {
    sin1_freq += sin1_freq_inc;
    if (sin1_freq > sin1_target_freq) {
      float new_freq = rnd_freq();
      while (new_freq >= sin1_freq)
	new_freq = rnd_freq();
      sin1_freq_inc = -(drand48() / 10 + .01);
      sin1_target_freq = new_freq;
    }
  } else {
    sin1_freq += sin1_freq_inc;
    if (sin1_freq < sin1_target_freq) {
      float new_freq = rnd_freq();
      while (new_freq <= sin1_freq)
	new_freq = rnd_freq();
      sin1_freq_inc = drand48() / 10 + .01;
      sin1_target_freq = new_freq;
    }
  }
  sin1_phi += sin1_phi_inc;
}


static void
init(Context_t *ctx)
{
  int i;
  Transform_t t;

  /* BOF keep gcc happy about uninitialized variables */
  memset(&t, 0, sizeof(t));

  P->origin.x = 1;
  P->origin.y = HMAXY;

  t.v_j_factor = HMAXY / 2 * .85;
  
  for (i = 0; i < P->size-1; i++) {
    float ya = HMAXY + t.v_j_factor * sin(sin1_freq * (float)(i) / (float)(ctx->input->size - 1) + sin1_phi);
    float yb = HMAXY + t.v_j_factor * sin(sin1_freq * (float)(i+1) / (float)(ctx->input->size - 1) + sin1_phi);
    
    t.v_i.x = 1.0 / (float)(ctx->input->size - 2) * MAXX;
    t.v_i.y = yb - ya;
    
    P->trans[i] = t;
  }
  P->trans[i] = t;

  Porteuse_init_alpha(P);
}


void
create(Context_t *ctx)
{
  if (ctx->input == NULL)
    options |= BEQ_DISABLED;
  else {
    P = Porteuse_new(ctx->input->size, A_MONO);
    sin1_freq = sin1_freq_min;
    sin1_target_freq = rnd_freq();
    sin1_freq_inc = drand48() / 10 + .01;
    init(ctx);
  }
}


void
destroy(__attribute__ ((unused)) Context_t *ctx)
{
  if (P != NULL)
    Porteuse_delete(P);
}


void
on_switch_on(__attribute__ ((unused)) Context_t *ctx)
{
  /* connect = b_rand_boolean(); */
}


void
run(Context_t *ctx)
{
  if (P == NULL)
    return;

  Buffer8_clear(passive_buffer(ctx));
  Porteuse_draw(P, ctx, connect);
  change_params();
  init(ctx);
}
