package tests::ReportEntryTest;

use strict;

use base qw/Lire::Test::TestCase/;

use Lire::Report::Group;
use Lire::Report::Entry;
use Lire::Report::TableInfo;

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->{'info'} = new Lire::Report::TableInfo();
    $self->{'info'}->create_column_info( 'value', 'numerical', 'int' );
    $self->{'info'}->create_group_info( 'group' );

    $self->{'group'} =
      new Lire::Report::Group( bless( {}, 'Lire::Report::Entry' ),
                               $self->{'info'} );
    return $self;
}

sub tear_down {
    my $self = shift->SUPER::tear_down();

    return $self;
}

sub test_new {
    my $self = $_[0];

    my $group = $self->{'group'};
    $self->assert_died( sub { new Lire::Report::Entry() },
                        qr/missing \'group\' parameter/ );

    $self->assert_died( sub { new Lire::Report::Entry( {} ) },
                        qr/\'group\' parameter should be a 'Lire::Report::Group' instance, not 'HASH/ );

    my $entry = new Lire::Report::Entry( $group );
    $self->assert_not_null( $entry, 'new() returned undef' );
    $self->assert( UNIVERSAL::isa( $entry, 'Lire::Report::Entry' ),
                   "new() should return instance of Lire::Report::Entry: $entry" );
    $self->assert_deep_equals( [], $entry->{'data'} );
    $self->assert_equals( $group, $entry->{'group'} );
}

sub test_add_value {
    my $self = $_[0];

    my $entry = new Lire::Report::Entry( $self->{'group'} );
    $entry->add_value( { 'content' => '10.0',
                         'value'   => '10.0',
                         'n'       => '5',
                         'total'   => '50',
                         'missing_cases' => 10,
                       } );
    $self->assert_deep_equals( [ { 'content' => '10.0',
                                   'value'   => '10.0',
                                   'n'       => '5',
                                   'total'   => '50',
                                   'type'    => 'value',
                                   'col_info' => $self->{'info'}->info_by_index(0),
                                   'missing_cases' => 10, } ],
                               $entry->{'data'} );

    $self->assert_died( sub { $entry->add_value( {} ) },
                        qr/\'Data_1\' parameter should be a \'Lire::Report::ColumnInfo\' instance, not \'Lire::Report::GroupInfo/ );

    my $entry2 = new Lire::Report::Entry( $self->{'group'} );
    $entry2->add_value( { 'content' => '10.0' } );
    $self->assert_deep_equals( [ { 'content' => '10.0',
                                   'value'   => '10.0',
                                   'n'       => undef,
                                   'total'   => undef,
                                   'type'    => 'value',
                                   'col_info' => $self->{'info'}->info_by_index(0),
                                   'missing_cases' => 0, } ],
                               $entry2->{'data'} );

    my $entry3 = new Lire::Report::Entry( $self->{'group'} );
    $entry3->add_value( {} );
    $self->assert_deep_equals( [ { 'content' => '',
                                   'value'   => '',
                                   'n'       => undef,
                                   'total'   => undef,
                                   'type'    => 'value',
                                   'col_info' => $self->{'info'}->info_by_index(0),
                                   'missing_cases' => 0, } ],
                               $entry3->{'data'} );
}

sub test_add_value_old_signature {
    my $self = $_[0];

    my $entry = new Lire::Report::Entry( $self->{'group'} );
    $entry->add_value( 'content' => 1 );
    $self->assert_deep_equals( [ { 'content' => 1,
                                   'value'   => 1,
                                   'n'       => undef,
                                   'total'   => undef,
                                   'type'    => 'value',
                                   'col_info' => $self->{'info'}->info_by_index(0),
                                   'missing_cases' => 0, } ],
                               $entry->{'data'} );
}

1;
