/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.lucene.tests.analysis.standard;

import static org.apache.lucene.tests.analysis.BaseTokenStreamTestCase.assertAnalyzesTo;

import org.apache.lucene.analysis.Analyzer;

/**
 * This class was automatically generated by generateJavaUnicodeWordBreakTest.pl from:
 * http://www.unicode.org/Public/12.1.0/ucd/auxiliary/WordBreakTest.txt
 *
 * <p>WordBreakTest.txt indicates the points in the provided character sequences at which conforming
 * implementations must and must not break words. This class tests for expected token extraction
 * from each of the test sequences in WordBreakTest.txt, where the expected tokens are those
 * character sequences bounded by word breaks and containing at least one character from one of the
 * following character sets:
 *
 * <p>\p{Script = Han} (From http://www.unicode.org/Public/12.1.0/ucd/Scripts.txt) \p{Script =
 * Hiragana} \p{LineBreak = Complex_Context} (From
 * http://www.unicode.org/Public/12.1.0/ucd/LineBreak.txt) \p{WordBreak = ALetter} (From
 * http://www.unicode.org/Public/12.1.0/ucd/auxiliary/WordBreakProperty.txt) \p{WordBreak =
 * Hebrew_Letter} \p{WordBreak = Katakana} \p{WordBreak = Numeric} \p{Extended_Pictographic} (From
 * http://www.unicode.org/Public/emoji/12.1/emoji-data.txt)
 */
public final class WordBreakTestUnicode_12_1_0 {

  public static void test(Analyzer analyzer) throws Exception {
    // ÷ 0001 ÷ 0001 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0001", new String[] {});

    // ÷ 0001 × 0308 ÷ 0001 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0001", new String[] {});

    // ÷ 0001 ÷ 000D ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\r", new String[] {});

    // ÷ 0001 × 0308 ÷ 000D ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\r", new String[] {});

    // ÷ 0001 ÷ 000A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\n", new String[] {});

    // ÷ 0001 × 0308 ÷ 000A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\n", new String[] {});

    // ÷ 0001 ÷ 000B ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u000B", new String[] {});

    // ÷ 0001 × 0308 ÷ 000B ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u000B", new String[] {});

    // ÷ 0001 ÷ 3031 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u3031", new String[] {"\u3031"});

    // ÷ 0001 × 0308 ÷ 3031 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u3031", new String[] {"\u3031"});

    // ÷ 0001 ÷ 0041 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0041", new String[] {"\u0041"});

    // ÷ 0001 × 0308 ÷ 0041 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0041", new String[] {"\u0041"});

    // ÷ 0001 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u003A", new String[] {});

    // ÷ 0001 × 0308 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u003A", new String[] {});

    // ÷ 0001 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u002C", new String[] {});

    // ÷ 0001 × 0308 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u002C", new String[] {});

    // ÷ 0001 ÷ 002E ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u002E", new String[] {});

    // ÷ 0001 × 0308 ÷ 002E ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u002E", new String[] {});

    // ÷ 0001 ÷ 0030 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0030", new String[] {"\u0030"});

    // ÷ 0001 × 0308 ÷ 0030 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0030", new String[] {"\u0030"});

    // ÷ 0001 ÷ 005F ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LOW LINE (ExtendNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u005F", new String[] {});

    // ÷ 0001 × 0308 ÷ 005F ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u005F", new String[] {});

    // ÷ 0001 ÷ 1F1E6 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\uD83C\uDDE6", new String[] {});

    // ÷ 0001 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 0001 ÷ 05D0 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u05D0", new String[] {"\u05D0"});

    // ÷ 0001 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 0001 ÷ 0022 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\"", new String[] {});

    // ÷ 0001 × 0308 ÷ 0022 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\"", new String[] {});

    // ÷ 0001 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0027", new String[] {});

    // ÷ 0001 × 0308 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0027", new String[] {});

    // ÷ 0001 ÷ 231A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u231A", new String[] {"\u231A"});

    // ÷ 0001 × 0308 ÷ 231A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u231A", new String[] {"\u231A"});

    // ÷ 0001 ÷ 0020 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0020", new String[] {});

    // ÷ 0001 × 0308 ÷ 0020 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0020", new String[] {});

    // ÷ 0001 × 00AD ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] SOFT HYPHEN (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u00AD", new String[] {});

    // ÷ 0001 × 0308 × 00AD ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u00AD", new String[] {});

    // ÷ 0001 × 0300 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0300", new String[] {});

    // ÷ 0001 × 0308 × 0300 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0300", new String[] {});

    // ÷ 0001 × 200D ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u200D", new String[] {});

    // ÷ 0001 × 0308 × 200D ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u200D", new String[] {});

    // ÷ 0001 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0001 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0001 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0001 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0001 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0001 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0001 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0001 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0001 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0001 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0001 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0001 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0001 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0001 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0001 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0001 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0001 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0001 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <START OF HEADING> (Other) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0001\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 000D ÷ 0001 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u0001", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 0001 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0001", new String[] {});

    // ÷ 000D ÷ 000D ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\r", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 000D ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\r", new String[] {});

    // ÷ 000D × 000A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) × [3.0] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\n", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 000A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\n", new String[] {});

    // ÷ 000D ÷ 000B ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u000B", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 000B ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u000B", new String[] {});

    // ÷ 000D ÷ 3031 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u3031", new String[] {"\u3031"});

    // ÷ 000D ÷ 0308 ÷ 3031 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u3031", new String[] {"\u3031"});

    // ÷ 000D ÷ 0041 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0041", new String[] {"\u0041"});

    // ÷ 000D ÷ 0308 ÷ 0041 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0041", new String[] {"\u0041"});

    // ÷ 000D ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u003A", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u003A", new String[] {});

    // ÷ 000D ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u002C", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u002C", new String[] {});

    // ÷ 000D ÷ 002E ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u002E", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 002E ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u002E", new String[] {});

    // ÷ 000D ÷ 0030 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0030", new String[] {"\u0030"});

    // ÷ 000D ÷ 0308 ÷ 0030 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0030", new String[] {"\u0030"});

    // ÷ 000D ÷ 005F ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LOW LINE (ExtendNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u005F", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 005F ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u005F", new String[] {});

    // ÷ 000D ÷ 1F1E6 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\uD83C\uDDE6", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 000D ÷ 05D0 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u05D0", new String[] {"\u05D0"});

    // ÷ 000D ÷ 0308 ÷ 05D0 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 000D ÷ 0022 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] QUOTATION MARK (Double_Quote)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\"", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 0022 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\"", new String[] {});

    // ÷ 000D ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u0027", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0027", new String[] {});

    // ÷ 000D ÷ 231A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u231A", new String[] {"\u231A"});

    // ÷ 000D ÷ 0308 ÷ 231A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u231A", new String[] {"\u231A"});

    // ÷ 000D ÷ 0020 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0020", new String[] {});

    // ÷ 000D ÷ 0308 ÷ 0020 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0020", new String[] {});

    // ÷ 000D ÷ 00AD ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] SOFT HYPHEN (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u00AD", new String[] {});

    // ÷ 000D ÷ 0308 × 00AD ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u00AD", new String[] {});

    // ÷ 000D ÷ 0300 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0300", new String[] {});

    // ÷ 000D ÷ 0308 × 0300 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0300", new String[] {});

    // ÷ 000D ÷ 200D ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u200D", new String[] {});

    // ÷ 000D ÷ 0308 × 200D ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u200D", new String[] {});

    // ÷ 000D ÷ 0061 × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 000D ÷ 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 000D ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u003A", new String[] {"\u0061"});

    // ÷ 000D ÷ 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 000D ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u0027", new String[] {"\u0061"});

    // ÷ 000D ÷ 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 000D ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 000D ÷ 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 000D ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0061\u002C", new String[] {"\u0061"});

    // ÷ 000D ÷ 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 000D ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0031\u003A", new String[] {"\u0031"});

    // ÷ 000D ÷ 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 000D ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0031\u0027", new String[] {"\u0031"});

    // ÷ 000D ÷ 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 000D ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0031\u002C", new String[] {"\u0031"});

    // ÷ 000D ÷ 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 000D ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 000D ÷ 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) ÷ [3.1]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 000A ÷ 0001 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0001", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 0001 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0001", new String[] {});

    // ÷ 000A ÷ 000D ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\r", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 000D ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\r", new String[] {});

    // ÷ 000A ÷ 000A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\n", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 000A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\n", new String[] {});

    // ÷ 000A ÷ 000B ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u000B", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 000B ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u000B", new String[] {});

    // ÷ 000A ÷ 3031 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u3031", new String[] {"\u3031"});

    // ÷ 000A ÷ 0308 ÷ 3031 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u3031", new String[] {"\u3031"});

    // ÷ 000A ÷ 0041 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN CAPITAL LETTER A (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\n\u0041", new String[] {"\u0041"});

    // ÷ 000A ÷ 0308 ÷ 0041 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0041", new String[] {"\u0041"});

    // ÷ 000A ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u003A", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u003A", new String[] {});

    // ÷ 000A ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u002C", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u002C", new String[] {});

    // ÷ 000A ÷ 002E ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u002E", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 002E ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u002E", new String[] {});

    // ÷ 000A ÷ 0030 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0030", new String[] {"\u0030"});

    // ÷ 000A ÷ 0308 ÷ 0030 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0030", new String[] {"\u0030"});

    // ÷ 000A ÷ 005F ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u005F", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 005F ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u005F", new String[] {});

    // ÷ 000A ÷ 1F1E6 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] REGIONAL INDICATOR SYMBOL LETTER A
    // (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\uD83C\uDDE6", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 000A ÷ 05D0 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] HEBREW LETTER ALEF (Hebrew_Letter)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u05D0", new String[] {"\u05D0"});

    // ÷ 000A ÷ 0308 ÷ 05D0 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 000A ÷ 0022 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\n\"", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 0022 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\"", new String[] {});

    // ÷ 000A ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0027", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0027", new String[] {});

    // ÷ 000A ÷ 231A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u231A", new String[] {"\u231A"});

    // ÷ 000A ÷ 0308 ÷ 231A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u231A", new String[] {"\u231A"});

    // ÷ 000A ÷ 0020 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0020", new String[] {});

    // ÷ 000A ÷ 0308 ÷ 0020 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0020", new String[] {});

    // ÷ 000A ÷ 00AD ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u00AD", new String[] {});

    // ÷ 000A ÷ 0308 × 00AD ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u00AD", new String[] {});

    // ÷ 000A ÷ 0300 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING GRAVE ACCENT (Extend_FE)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0300", new String[] {});

    // ÷ 000A ÷ 0308 × 0300 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0300", new String[] {});

    // ÷ 000A ÷ 200D ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u200D", new String[] {});

    // ÷ 000A ÷ 0308 × 200D ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u200D", new String[] {});

    // ÷ 000A ÷ 0061 × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 000A ÷ 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 000A ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u003A", new String[] {"\u0061"});

    // ÷ 000A ÷ 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 000A ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u0027", new String[] {"\u0061"});

    // ÷ 000A ÷ 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 000A ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 000A ÷ 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 000A ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0061\u002C", new String[] {"\u0061"});

    // ÷ 000A ÷ 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 000A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0031\u003A", new String[] {"\u0031"});

    // ÷ 000A ÷ 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 000A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0031\u0027", new String[] {"\u0031"});

    // ÷ 000A ÷ 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 000A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0031\u002C", new String[] {"\u0031"});

    // ÷ 000A ÷ 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 000A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 000A ÷ 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <LINE FEED (LF)> (LF) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\n\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 000B ÷ 0001 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0001", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 0001 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0001", new String[] {});

    // ÷ 000B ÷ 000D ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\r", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 000D ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\r", new String[] {});

    // ÷ 000B ÷ 000A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\n", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 000A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\n", new String[] {});

    // ÷ 000B ÷ 000B ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u000B", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 000B ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u000B", new String[] {});

    // ÷ 000B ÷ 3031 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u3031", new String[] {"\u3031"});

    // ÷ 000B ÷ 0308 ÷ 3031 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u3031", new String[] {"\u3031"});

    // ÷ 000B ÷ 0041 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0041", new String[] {"\u0041"});

    // ÷ 000B ÷ 0308 ÷ 0041 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0041", new String[] {"\u0041"});

    // ÷ 000B ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u003A", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u003A", new String[] {});

    // ÷ 000B ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u002C", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u002C", new String[] {});

    // ÷ 000B ÷ 002E ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u002E", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 002E ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u002E", new String[] {});

    // ÷ 000B ÷ 0030 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0030", new String[] {"\u0030"});

    // ÷ 000B ÷ 0308 ÷ 0030 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0030", new String[] {"\u0030"});

    // ÷ 000B ÷ 005F ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LOW LINE (ExtendNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u005F", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 005F ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u005F", new String[] {});

    // ÷ 000B ÷ 1F1E6 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\uD83C\uDDE6", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 000B ÷ 05D0 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u05D0", new String[] {"\u05D0"});

    // ÷ 000B ÷ 0308 ÷ 05D0 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 000B ÷ 0022 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] QUOTATION MARK (Double_Quote)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\"", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 0022 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\"", new String[] {});

    // ÷ 000B ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0027", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0027", new String[] {});

    // ÷ 000B ÷ 231A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u231A", new String[] {"\u231A"});

    // ÷ 000B ÷ 0308 ÷ 231A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u231A", new String[] {"\u231A"});

    // ÷ 000B ÷ 0020 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0020", new String[] {});

    // ÷ 000B ÷ 0308 ÷ 0020 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0020", new String[] {});

    // ÷ 000B ÷ 00AD ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] SOFT HYPHEN (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u00AD", new String[] {});

    // ÷ 000B ÷ 0308 × 00AD ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u00AD", new String[] {});

    // ÷ 000B ÷ 0300 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0300", new String[] {});

    // ÷ 000B ÷ 0308 × 0300 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0300", new String[] {});

    // ÷ 000B ÷ 200D ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u200D", new String[] {});

    // ÷ 000B ÷ 0308 × 200D ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u200D", new String[] {});

    // ÷ 000B ÷ 0061 × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 000B ÷ 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 000B ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u003A", new String[] {"\u0061"});

    // ÷ 000B ÷ 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 000B ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u0027", new String[] {"\u0061"});

    // ÷ 000B ÷ 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 000B ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 000B ÷ 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 000B ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0061\u002C", new String[] {"\u0061"});

    // ÷ 000B ÷ 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 000B ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0031\u003A", new String[] {"\u0031"});

    // ÷ 000B ÷ 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 000B ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0031\u0027", new String[] {"\u0031"});

    // ÷ 000B ÷ 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 000B ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0031\u002C", new String[] {"\u0031"});

    // ÷ 000B ÷ 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 000B ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 000B ÷ 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] <LINE TABULATION> (Newline) ÷ [3.1]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u000B\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 3031 ÷ 0001 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] <START OF HEADING>
    // (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0001", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 0001 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0001", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 000D ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [3.2] <CARRIAGE RETURN
    // (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\r", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 000D ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\r", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 000A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [3.2] <LINE FEED (LF)>
    // (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\n", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 000A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\n", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 000B ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [3.2] <LINE TABULATION>
    // (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u000B", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 000B ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u000B", new String[] {"\u3031\u0308"});

    // ÷ 3031 × 3031 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [13.0] VERTICAL KANA
    // REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u3031", new String[] {"\u3031\u3031"});

    // ÷ 3031 × 0308 × 3031 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [13.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u3031", new String[] {"\u3031\u0308\u3031"});

    // ÷ 3031 ÷ 0041 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN CAPITAL
    // LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0041", new String[] {"\u3031", "\u0041"});

    // ÷ 3031 × 0308 ÷ 0041 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0041", new String[] {"\u3031\u0308", "\u0041"});

    // ÷ 3031 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] COLON (MidLetter)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u003A", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u003A", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u002C", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u002C", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 002E ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u002E", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 002E ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u002E", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 0030 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ZERO
    // (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0030", new String[] {"\u3031", "\u0030"});

    // ÷ 3031 × 0308 ÷ 0030 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0030", new String[] {"\u3031\u0308", "\u0030"});

    // ÷ 3031 × 005F ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [13.1] LOW LINE
    // (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u005F", new String[] {"\u3031\u005F"});

    // ÷ 3031 × 0308 × 005F ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u005F", new String[] {"\u3031\u0308\u005F"});

    // ÷ 3031 ÷ 1F1E6 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] REGIONAL
    // INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\uD83C\uDDE6", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\uD83C\uDDE6", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 05D0 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u05D0", new String[] {"\u3031", "\u05D0"});

    // ÷ 3031 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u05D0", new String[] {"\u3031\u0308", "\u05D0"});

    // ÷ 3031 ÷ 0022 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\"", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 0022 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\"", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0027", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0027", new String[] {"\u3031\u0308"});

    // ÷ 3031 ÷ 231A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] WATCH (ExtPict) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u231A", new String[] {"\u3031", "\u231A"});

    // ÷ 3031 × 0308 ÷ 231A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u231A", new String[] {"\u3031\u0308", "\u231A"});

    // ÷ 3031 ÷ 0020 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] SPACE (WSegSpace)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0020", new String[] {"\u3031"});

    // ÷ 3031 × 0308 ÷ 0020 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0020", new String[] {"\u3031\u0308"});

    // ÷ 3031 × 00AD ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] SOFT HYPHEN
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u00AD", new String[] {"\u3031\u00AD"});

    // ÷ 3031 × 0308 × 00AD ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u00AD", new String[] {"\u3031\u0308\u00AD"});

    // ÷ 3031 × 0300 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING GRAVE
    // ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0300", new String[] {"\u3031\u0300"});

    // ÷ 3031 × 0308 × 0300 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0300", new String[] {"\u3031\u0308\u0300"});

    // ÷ 3031 × 200D ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u200D", new String[] {"\u3031\u200D"});

    // ÷ 3031 × 0308 × 200D ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u200D", new String[] {"\u3031\u0308\u200D"});

    // ÷ 3031 ÷ 0061 × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u2060", new String[] {"\u3031", "\u0061\u2060"});

    // ÷ 3031 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u3031\u0308\u0061\u2060", new String[] {"\u3031\u0308", "\u0061\u2060"});

    // ÷ 3031 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u003A", new String[] {"\u3031", "\u0061"});

    // ÷ 3031 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0061\u003A", new String[] {"\u3031\u0308", "\u0061"});

    // ÷ 3031 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u0027", new String[] {"\u3031", "\u0061"});

    // ÷ 3031 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0061\u0027", new String[] {"\u3031\u0308", "\u0061"});

    // ÷ 3031 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0]
    // LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u0027\u2060", new String[] {"\u3031", "\u0061"});

    // ÷ 3031 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u3031\u0308\u0061\u0027\u2060", new String[] {"\u3031\u0308", "\u0061"});

    // ÷ 3031 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0061\u002C", new String[] {"\u3031", "\u0061"});

    // ÷ 3031 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0061\u002C", new String[] {"\u3031\u0308", "\u0061"});

    // ÷ 3031 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0031\u003A", new String[] {"\u3031", "\u0031"});

    // ÷ 3031 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0031\u003A", new String[] {"\u3031\u0308", "\u0031"});

    // ÷ 3031 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0031\u0027", new String[] {"\u3031", "\u0031"});

    // ÷ 3031 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0031\u0027", new String[] {"\u3031\u0308", "\u0031"});

    // ÷ 3031 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0031\u002C", new String[] {"\u3031", "\u0031"});

    // ÷ 3031 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0308\u0031\u002C", new String[] {"\u3031\u0308", "\u0031"});

    // ÷ 3031 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [999.0]
    // DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u0031\u002E\u2060", new String[] {"\u3031", "\u0031"});

    // ÷ 3031 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u3031\u0308\u0031\u002E\u2060", new String[] {"\u3031\u0308", "\u0031"});

    // ÷ 0041 ÷ 0001 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] <START OF HEADING>
    // (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0001", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0001", new String[] {"\u0041\u0308"});

    // ÷ 0041 ÷ 000D ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [3.2] <CARRIAGE RETURN (CR)>
    // (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\r", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\r", new String[] {"\u0041\u0308"});

    // ÷ 0041 ÷ 000A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [3.2] <LINE FEED (LF)> (LF) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0041\n", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\n", new String[] {"\u0041\u0308"});

    // ÷ 0041 ÷ 000B ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [3.2] <LINE TABULATION>
    // (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u000B", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u000B", new String[] {"\u0041\u0308"});

    // ÷ 0041 ÷ 3031 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] VERTICAL KANA REPEAT
    // MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u3031", new String[] {"\u0041", "\u3031"});

    // ÷ 0041 × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u3031", new String[] {"\u0041\u0308", "\u3031"});

    // ÷ 0041 × 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0041", new String[] {"\u0041\u0041"});

    // ÷ 0041 × 0308 × 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0041", new String[] {"\u0041\u0308\u0041"});

    // ÷ 0041 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u003A", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u003A", new String[] {"\u0041\u0308"});

    // ÷ 0041 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u002C", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u002C", new String[] {"\u0041\u0308"});

    // ÷ 0041 ÷ 002E ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] FULL STOP (MidNumLet)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u002E", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 002E ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u002E", new String[] {"\u0041\u0308"});

    // ÷ 0041 × 0030 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ZERO (Numeric) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0030", new String[] {"\u0041\u0030"});

    // ÷ 0041 × 0308 × 0030 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [9.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0030", new String[] {"\u0041\u0308\u0030"});

    // ÷ 0041 × 005F ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [13.1] LOW LINE (ExtendNumLet)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u005F", new String[] {"\u0041\u005F"});

    // ÷ 0041 × 0308 × 005F ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u005F", new String[] {"\u0041\u0308\u005F"});

    // ÷ 0041 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] REGIONAL INDICATOR
    // SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\uD83C\uDDE6", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\uD83C\uDDE6", new String[] {"\u0041\u0308"});

    // ÷ 0041 × 05D0 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u05D0", new String[] {"\u0041\u05D0"});

    // ÷ 0041 × 0308 × 05D0 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u05D0", new String[] {"\u0041\u0308\u05D0"});

    // ÷ 0041 ÷ 0022 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\"", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 0022 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\"", new String[] {"\u0041\u0308"});

    // ÷ 0041 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0027", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0027", new String[] {"\u0041\u0308"});

    // ÷ 0041 ÷ 231A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] WATCH (ExtPict) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u231A", new String[] {"\u0041", "\u231A"});

    // ÷ 0041 × 0308 ÷ 231A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u231A", new String[] {"\u0041\u0308", "\u231A"});

    // ÷ 0041 ÷ 0020 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] SPACE (WSegSpace) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0020", new String[] {"\u0041"});

    // ÷ 0041 × 0308 ÷ 0020 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0020", new String[] {"\u0041\u0308"});

    // ÷ 0041 × 00AD ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] SOFT HYPHEN (Format_FE)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u00AD", new String[] {"\u0041\u00AD"});

    // ÷ 0041 × 0308 × 00AD ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u00AD", new String[] {"\u0041\u0308\u00AD"});

    // ÷ 0041 × 0300 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0300", new String[] {"\u0041\u0300"});

    // ÷ 0041 × 0308 × 0300 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0300", new String[] {"\u0041\u0308\u0300"});

    // ÷ 0041 × 200D ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u200D", new String[] {"\u0041\u200D"});

    // ÷ 0041 × 0308 × 200D ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u200D", new String[] {"\u0041\u0308\u200D"});

    // ÷ 0041 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL
    // LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u2060", new String[] {"\u0041\u0061\u2060"});

    // ÷ 0041 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE)
    // ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0041\u0308\u0061\u2060", new String[] {"\u0041\u0308\u0061\u2060"});

    // ÷ 0041 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u003A", new String[] {"\u0041\u0061"});

    // ÷ 0041 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0061\u003A", new String[] {"\u0041\u0308\u0061"});

    // ÷ 0041 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u0027", new String[] {"\u0041\u0061"});

    // ÷ 0041 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0061\u0027", new String[] {"\u0041\u0308\u0061"});

    // ÷ 0041 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN
    // SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u0027\u2060", new String[] {"\u0041\u0061"});

    // ÷ 0041 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0041\u0308\u0061\u0027\u2060", new String[] {"\u0041\u0308\u0061"});

    // ÷ 0041 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0061\u002C", new String[] {"\u0041\u0061"});

    // ÷ 0041 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0061\u002C", new String[] {"\u0041\u0308\u0061"});

    // ÷ 0041 × 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0031\u003A", new String[] {"\u0041\u0031"});

    // ÷ 0041 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0031\u003A", new String[] {"\u0041\u0308\u0031"});

    // ÷ 0041 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ONE
    // (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0031\u0027", new String[] {"\u0041\u0031"});

    // ÷ 0041 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0031\u0027", new String[] {"\u0041\u0308\u0031"});

    // ÷ 0041 × 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0031\u002C", new String[] {"\u0041\u0031"});

    // ÷ 0041 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0308\u0031\u002C", new String[] {"\u0041\u0308\u0031"});

    // ÷ 0041 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0031\u002E\u2060", new String[] {"\u0041\u0031"});

    // ÷ 0041 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0041\u0308\u0031\u002E\u2060", new String[] {"\u0041\u0308\u0031"});

    // ÷ 003A ÷ 0001 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0001", new String[] {});

    // ÷ 003A × 0308 ÷ 0001 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0001", new String[] {});

    // ÷ 003A ÷ 000D ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\r", new String[] {});

    // ÷ 003A × 0308 ÷ 000D ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\r", new String[] {});

    // ÷ 003A ÷ 000A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\n", new String[] {});

    // ÷ 003A × 0308 ÷ 000A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\n", new String[] {});

    // ÷ 003A ÷ 000B ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u000B", new String[] {});

    // ÷ 003A × 0308 ÷ 000B ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u000B", new String[] {});

    // ÷ 003A ÷ 3031 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u3031", new String[] {"\u3031"});

    // ÷ 003A × 0308 ÷ 3031 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u3031", new String[] {"\u3031"});

    // ÷ 003A ÷ 0041 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0041", new String[] {"\u0041"});

    // ÷ 003A × 0308 ÷ 0041 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0041", new String[] {"\u0041"});

    // ÷ 003A ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u003A", new String[] {});

    // ÷ 003A × 0308 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u003A", new String[] {});

    // ÷ 003A ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u002C", new String[] {});

    // ÷ 003A × 0308 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u002C", new String[] {});

    // ÷ 003A ÷ 002E ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u002E", new String[] {});

    // ÷ 003A × 0308 ÷ 002E ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u002E", new String[] {});

    // ÷ 003A ÷ 0030 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0030", new String[] {"\u0030"});

    // ÷ 003A × 0308 ÷ 0030 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0030", new String[] {"\u0030"});

    // ÷ 003A ÷ 005F ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u005F", new String[] {});

    // ÷ 003A × 0308 ÷ 005F ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u005F", new String[] {});

    // ÷ 003A ÷ 1F1E6 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A
    // (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\uD83C\uDDE6", new String[] {});

    // ÷ 003A × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 003A ÷ 05D0 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u05D0", new String[] {"\u05D0"});

    // ÷ 003A × 0308 ÷ 05D0 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 003A ÷ 0022 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\"", new String[] {});

    // ÷ 003A × 0308 ÷ 0022 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\"", new String[] {});

    // ÷ 003A ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0027", new String[] {});

    // ÷ 003A × 0308 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0027", new String[] {});

    // ÷ 003A ÷ 231A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u231A", new String[] {"\u231A"});

    // ÷ 003A × 0308 ÷ 231A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u231A", new String[] {"\u231A"});

    // ÷ 003A ÷ 0020 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0020", new String[] {});

    // ÷ 003A × 0308 ÷ 0020 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0020", new String[] {});

    // ÷ 003A × 00AD ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u00AD", new String[] {});

    // ÷ 003A × 0308 × 00AD ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u00AD", new String[] {});

    // ÷ 003A × 0300 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0300", new String[] {});

    // ÷ 003A × 0308 × 0300 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0300", new String[] {});

    // ÷ 003A × 200D ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u200D", new String[] {});

    // ÷ 003A × 0308 × 200D ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u200D", new String[] {});

    // ÷ 003A ÷ 0061 × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 003A × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 003A ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u003A", new String[] {"\u0061"});

    // ÷ 003A × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 003A ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u0027", new String[] {"\u0061"});

    // ÷ 003A × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 003A ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 003A × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 003A ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0061\u002C", new String[] {"\u0061"});

    // ÷ 003A × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 003A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0031\u003A", new String[] {"\u0031"});

    // ÷ 003A × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 003A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0031\u0027", new String[] {"\u0031"});

    // ÷ 003A × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 003A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0031\u002C", new String[] {"\u0031"});

    // ÷ 003A × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 003A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 003A × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COLON (MidLetter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u003A\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 002C ÷ 0001 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0001", new String[] {});

    // ÷ 002C × 0308 ÷ 0001 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0001", new String[] {});

    // ÷ 002C ÷ 000D ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\r", new String[] {});

    // ÷ 002C × 0308 ÷ 000D ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\r", new String[] {});

    // ÷ 002C ÷ 000A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\n", new String[] {});

    // ÷ 002C × 0308 ÷ 000A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\n", new String[] {});

    // ÷ 002C ÷ 000B ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u000B", new String[] {});

    // ÷ 002C × 0308 ÷ 000B ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u000B", new String[] {});

    // ÷ 002C ÷ 3031 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u3031", new String[] {"\u3031"});

    // ÷ 002C × 0308 ÷ 3031 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u3031", new String[] {"\u3031"});

    // ÷ 002C ÷ 0041 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0041", new String[] {"\u0041"});

    // ÷ 002C × 0308 ÷ 0041 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0041", new String[] {"\u0041"});

    // ÷ 002C ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u003A", new String[] {});

    // ÷ 002C × 0308 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u003A", new String[] {});

    // ÷ 002C ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u002C", new String[] {});

    // ÷ 002C × 0308 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u002C", new String[] {});

    // ÷ 002C ÷ 002E ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u002E", new String[] {});

    // ÷ 002C × 0308 ÷ 002E ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u002E", new String[] {});

    // ÷ 002C ÷ 0030 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0030", new String[] {"\u0030"});

    // ÷ 002C × 0308 ÷ 0030 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0030", new String[] {"\u0030"});

    // ÷ 002C ÷ 005F ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u005F", new String[] {});

    // ÷ 002C × 0308 ÷ 005F ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u005F", new String[] {});

    // ÷ 002C ÷ 1F1E6 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\uD83C\uDDE6", new String[] {});

    // ÷ 002C × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 002C ÷ 05D0 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u05D0", new String[] {"\u05D0"});

    // ÷ 002C × 0308 ÷ 05D0 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 002C ÷ 0022 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\"", new String[] {});

    // ÷ 002C × 0308 ÷ 0022 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\"", new String[] {});

    // ÷ 002C ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0027", new String[] {});

    // ÷ 002C × 0308 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0027", new String[] {});

    // ÷ 002C ÷ 231A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u231A", new String[] {"\u231A"});

    // ÷ 002C × 0308 ÷ 231A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u231A", new String[] {"\u231A"});

    // ÷ 002C ÷ 0020 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0020", new String[] {});

    // ÷ 002C × 0308 ÷ 0020 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0020", new String[] {});

    // ÷ 002C × 00AD ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u00AD", new String[] {});

    // ÷ 002C × 0308 × 00AD ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u00AD", new String[] {});

    // ÷ 002C × 0300 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0300", new String[] {});

    // ÷ 002C × 0308 × 0300 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0300", new String[] {});

    // ÷ 002C × 200D ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u200D", new String[] {});

    // ÷ 002C × 0308 × 200D ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u200D", new String[] {});

    // ÷ 002C ÷ 0061 × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 002C × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 002C ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u003A", new String[] {"\u0061"});

    // ÷ 002C × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 002C ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u0027", new String[] {"\u0061"});

    // ÷ 002C × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 002C ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 002C × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 002C ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0061\u002C", new String[] {"\u0061"});

    // ÷ 002C × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 002C ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0031\u003A", new String[] {"\u0031"});

    // ÷ 002C × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 002C ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0031\u0027", new String[] {"\u0031"});

    // ÷ 002C × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 002C ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0031\u002C", new String[] {"\u0031"});

    // ÷ 002C × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 002C ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 002C × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COMMA (MidNum) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002C\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 002E ÷ 0001 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0001", new String[] {});

    // ÷ 002E × 0308 ÷ 0001 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0001", new String[] {});

    // ÷ 002E ÷ 000D ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\r", new String[] {});

    // ÷ 002E × 0308 ÷ 000D ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\r", new String[] {});

    // ÷ 002E ÷ 000A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\n", new String[] {});

    // ÷ 002E × 0308 ÷ 000A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\n", new String[] {});

    // ÷ 002E ÷ 000B ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u000B", new String[] {});

    // ÷ 002E × 0308 ÷ 000B ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u000B", new String[] {});

    // ÷ 002E ÷ 3031 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u3031", new String[] {"\u3031"});

    // ÷ 002E × 0308 ÷ 3031 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u3031", new String[] {"\u3031"});

    // ÷ 002E ÷ 0041 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0041", new String[] {"\u0041"});

    // ÷ 002E × 0308 ÷ 0041 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0041", new String[] {"\u0041"});

    // ÷ 002E ÷ 003A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u003A", new String[] {});

    // ÷ 002E × 0308 ÷ 003A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u003A", new String[] {});

    // ÷ 002E ÷ 002C ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u002C", new String[] {});

    // ÷ 002E × 0308 ÷ 002C ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u002C", new String[] {});

    // ÷ 002E ÷ 002E ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u002E", new String[] {});

    // ÷ 002E × 0308 ÷ 002E ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u002E", new String[] {});

    // ÷ 002E ÷ 0030 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0030", new String[] {"\u0030"});

    // ÷ 002E × 0308 ÷ 0030 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0030", new String[] {"\u0030"});

    // ÷ 002E ÷ 005F ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u005F", new String[] {});

    // ÷ 002E × 0308 ÷ 005F ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u005F", new String[] {});

    // ÷ 002E ÷ 1F1E6 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER
    // A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\uD83C\uDDE6", new String[] {});

    // ÷ 002E × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 002E ÷ 05D0 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u05D0", new String[] {"\u05D0"});

    // ÷ 002E × 0308 ÷ 05D0 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 002E ÷ 0022 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u002E\"", new String[] {});

    // ÷ 002E × 0308 ÷ 0022 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\"", new String[] {});

    // ÷ 002E ÷ 0027 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0027", new String[] {});

    // ÷ 002E × 0308 ÷ 0027 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0027", new String[] {});

    // ÷ 002E ÷ 231A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u231A", new String[] {"\u231A"});

    // ÷ 002E × 0308 ÷ 231A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u231A", new String[] {"\u231A"});

    // ÷ 002E ÷ 0020 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0020", new String[] {});

    // ÷ 002E × 0308 ÷ 0020 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0020", new String[] {});

    // ÷ 002E × 00AD ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u00AD", new String[] {});

    // ÷ 002E × 0308 × 00AD ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u00AD", new String[] {});

    // ÷ 002E × 0300 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING GRAVE ACCENT (Extend_FE)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0300", new String[] {});

    // ÷ 002E × 0308 × 0300 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0300", new String[] {});

    // ÷ 002E × 200D ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u200D", new String[] {});

    // ÷ 002E × 0308 × 200D ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u200D", new String[] {});

    // ÷ 002E ÷ 0061 × 2060 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 002E × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 002E ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0061\u003A", new String[] {"\u0061"});

    // ÷ 002E × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 002E ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0061\u0027", new String[] {"\u0061"});

    // ÷ 002E × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 002E ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER
    // A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 002E × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 002E ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0061\u002C", new String[] {"\u0061"});

    // ÷ 002E × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 002E ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0031\u003A", new String[] {"\u0031"});

    // ÷ 002E × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 002E ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0031\u0027", new String[] {"\u0031"});

    // ÷ 002E × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 002E ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0031\u002C", new String[] {"\u0031"});

    // ÷ 002E × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 002E ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric)
    // ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 002E × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] FULL STOP (MidNumLet) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u002E\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0030 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0001", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0001", new String[] {"\u0030\u0308"});

    // ÷ 0030 ÷ 000D ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\r", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\r", new String[] {"\u0030\u0308"});

    // ÷ 0030 ÷ 000A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\n", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\n", new String[] {"\u0030\u0308"});

    // ÷ 0030 ÷ 000B ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u000B", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u000B", new String[] {"\u0030\u0308"});

    // ÷ 0030 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u3031", new String[] {"\u0030", "\u3031"});

    // ÷ 0030 × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u3031", new String[] {"\u0030\u0308", "\u3031"});

    // ÷ 0030 × 0041 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN CAPITAL LETTER A (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0041", new String[] {"\u0030\u0041"});

    // ÷ 0030 × 0308 × 0041 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [10.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0041", new String[] {"\u0030\u0308\u0041"});

    // ÷ 0030 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u003A", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u003A", new String[] {"\u0030\u0308"});

    // ÷ 0030 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u002C", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u002C", new String[] {"\u0030\u0308"});

    // ÷ 0030 ÷ 002E ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u002E", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 002E ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u002E", new String[] {"\u0030\u0308"});

    // ÷ 0030 × 0030 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0030", new String[] {"\u0030\u0030"});

    // ÷ 0030 × 0308 × 0030 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [8.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0030", new String[] {"\u0030\u0308\u0030"});

    // ÷ 0030 × 005F ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u005F", new String[] {"\u0030\u005F"});

    // ÷ 0030 × 0308 × 005F ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u005F", new String[] {"\u0030\u0308\u005F"});

    // ÷ 0030 ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER
    // A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\uD83C\uDDE6", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\uD83C\uDDE6", new String[] {"\u0030\u0308"});

    // ÷ 0030 × 05D0 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] HEBREW LETTER ALEF (Hebrew_Letter)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u05D0", new String[] {"\u0030\u05D0"});

    // ÷ 0030 × 0308 × 05D0 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [10.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u05D0", new String[] {"\u0030\u0308\u05D0"});

    // ÷ 0030 ÷ 0022 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0030\"", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 0022 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\"", new String[] {"\u0030\u0308"});

    // ÷ 0030 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0027", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0027", new String[] {"\u0030\u0308"});

    // ÷ 0030 ÷ 231A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u231A", new String[] {"\u0030", "\u231A"});

    // ÷ 0030 × 0308 ÷ 231A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u231A", new String[] {"\u0030\u0308", "\u231A"});

    // ÷ 0030 ÷ 0020 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0020", new String[] {"\u0030"});

    // ÷ 0030 × 0308 ÷ 0020 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0020", new String[] {"\u0030\u0308"});

    // ÷ 0030 × 00AD ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u00AD", new String[] {"\u0030\u00AD"});

    // ÷ 0030 × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u00AD", new String[] {"\u0030\u0308\u00AD"});

    // ÷ 0030 × 0300 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0300", new String[] {"\u0030\u0300"});

    // ÷ 0030 × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0300", new String[] {"\u0030\u0308\u0300"});

    // ÷ 0030 × 200D ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u200D", new String[] {"\u0030\u200D"});

    // ÷ 0030 × 0308 × 200D ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u200D", new String[] {"\u0030\u0308\u200D"});

    // ÷ 0030 × 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u2060", new String[] {"\u0030\u0061\u2060"});

    // ÷ 0030 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0030\u0308\u0061\u2060", new String[] {"\u0030\u0308\u0061\u2060"});

    // ÷ 0030 × 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u003A", new String[] {"\u0030\u0061"});

    // ÷ 0030 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0061\u003A", new String[] {"\u0030\u0308\u0061"});

    // ÷ 0030 × 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u0027", new String[] {"\u0030\u0061"});

    // ÷ 0030 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0061\u0027", new String[] {"\u0030\u0308\u0061"});

    // ÷ 0030 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u0027\u2060", new String[] {"\u0030\u0061"});

    // ÷ 0030 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0030\u0308\u0061\u0027\u2060", new String[] {"\u0030\u0308\u0061"});

    // ÷ 0030 × 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [10.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0061\u002C", new String[] {"\u0030\u0061"});

    // ÷ 0030 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [10.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0061\u002C", new String[] {"\u0030\u0308\u0061"});

    // ÷ 0030 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0031\u003A", new String[] {"\u0030\u0031"});

    // ÷ 0030 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0031\u003A", new String[] {"\u0030\u0308\u0031"});

    // ÷ 0030 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ONE (Numeric) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0031\u0027", new String[] {"\u0030\u0031"});

    // ÷ 0030 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0031\u0027", new String[] {"\u0030\u0308\u0031"});

    // ÷ 0030 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0031\u002C", new String[] {"\u0030\u0031"});

    // ÷ 0030 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0308\u0031\u002C", new String[] {"\u0030\u0308\u0031"});

    // ÷ 0030 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [8.0] DIGIT ONE (Numeric) ÷
    // [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u0031\u002E\u2060", new String[] {"\u0030\u0031"});

    // ÷ 0030 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [8.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0030\u0308\u0031\u002E\u2060", new String[] {"\u0030\u0308\u0031"});

    // ÷ 005F ÷ 0001 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0001", new String[] {});

    // ÷ 005F × 0308 ÷ 0001 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0001", new String[] {});

    // ÷ 005F ÷ 000D ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u005F\r", new String[] {});

    // ÷ 005F × 0308 ÷ 000D ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\r", new String[] {});

    // ÷ 005F ÷ 000A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\n", new String[] {});

    // ÷ 005F × 0308 ÷ 000A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\n", new String[] {});

    // ÷ 005F ÷ 000B ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u000B", new String[] {});

    // ÷ 005F × 0308 ÷ 000B ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u000B", new String[] {});

    // ÷ 005F × 3031 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u3031", new String[] {"\u005F\u3031"});

    // ÷ 005F × 0308 × 3031 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u3031", new String[] {"\u005F\u0308\u3031"});

    // ÷ 005F × 0041 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN CAPITAL LETTER A (ALetter)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0041", new String[] {"\u005F\u0041"});

    // ÷ 005F × 0308 × 0041 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0041", new String[] {"\u005F\u0308\u0041"});

    // ÷ 005F ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u003A", new String[] {});

    // ÷ 005F × 0308 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u003A", new String[] {});

    // ÷ 005F ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u002C", new String[] {});

    // ÷ 005F × 0308 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u002C", new String[] {});

    // ÷ 005F ÷ 002E ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u002E", new String[] {});

    // ÷ 005F × 0308 ÷ 002E ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u002E", new String[] {});

    // ÷ 005F × 0030 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0030", new String[] {"\u005F\u0030"});

    // ÷ 005F × 0308 × 0030 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0030", new String[] {"\u005F\u0308\u0030"});

    // ÷ 005F × 005F ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u005F", new String[] {});

    // ÷ 005F × 0308 × 005F ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u005F", new String[] {});

    // ÷ 005F ÷ 1F1E6 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\uD83C\uDDE6", new String[] {});

    // ÷ 005F × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 005F × 05D0 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u05D0", new String[] {"\u005F\u05D0"});

    // ÷ 005F × 0308 × 05D0 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u05D0", new String[] {"\u005F\u0308\u05D0"});

    // ÷ 005F ÷ 0022 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u005F\"", new String[] {});

    // ÷ 005F × 0308 ÷ 0022 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\"", new String[] {});

    // ÷ 005F ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0027", new String[] {});

    // ÷ 005F × 0308 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0027", new String[] {});

    // ÷ 005F ÷ 231A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u231A", new String[] {"\u231A"});

    // ÷ 005F × 0308 ÷ 231A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u231A", new String[] {"\u231A"});

    // ÷ 005F ÷ 0020 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0020", new String[] {});

    // ÷ 005F × 0308 ÷ 0020 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0020", new String[] {});

    // ÷ 005F × 00AD ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u00AD", new String[] {});

    // ÷ 005F × 0308 × 00AD ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u00AD", new String[] {});

    // ÷ 005F × 0300 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0300", new String[] {});

    // ÷ 005F × 0308 × 0300 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0300", new String[] {});

    // ÷ 005F × 200D ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u200D", new String[] {});

    // ÷ 005F × 0308 × 200D ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u200D", new String[] {});

    // ÷ 005F × 0061 × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u2060", new String[] {"\u005F\u0061\u2060"});

    // ÷ 005F × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u005F\u0308\u0061\u2060", new String[] {"\u005F\u0308\u0061\u2060"});

    // ÷ 005F × 0061 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u003A", new String[] {"\u005F\u0061"});

    // ÷ 005F × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0061\u003A", new String[] {"\u005F\u0308\u0061"});

    // ÷ 005F × 0061 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u0027", new String[] {"\u005F\u0061"});

    // ÷ 005F × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0061\u0027", new String[] {"\u005F\u0308\u0061"});

    // ÷ 005F × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER
    // A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u0027\u2060", new String[] {"\u005F\u0061"});

    // ÷ 005F × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u005F\u0308\u0061\u0027\u2060", new String[] {"\u005F\u0308\u0061"});

    // ÷ 005F × 0061 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0061\u002C", new String[] {"\u005F\u0061"});

    // ÷ 005F × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0061\u002C", new String[] {"\u005F\u0308\u0061"});

    // ÷ 005F × 0031 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0031\u003A", new String[] {"\u005F\u0031"});

    // ÷ 005F × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0031\u003A", new String[] {"\u005F\u0308\u0031"});

    // ÷ 005F × 0031 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0031\u0027", new String[] {"\u005F\u0031"});

    // ÷ 005F × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0031\u0027", new String[] {"\u005F\u0308\u0031"});

    // ÷ 005F × 0031 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0031\u002C", new String[] {"\u005F\u0031"});

    // ÷ 005F × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0308\u0031\u002C", new String[] {"\u005F\u0308\u0031"});

    // ÷ 005F × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u005F\u0031\u002E\u2060", new String[] {"\u005F\u0031"});

    // ÷ 005F × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LOW LINE (ExtendNumLet) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u005F\u0308\u0031\u002E\u2060", new String[] {"\u005F\u0308\u0031"});

    // ÷ 1F1E6 ÷ 0001 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] <START OF
    // HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0001", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 0001 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0001", new String[] {});

    // ÷ 1F1E6 ÷ 000D ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [3.2] <CARRIAGE RETURN
    // (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\r", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 000D ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\r", new String[] {});

    // ÷ 1F1E6 ÷ 000A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [3.2] <LINE FEED (LF)>
    // (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\n", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 000A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\n", new String[] {});

    // ÷ 1F1E6 ÷ 000B ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [3.2] <LINE
    // TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u000B", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 000B ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u000B", new String[] {});

    // ÷ 1F1E6 ÷ 3031 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] VERTICAL KANA
    // REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u3031", new String[] {"\u3031"});

    // ÷ 1F1E6 × 0308 ÷ 3031 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u3031", new String[] {"\u3031"});

    // ÷ 1F1E6 ÷ 0041 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] LATIN CAPITAL
    // LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0041", new String[] {"\u0041"});

    // ÷ 1F1E6 × 0308 ÷ 0041 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0041", new String[] {"\u0041"});

    // ÷ 1F1E6 ÷ 003A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] COLON
    // (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u003A", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 003A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u003A", new String[] {});

    // ÷ 1F1E6 ÷ 002C ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] COMMA (MidNum)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u002C", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 002C ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u002C", new String[] {});

    // ÷ 1F1E6 ÷ 002E ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u002E", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 002E ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u002E", new String[] {});

    // ÷ 1F1E6 ÷ 0030 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] DIGIT ZERO
    // (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0030", new String[] {"\u0030"});

    // ÷ 1F1E6 × 0308 ÷ 0030 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0030", new String[] {"\u0030"});

    // ÷ 1F1E6 ÷ 005F ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] LOW LINE
    // (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u005F", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 005F ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u005F", new String[] {});

    // ÷ 1F1E6 × 1F1E6 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [15.0] REGIONAL
    // INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\uD83C\uDDE6\uD83C\uDDE6", new String[] {"\uD83C\uDDE6\uD83C\uDDE6"});

    // ÷ 1F1E6 × 0308 × 1F1E6 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [15.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\uD83C\uDDE6\u0308\uD83C\uDDE6",
        new String[] {"\uD83C\uDDE6\u0308\uD83C\uDDE6"});

    // ÷ 1F1E6 ÷ 05D0 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] HEBREW LETTER
    // ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u05D0", new String[] {"\u05D0"});

    // ÷ 1F1E6 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 1F1E6 ÷ 0022 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\"", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 0022 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\"", new String[] {});

    // ÷ 1F1E6 ÷ 0027 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0027", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 0027 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0027", new String[] {});

    // ÷ 1F1E6 ÷ 231A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] WATCH
    // (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u231A", new String[] {"\u231A"});

    // ÷ 1F1E6 × 0308 ÷ 231A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u231A", new String[] {"\u231A"});

    // ÷ 1F1E6 ÷ 0020 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] SPACE
    // (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0020", new String[] {});

    // ÷ 1F1E6 × 0308 ÷ 0020 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0020", new String[] {});

    // ÷ 1F1E6 × 00AD ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] SOFT HYPHEN
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u00AD", new String[] {});

    // ÷ 1F1E6 × 0308 × 00AD ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u00AD", new String[] {});

    // ÷ 1F1E6 × 0300 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING GRAVE
    // ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0300", new String[] {});

    // ÷ 1F1E6 × 0308 × 0300 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0300", new String[] {});

    // ÷ 1F1E6 × 200D ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] ZERO WIDTH
    // JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u200D", new String[] {});

    // ÷ 1F1E6 × 0308 × 200D ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u200D", new String[] {});

    // ÷ 1F1E6 ÷ 0061 × 2060 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] LATIN
    // SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 1F1E6 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 1F1E6 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] LATIN
    // SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0061\u003A", new String[] {"\u0061"});

    // ÷ 1F1E6 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 1F1E6 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] LATIN
    // SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0061\u0027", new String[] {"\u0061"});

    // ÷ 1F1E6 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 1F1E6 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0]
    // LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 1F1E6 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 1F1E6 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] LATIN
    // SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0061\u002C", new String[] {"\u0061"});

    // ÷ 1F1E6 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 1F1E6 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] DIGIT
    // ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0031\u003A", new String[] {"\u0031"});

    // ÷ 1F1E6 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 1F1E6 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] DIGIT
    // ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0031\u0027", new String[] {"\u0031"});

    // ÷ 1F1E6 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 1F1E6 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0] DIGIT
    // ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0031\u002C", new String[] {"\u0031"});

    // ÷ 1F1E6 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 1F1E6 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [999.0]
    // DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 1F1E6 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\uD83C\uDDE6\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 05D0 ÷ 0001 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] <START OF HEADING>
    // (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0001", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 0001 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0001", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 ÷ 000D ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [3.2] <CARRIAGE RETURN (CR)>
    // (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\r", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 000D ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\r", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 ÷ 000A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [3.2] <LINE FEED (LF)> (LF)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\n", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 000A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\n", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 ÷ 000B ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [3.2] <LINE TABULATION>
    // (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u000B", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 000B ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u000B", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 ÷ 3031 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] VERTICAL KANA REPEAT
    // MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u3031", new String[] {"\u05D0", "\u3031"});

    // ÷ 05D0 × 0308 ÷ 3031 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u3031", new String[] {"\u05D0\u0308", "\u3031"});

    // ÷ 05D0 × 0041 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [5.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0041", new String[] {"\u05D0\u0041"});

    // ÷ 05D0 × 0308 × 0041 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0041", new String[] {"\u05D0\u0308\u0041"});

    // ÷ 05D0 ÷ 003A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u003A", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 003A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u003A", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 ÷ 002C ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u002C", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 002C ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u002C", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 ÷ 002E ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u002E", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 002E ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u002E", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 × 0030 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [9.0] DIGIT ZERO (Numeric) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0030", new String[] {"\u05D0\u0030"});

    // ÷ 05D0 × 0308 × 0030 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [9.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0030", new String[] {"\u05D0\u0308\u0030"});

    // ÷ 05D0 × 005F ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [13.1] LOW LINE
    // (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u005F", new String[] {"\u05D0\u005F"});

    // ÷ 05D0 × 0308 × 005F ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u005F", new String[] {"\u05D0\u0308\u005F"});

    // ÷ 05D0 ÷ 1F1E6 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] REGIONAL INDICATOR
    // SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\uD83C\uDDE6", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\uD83C\uDDE6", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 × 05D0 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [5.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u05D0", new String[] {"\u05D0\u05D0"});

    // ÷ 05D0 × 0308 × 05D0 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u05D0", new String[] {"\u05D0\u0308\u05D0"});

    // ÷ 05D0 ÷ 0022 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\"", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 0022 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\"", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 × 0027 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [7.1] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0027", new String[] {"\u05D0\u0027"});

    // ÷ 05D0 × 0308 × 0027 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [7.1] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0027", new String[] {"\u05D0\u0308\u0027"});

    // ÷ 05D0 ÷ 231A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] WATCH (ExtPict) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u231A", new String[] {"\u05D0", "\u231A"});

    // ÷ 05D0 × 0308 ÷ 231A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u231A", new String[] {"\u05D0\u0308", "\u231A"});

    // ÷ 05D0 ÷ 0020 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [999.0] SPACE (WSegSpace) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0020", new String[] {"\u05D0"});

    // ÷ 05D0 × 0308 ÷ 0020 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0020", new String[] {"\u05D0\u0308"});

    // ÷ 05D0 × 00AD ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] SOFT HYPHEN
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u00AD", new String[] {"\u05D0\u00AD"});

    // ÷ 05D0 × 0308 × 00AD ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u00AD", new String[] {"\u05D0\u0308\u00AD"});

    // ÷ 05D0 × 0300 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0300", new String[] {"\u05D0\u0300"});

    // ÷ 05D0 × 0308 × 0300 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0300", new String[] {"\u05D0\u0308\u0300"});

    // ÷ 05D0 × 200D ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u200D", new String[] {"\u05D0\u200D"});

    // ÷ 05D0 × 0308 × 200D ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u200D", new String[] {"\u05D0\u0308\u200D"});

    // ÷ 05D0 × 0061 × 2060 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [5.0] LATIN SMALL
    // LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0061\u2060", new String[] {"\u05D0\u0061\u2060"});

    // ÷ 05D0 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u05D0\u0308\u0061\u2060", new String[] {"\u05D0\u0308\u0061\u2060"});

    // ÷ 05D0 × 0061 ÷ 003A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [5.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0061\u003A", new String[] {"\u05D0\u0061"});

    // ÷ 05D0 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0061\u003A", new String[] {"\u05D0\u0308\u0061"});

    // ÷ 05D0 × 0061 ÷ 0027 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [5.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0061\u0027", new String[] {"\u05D0\u0061"});

    // ÷ 05D0 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0061\u0027", new String[] {"\u05D0\u0308\u0061"});

    // ÷ 05D0 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [5.0] LATIN
    // SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0061\u0027\u2060", new String[] {"\u05D0\u0061"});

    // ÷ 05D0 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u05D0\u0308\u0061\u0027\u2060", new String[] {"\u05D0\u0308\u0061"});

    // ÷ 05D0 × 0061 ÷ 002C ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [5.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0061\u002C", new String[] {"\u05D0\u0061"});

    // ÷ 05D0 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0061\u002C", new String[] {"\u05D0\u0308\u0061"});

    // ÷ 05D0 × 0031 ÷ 003A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [9.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0031\u003A", new String[] {"\u05D0\u0031"});

    // ÷ 05D0 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0031\u003A", new String[] {"\u05D0\u0308\u0031"});

    // ÷ 05D0 × 0031 ÷ 0027 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [9.0] DIGIT ONE
    // (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0031\u0027", new String[] {"\u05D0\u0031"});

    // ÷ 05D0 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0031\u0027", new String[] {"\u05D0\u0308\u0031"});

    // ÷ 05D0 × 0031 ÷ 002C ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [9.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0031\u002C", new String[] {"\u05D0\u0031"});

    // ÷ 05D0 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0308\u0031\u002C", new String[] {"\u05D0\u0308\u0031"});

    // ÷ 05D0 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [9.0] DIGIT
    // ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0031\u002E\u2060", new String[] {"\u05D0\u0031"});

    // ÷ 05D0 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u05D0\u0308\u0031\u002E\u2060", new String[] {"\u05D0\u0308\u0031"});

    // ÷ 0022 ÷ 0001 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] <START OF HEADING>
    // (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0001", new String[] {});

    // ÷ 0022 × 0308 ÷ 0001 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0001", new String[] {});

    // ÷ 0022 ÷ 000D ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\r", new String[] {});

    // ÷ 0022 × 0308 ÷ 000D ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\r", new String[] {});

    // ÷ 0022 ÷ 000A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [3.2] <LINE FEED (LF)> (LF) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\n", new String[] {});

    // ÷ 0022 × 0308 ÷ 000A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\n", new String[] {});

    // ÷ 0022 ÷ 000B ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [3.2] <LINE TABULATION> (Newline)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u000B", new String[] {});

    // ÷ 0022 × 0308 ÷ 000B ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u000B", new String[] {});

    // ÷ 0022 ÷ 3031 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u3031", new String[] {"\u3031"});

    // ÷ 0022 × 0308 ÷ 3031 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u3031", new String[] {"\u3031"});

    // ÷ 0022 ÷ 0041 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0041", new String[] {"\u0041"});

    // ÷ 0022 × 0308 ÷ 0041 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0041", new String[] {"\u0041"});

    // ÷ 0022 ÷ 003A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u003A", new String[] {});

    // ÷ 0022 × 0308 ÷ 003A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u003A", new String[] {});

    // ÷ 0022 ÷ 002C ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u002C", new String[] {});

    // ÷ 0022 × 0308 ÷ 002C ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u002C", new String[] {});

    // ÷ 0022 ÷ 002E ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\u002E", new String[] {});

    // ÷ 0022 × 0308 ÷ 002E ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u002E", new String[] {});

    // ÷ 0022 ÷ 0030 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] DIGIT ZERO (Numeric) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\u0030", new String[] {"\u0030"});

    // ÷ 0022 × 0308 ÷ 0030 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0030", new String[] {"\u0030"});

    // ÷ 0022 ÷ 005F ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] LOW LINE (ExtendNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\u005F", new String[] {});

    // ÷ 0022 × 0308 ÷ 005F ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u005F", new String[] {});

    // ÷ 0022 ÷ 1F1E6 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] REGIONAL INDICATOR
    // SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\uD83C\uDDE6", new String[] {});

    // ÷ 0022 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 0022 ÷ 05D0 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u05D0", new String[] {"\u05D0"});

    // ÷ 0022 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 0022 ÷ 0022 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\"", new String[] {});

    // ÷ 0022 × 0308 ÷ 0022 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\"", new String[] {});

    // ÷ 0022 ÷ 0027 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] APOSTROPHE (Single_Quote)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0027", new String[] {});

    // ÷ 0022 × 0308 ÷ 0027 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0027", new String[] {});

    // ÷ 0022 ÷ 231A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u231A", new String[] {"\u231A"});

    // ÷ 0022 × 0308 ÷ 231A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u231A", new String[] {"\u231A"});

    // ÷ 0022 ÷ 0020 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0020", new String[] {});

    // ÷ 0022 × 0308 ÷ 0020 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0020", new String[] {});

    // ÷ 0022 × 00AD ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] SOFT HYPHEN (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\u00AD", new String[] {});

    // ÷ 0022 × 0308 × 00AD ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u00AD", new String[] {});

    // ÷ 0022 × 0300 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0300", new String[] {});

    // ÷ 0022 × 0308 × 0300 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0300", new String[] {});

    // ÷ 0022 × 200D ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] ZERO WIDTH JOINER (ZWJ_FE)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u200D", new String[] {});

    // ÷ 0022 × 0308 × 200D ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u200D", new String[] {});

    // ÷ 0022 ÷ 0061 × 2060 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] LATIN SMALL LETTER
    // A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0022 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0022 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] LATIN SMALL LETTER
    // A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0022 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0022 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] LATIN SMALL LETTER
    // A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0022 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0022 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0022 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0022 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] LATIN SMALL LETTER
    // A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0022 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0022 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0022 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0022 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0022 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0022 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0022 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0022 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0022 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] QUOTATION MARK (Double_Quote) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\"\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0027 ÷ 0001 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0001", new String[] {});

    // ÷ 0027 × 0308 ÷ 0001 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0001", new String[] {});

    // ÷ 0027 ÷ 000D ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\r", new String[] {});

    // ÷ 0027 × 0308 ÷ 000D ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\r", new String[] {});

    // ÷ 0027 ÷ 000A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\n", new String[] {});

    // ÷ 0027 × 0308 ÷ 000A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\n", new String[] {});

    // ÷ 0027 ÷ 000B ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u000B", new String[] {});

    // ÷ 0027 × 0308 ÷ 000B ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u000B", new String[] {});

    // ÷ 0027 ÷ 3031 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u3031", new String[] {"\u3031"});

    // ÷ 0027 × 0308 ÷ 3031 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u3031", new String[] {"\u3031"});

    // ÷ 0027 ÷ 0041 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0041", new String[] {"\u0041"});

    // ÷ 0027 × 0308 ÷ 0041 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0041", new String[] {"\u0041"});

    // ÷ 0027 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u003A", new String[] {});

    // ÷ 0027 × 0308 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u003A", new String[] {});

    // ÷ 0027 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u002C", new String[] {});

    // ÷ 0027 × 0308 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u002C", new String[] {});

    // ÷ 0027 ÷ 002E ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u002E", new String[] {});

    // ÷ 0027 × 0308 ÷ 002E ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u002E", new String[] {});

    // ÷ 0027 ÷ 0030 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0030", new String[] {"\u0030"});

    // ÷ 0027 × 0308 ÷ 0030 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0030", new String[] {"\u0030"});

    // ÷ 0027 ÷ 005F ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] LOW LINE (ExtendNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u005F", new String[] {});

    // ÷ 0027 × 0308 ÷ 005F ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u005F", new String[] {});

    // ÷ 0027 ÷ 1F1E6 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\uD83C\uDDE6", new String[] {});

    // ÷ 0027 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 0027 ÷ 05D0 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u05D0", new String[] {"\u05D0"});

    // ÷ 0027 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 0027 ÷ 0022 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] QUOTATION MARK (Double_Quote)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\"", new String[] {});

    // ÷ 0027 × 0308 ÷ 0022 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\"", new String[] {});

    // ÷ 0027 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0027", new String[] {});

    // ÷ 0027 × 0308 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0027", new String[] {});

    // ÷ 0027 ÷ 231A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u231A", new String[] {"\u231A"});

    // ÷ 0027 × 0308 ÷ 231A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u231A", new String[] {"\u231A"});

    // ÷ 0027 ÷ 0020 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0020", new String[] {});

    // ÷ 0027 × 0308 ÷ 0020 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0020", new String[] {});

    // ÷ 0027 × 00AD ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u00AD", new String[] {});

    // ÷ 0027 × 0308 × 00AD ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u00AD", new String[] {});

    // ÷ 0027 × 0300 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0300", new String[] {});

    // ÷ 0027 × 0308 × 0300 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0300", new String[] {});

    // ÷ 0027 × 200D ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u200D", new String[] {});

    // ÷ 0027 × 0308 × 200D ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u200D", new String[] {});

    // ÷ 0027 ÷ 0061 × 2060 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0027 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0027 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0027 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0027 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0027 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0027 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0027 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0027 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0027 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0027 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0027 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0027 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0027 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0027 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0027 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0027 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0027 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] APOSTROPHE (Single_Quote) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0027\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 231A ÷ 0001 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0001", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 0001 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0001", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 000D ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\r", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 000D ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\r", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 000A ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\n", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 000A ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\n", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 000B ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u000B", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 000B ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u000B", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 3031 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u3031", new String[] {"\u231A", "\u3031"});

    // ÷ 231A × 0308 ÷ 3031 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u3031", new String[] {"\u231A\u0308", "\u3031"});

    // ÷ 231A ÷ 0041 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0041", new String[] {"\u231A", "\u0041"});

    // ÷ 231A × 0308 ÷ 0041 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0041", new String[] {"\u231A\u0308", "\u0041"});

    // ÷ 231A ÷ 003A ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u003A", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 003A ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u003A", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 002C ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u002C", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 002C ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u002C", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 002E ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u002E", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 002E ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u002E", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 0030 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0030", new String[] {"\u231A", "\u0030"});

    // ÷ 231A × 0308 ÷ 0030 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0030", new String[] {"\u231A\u0308", "\u0030"});

    // ÷ 231A ÷ 005F ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u005F", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 005F ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u005F", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 1F1E6 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A
    // (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\uD83C\uDDE6", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\uD83C\uDDE6", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 05D0 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u05D0", new String[] {"\u231A", "\u05D0"});

    // ÷ 231A × 0308 ÷ 05D0 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u05D0", new String[] {"\u231A\u0308", "\u05D0"});

    // ÷ 231A ÷ 0022 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\"", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 0022 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\"", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 0027 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0027", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 0027 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0027", new String[] {"\u231A\u0308"});

    // ÷ 231A ÷ 231A ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u231A", new String[] {"\u231A", "\u231A"});

    // ÷ 231A × 0308 ÷ 231A ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u231A", new String[] {"\u231A\u0308", "\u231A"});

    // ÷ 231A ÷ 0020 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0020", new String[] {"\u231A"});

    // ÷ 231A × 0308 ÷ 0020 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0020", new String[] {"\u231A\u0308"});

    // ÷ 231A × 00AD ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u00AD", new String[] {"\u231A\u00AD"});

    // ÷ 231A × 0308 × 00AD ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u00AD", new String[] {"\u231A\u0308\u00AD"});

    // ÷ 231A × 0300 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0300", new String[] {"\u231A\u0300"});

    // ÷ 231A × 0308 × 0300 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0300", new String[] {"\u231A\u0308\u0300"});

    // ÷ 231A × 200D ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u200D", new String[] {"\u231A\u200D"});

    // ÷ 231A × 0308 × 200D ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u200D", new String[] {"\u231A\u0308\u200D"});

    // ÷ 231A ÷ 0061 × 2060 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0061\u2060", new String[] {"\u231A", "\u0061\u2060"});

    // ÷ 231A × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u231A\u0308\u0061\u2060", new String[] {"\u231A\u0308", "\u0061\u2060"});

    // ÷ 231A ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0061\u003A", new String[] {"\u231A", "\u0061"});

    // ÷ 231A × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0061\u003A", new String[] {"\u231A\u0308", "\u0061"});

    // ÷ 231A ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0061\u0027", new String[] {"\u231A", "\u0061"});

    // ÷ 231A × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0061\u0027", new String[] {"\u231A\u0308", "\u0061"});

    // ÷ 231A ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0061\u0027\u2060", new String[] {"\u231A", "\u0061"});

    // ÷ 231A × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u231A\u0308\u0061\u0027\u2060", new String[] {"\u231A\u0308", "\u0061"});

    // ÷ 231A ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0061\u002C", new String[] {"\u231A", "\u0061"});

    // ÷ 231A × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0061\u002C", new String[] {"\u231A\u0308", "\u0061"});

    // ÷ 231A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0031\u003A", new String[] {"\u231A", "\u0031"});

    // ÷ 231A × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0031\u003A", new String[] {"\u231A\u0308", "\u0031"});

    // ÷ 231A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0031\u0027", new String[] {"\u231A", "\u0031"});

    // ÷ 231A × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0031\u0027", new String[] {"\u231A\u0308", "\u0031"});

    // ÷ 231A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0031\u002C", new String[] {"\u231A", "\u0031"});

    // ÷ 231A × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0308\u0031\u002C", new String[] {"\u231A\u0308", "\u0031"});

    // ÷ 231A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] WATCH (ExtPict) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u231A\u0031\u002E\u2060", new String[] {"\u231A", "\u0031"});

    // ÷ 231A × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] WATCH (ExtPict) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u231A\u0308\u0031\u002E\u2060", new String[] {"\u231A\u0308", "\u0031"});

    // ÷ 0020 ÷ 0001 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0001", new String[] {});

    // ÷ 0020 × 0308 ÷ 0001 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0001", new String[] {});

    // ÷ 0020 ÷ 000D ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\r", new String[] {});

    // ÷ 0020 × 0308 ÷ 000D ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\r", new String[] {});

    // ÷ 0020 ÷ 000A ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\n", new String[] {});

    // ÷ 0020 × 0308 ÷ 000A ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\n", new String[] {});

    // ÷ 0020 ÷ 000B ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u000B", new String[] {});

    // ÷ 0020 × 0308 ÷ 000B ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u000B", new String[] {});

    // ÷ 0020 ÷ 3031 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u3031", new String[] {"\u3031"});

    // ÷ 0020 × 0308 ÷ 3031 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u3031", new String[] {"\u3031"});

    // ÷ 0020 ÷ 0041 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0041", new String[] {"\u0041"});

    // ÷ 0020 × 0308 ÷ 0041 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0041", new String[] {"\u0041"});

    // ÷ 0020 ÷ 003A ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u003A", new String[] {});

    // ÷ 0020 × 0308 ÷ 003A ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u003A", new String[] {});

    // ÷ 0020 ÷ 002C ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u002C", new String[] {});

    // ÷ 0020 × 0308 ÷ 002C ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u002C", new String[] {});

    // ÷ 0020 ÷ 002E ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u002E", new String[] {});

    // ÷ 0020 × 0308 ÷ 002E ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u002E", new String[] {});

    // ÷ 0020 ÷ 0030 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0030", new String[] {"\u0030"});

    // ÷ 0020 × 0308 ÷ 0030 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0030", new String[] {"\u0030"});

    // ÷ 0020 ÷ 005F ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u005F", new String[] {});

    // ÷ 0020 × 0308 ÷ 005F ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u005F", new String[] {});

    // ÷ 0020 ÷ 1F1E6 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A
    // (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\uD83C\uDDE6", new String[] {});

    // ÷ 0020 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 0020 ÷ 05D0 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u05D0", new String[] {"\u05D0"});

    // ÷ 0020 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 0020 ÷ 0022 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\"", new String[] {});

    // ÷ 0020 × 0308 ÷ 0022 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\"", new String[] {});

    // ÷ 0020 ÷ 0027 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0027", new String[] {});

    // ÷ 0020 × 0308 ÷ 0027 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0027", new String[] {});

    // ÷ 0020 ÷ 231A ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u231A", new String[] {"\u231A"});

    // ÷ 0020 × 0308 ÷ 231A ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u231A", new String[] {"\u231A"});

    // ÷ 0020 × 0020 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [3.4] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0020", new String[] {});

    // ÷ 0020 × 0308 ÷ 0020 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0020", new String[] {});

    // ÷ 0020 × 00AD ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u00AD", new String[] {});

    // ÷ 0020 × 0308 × 00AD ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u00AD", new String[] {});

    // ÷ 0020 × 0300 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0300", new String[] {});

    // ÷ 0020 × 0308 × 0300 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0300", new String[] {});

    // ÷ 0020 × 200D ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u200D", new String[] {});

    // ÷ 0020 × 0308 × 200D ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u200D", new String[] {});

    // ÷ 0020 ÷ 0061 × 2060 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0020 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0020 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0020 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0020 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0020 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0020 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0020 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0020 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0020 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0020 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0020 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0020 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0020 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0020 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0020 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0020 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] SPACE (WSegSpace) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0020 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 00AD ÷ 0001 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0001", new String[] {});

    // ÷ 00AD × 0308 ÷ 0001 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0001", new String[] {});

    // ÷ 00AD ÷ 000D ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\r", new String[] {});

    // ÷ 00AD × 0308 ÷ 000D ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\r", new String[] {});

    // ÷ 00AD ÷ 000A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\n", new String[] {});

    // ÷ 00AD × 0308 ÷ 000A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\n", new String[] {});

    // ÷ 00AD ÷ 000B ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u000B", new String[] {});

    // ÷ 00AD × 0308 ÷ 000B ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u000B", new String[] {});

    // ÷ 00AD ÷ 3031 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u3031", new String[] {"\u3031"});

    // ÷ 00AD × 0308 ÷ 3031 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u3031", new String[] {"\u3031"});

    // ÷ 00AD ÷ 0041 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0041", new String[] {"\u0041"});

    // ÷ 00AD × 0308 ÷ 0041 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0041", new String[] {"\u0041"});

    // ÷ 00AD ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u003A", new String[] {});

    // ÷ 00AD × 0308 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u003A", new String[] {});

    // ÷ 00AD ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u002C", new String[] {});

    // ÷ 00AD × 0308 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u002C", new String[] {});

    // ÷ 00AD ÷ 002E ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u002E", new String[] {});

    // ÷ 00AD × 0308 ÷ 002E ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u002E", new String[] {});

    // ÷ 00AD ÷ 0030 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0030", new String[] {"\u0030"});

    // ÷ 00AD × 0308 ÷ 0030 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0030", new String[] {"\u0030"});

    // ÷ 00AD ÷ 005F ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u005F", new String[] {});

    // ÷ 00AD × 0308 ÷ 005F ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u005F", new String[] {});

    // ÷ 00AD ÷ 1F1E6 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\uD83C\uDDE6", new String[] {});

    // ÷ 00AD × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 00AD ÷ 05D0 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u05D0", new String[] {"\u05D0"});

    // ÷ 00AD × 0308 ÷ 05D0 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 00AD ÷ 0022 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\"", new String[] {});

    // ÷ 00AD × 0308 ÷ 0022 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\"", new String[] {});

    // ÷ 00AD ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0027", new String[] {});

    // ÷ 00AD × 0308 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0027", new String[] {});

    // ÷ 00AD ÷ 231A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u231A", new String[] {"\u231A"});

    // ÷ 00AD × 0308 ÷ 231A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u231A", new String[] {"\u231A"});

    // ÷ 00AD ÷ 0020 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0020", new String[] {});

    // ÷ 00AD × 0308 ÷ 0020 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0020", new String[] {});

    // ÷ 00AD × 00AD ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u00AD", new String[] {});

    // ÷ 00AD × 0308 × 00AD ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u00AD", new String[] {});

    // ÷ 00AD × 0300 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0300", new String[] {});

    // ÷ 00AD × 0308 × 0300 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0300", new String[] {});

    // ÷ 00AD × 200D ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u200D", new String[] {});

    // ÷ 00AD × 0308 × 200D ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u200D", new String[] {});

    // ÷ 00AD ÷ 0061 × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 00AD × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 00AD ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u003A", new String[] {"\u0061"});

    // ÷ 00AD × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 00AD ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u0027", new String[] {"\u0061"});

    // ÷ 00AD × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 00AD ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 00AD × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 00AD ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0061\u002C", new String[] {"\u0061"});

    // ÷ 00AD × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 00AD ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0031\u003A", new String[] {"\u0031"});

    // ÷ 00AD × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 00AD ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0031\u0027", new String[] {"\u0031"});

    // ÷ 00AD × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 00AD ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0031\u002C", new String[] {"\u0031"});

    // ÷ 00AD × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 00AD ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 00AD × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] SOFT HYPHEN (Format_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u00AD\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0300 ÷ 0001 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] <START OF HEADING>
    // (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0001", new String[] {});

    // ÷ 0300 × 0308 ÷ 0001 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0001", new String[] {});

    // ÷ 0300 ÷ 000D ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)>
    // (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\r", new String[] {});

    // ÷ 0300 × 0308 ÷ 000D ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\r", new String[] {});

    // ÷ 0300 ÷ 000A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\n", new String[] {});

    // ÷ 0300 × 0308 ÷ 000A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\n", new String[] {});

    // ÷ 0300 ÷ 000B ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [3.2] <LINE TABULATION>
    // (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u000B", new String[] {});

    // ÷ 0300 × 0308 ÷ 000B ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u000B", new String[] {});

    // ÷ 0300 ÷ 3031 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT
    // MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u3031", new String[] {"\u3031"});

    // ÷ 0300 × 0308 ÷ 3031 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u3031", new String[] {"\u3031"});

    // ÷ 0300 ÷ 0041 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER
    // A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0041", new String[] {"\u0041"});

    // ÷ 0300 × 0308 ÷ 0041 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0041", new String[] {"\u0041"});

    // ÷ 0300 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u003A", new String[] {});

    // ÷ 0300 × 0308 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u003A", new String[] {});

    // ÷ 0300 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u002C", new String[] {});

    // ÷ 0300 × 0308 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u002C", new String[] {});

    // ÷ 0300 ÷ 002E ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u002E", new String[] {});

    // ÷ 0300 × 0308 ÷ 002E ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u002E", new String[] {});

    // ÷ 0300 ÷ 0030 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0030", new String[] {"\u0030"});

    // ÷ 0300 × 0308 ÷ 0030 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0030", new String[] {"\u0030"});

    // ÷ 0300 ÷ 005F ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LOW LINE
    // (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u005F", new String[] {});

    // ÷ 0300 × 0308 ÷ 005F ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u005F", new String[] {});

    // ÷ 0300 ÷ 1F1E6 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] REGIONAL INDICATOR
    // SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\uD83C\uDDE6", new String[] {});

    // ÷ 0300 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 0300 ÷ 05D0 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u05D0", new String[] {"\u05D0"});

    // ÷ 0300 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 0300 ÷ 0022 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\"", new String[] {});

    // ÷ 0300 × 0308 ÷ 0022 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\"", new String[] {});

    // ÷ 0300 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0027", new String[] {});

    // ÷ 0300 × 0308 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0027", new String[] {});

    // ÷ 0300 ÷ 231A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u231A", new String[] {"\u231A"});

    // ÷ 0300 × 0308 ÷ 231A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u231A", new String[] {"\u231A"});

    // ÷ 0300 ÷ 0020 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0020", new String[] {});

    // ÷ 0300 × 0308 ÷ 0020 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0020", new String[] {});

    // ÷ 0300 × 00AD ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] SOFT HYPHEN
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u00AD", new String[] {});

    // ÷ 0300 × 0308 × 00AD ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u00AD", new String[] {});

    // ÷ 0300 × 0300 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0300", new String[] {});

    // ÷ 0300 × 0308 × 0300 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0300", new String[] {});

    // ÷ 0300 × 200D ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u200D", new String[] {});

    // ÷ 0300 × 0308 × 200D ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u200D", new String[] {});

    // ÷ 0300 ÷ 0061 × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0300 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 0300 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0300 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 0300 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0300 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 0300 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN
    // SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0300 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 0300 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0300 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 0300 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0300 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 0300 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0300 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 0300 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0300 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 0300 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) ÷ [999.0] DIGIT
    // ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0300 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] COMBINING GRAVE ACCENT (Extend_FE) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0300\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 200D ÷ 0001 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0001", new String[] {});

    // ÷ 200D × 0308 ÷ 0001 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0001", new String[] {});

    // ÷ 200D ÷ 000D ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\r", new String[] {});

    // ÷ 200D × 0308 ÷ 000D ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\r", new String[] {});

    // ÷ 200D ÷ 000A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\n", new String[] {});

    // ÷ 200D × 0308 ÷ 000A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\n", new String[] {});

    // ÷ 200D ÷ 000B ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u000B", new String[] {});

    // ÷ 200D × 0308 ÷ 000B ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u000B", new String[] {});

    // ÷ 200D ÷ 3031 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u3031", new String[] {"\u3031"});

    // ÷ 200D × 0308 ÷ 3031 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u3031", new String[] {"\u3031"});

    // ÷ 200D ÷ 0041 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0041", new String[] {"\u0041"});

    // ÷ 200D × 0308 ÷ 0041 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0041", new String[] {"\u0041"});

    // ÷ 200D ÷ 003A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u003A", new String[] {});

    // ÷ 200D × 0308 ÷ 003A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u003A", new String[] {});

    // ÷ 200D ÷ 002C ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u002C", new String[] {});

    // ÷ 200D × 0308 ÷ 002C ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u002C", new String[] {});

    // ÷ 200D ÷ 002E ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u002E", new String[] {});

    // ÷ 200D × 0308 ÷ 002E ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u002E", new String[] {});

    // ÷ 200D ÷ 0030 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0030", new String[] {"\u0030"});

    // ÷ 200D × 0308 ÷ 0030 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0030", new String[] {"\u0030"});

    // ÷ 200D ÷ 005F ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u005F", new String[] {});

    // ÷ 200D × 0308 ÷ 005F ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u005F", new String[] {});

    // ÷ 200D ÷ 1F1E6 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\uD83C\uDDE6", new String[] {});

    // ÷ 200D × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\uD83C\uDDE6", new String[] {});

    // ÷ 200D ÷ 05D0 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u05D0", new String[] {"\u05D0"});

    // ÷ 200D × 0308 ÷ 05D0 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u05D0", new String[] {"\u05D0"});

    // ÷ 200D ÷ 0022 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\"", new String[] {});

    // ÷ 200D × 0308 ÷ 0022 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\"", new String[] {});

    // ÷ 200D ÷ 0027 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0027", new String[] {});

    // ÷ 200D × 0308 ÷ 0027 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0027", new String[] {});

    // ÷ 200D × 231A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [3.3] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u231A", new String[] {"\u200D\u231A"});

    // ÷ 200D × 0308 ÷ 231A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u231A", new String[] {"\u231A"});

    // ÷ 200D ÷ 0020 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0020", new String[] {});

    // ÷ 200D × 0308 ÷ 0020 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0020", new String[] {});

    // ÷ 200D × 00AD ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u00AD", new String[] {});

    // ÷ 200D × 0308 × 00AD ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u00AD", new String[] {});

    // ÷ 200D × 0300 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0300", new String[] {});

    // ÷ 200D × 0308 × 0300 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0300", new String[] {});

    // ÷ 200D × 200D ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u200D", new String[] {});

    // ÷ 200D × 0308 × 200D ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u200D", new String[] {});

    // ÷ 200D ÷ 0061 × 2060 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 200D × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0061\u2060", new String[] {"\u0061\u2060"});

    // ÷ 200D ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0061\u003A", new String[] {"\u0061"});

    // ÷ 200D × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0061\u003A", new String[] {"\u0061"});

    // ÷ 200D ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0061\u0027", new String[] {"\u0061"});

    // ÷ 200D × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0061\u0027", new String[] {"\u0061"});

    // ÷ 200D ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 200D × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0061\u0027\u2060", new String[] {"\u0061"});

    // ÷ 200D ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0061\u002C", new String[] {"\u0061"});

    // ÷ 200D × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0061\u002C", new String[] {"\u0061"});

    // ÷ 200D ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0031\u003A", new String[] {"\u0031"});

    // ÷ 200D × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0031\u003A", new String[] {"\u0031"});

    // ÷ 200D ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0031\u0027", new String[] {"\u0031"});

    // ÷ 200D × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0031\u0027", new String[] {"\u0031"});

    // ÷ 200D ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0031\u002C", new String[] {"\u0031"});

    // ÷ 200D × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0031\u002C", new String[] {"\u0031"});

    // ÷ 200D ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 200D × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u200D\u0308\u0031\u002E\u2060", new String[] {"\u0031"});

    // ÷ 0061 × 2060 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0001", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0001", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\r", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\r", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\n", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\n", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u000B", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u000B", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u3031", new String[] {"\u0061\u2060", "\u3031"});

    // ÷ 0061 × 2060 × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u3031", new String[] {"\u0061\u2060\u0308", "\u3031"});

    // ÷ 0061 × 2060 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [5.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0041", new String[] {"\u0061\u2060\u0041"});

    // ÷ 0061 × 2060 × 0308 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN CAPITAL LETTER A (ALetter)
    // ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0041", new String[] {"\u0061\u2060\u0308\u0041"});

    // ÷ 0061 × 2060 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u003A", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u003A", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u002C", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u002C", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u002E", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u002E", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 × 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [9.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0030", new String[] {"\u0061\u2060\u0030"});

    // ÷ 0061 × 2060 × 0308 × 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0030", new String[] {"\u0061\u2060\u0308\u0030"});

    // ÷ 0061 × 2060 × 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u005F", new String[] {"\u0061\u2060\u005F"});

    // ÷ 0061 × 2060 × 0308 × 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u005F", new String[] {"\u0061\u2060\u0308\u005F"});

    // ÷ 0061 × 2060 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\uD83C\uDDE6", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\uD83C\uDDE6", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 × 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [5.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u05D0", new String[] {"\u0061\u2060\u05D0"});

    // ÷ 0061 × 2060 × 0308 × 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u05D0", new String[] {"\u0061\u2060\u0308\u05D0"});

    // ÷ 0061 × 2060 ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\"", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\"", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0027", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0027", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u231A", new String[] {"\u0061\u2060", "\u231A"});

    // ÷ 0061 × 2060 × 0308 ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u231A", new String[] {"\u0061\u2060\u0308", "\u231A"});

    // ÷ 0061 × 2060 ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0020", new String[] {"\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0308\u0020", new String[] {"\u0061\u2060\u0308"});

    // ÷ 0061 × 2060 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u00AD", new String[] {"\u0061\u2060\u00AD"});

    // ÷ 0061 × 2060 × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u00AD", new String[] {"\u0061\u2060\u0308\u00AD"});

    // ÷ 0061 × 2060 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0300", new String[] {"\u0061\u2060\u0300"});

    // ÷ 0061 × 2060 × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0300", new String[] {"\u0061\u2060\u0308\u0300"});

    // ÷ 0061 × 2060 × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u200D", new String[] {"\u0061\u2060\u200D"});

    // ÷ 0061 × 2060 × 0308 × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u200D", new String[] {"\u0061\u2060\u0308\u200D"});

    // ÷ 0061 × 2060 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0061\u2060", new String[] {"\u0061\u2060\u0061\u2060"});

    // ÷ 0061 × 2060 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u2060\u0308\u0061\u2060",
        new String[] {"\u0061\u2060\u0308\u0061\u2060"});

    // ÷ 0061 × 2060 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0061\u003A", new String[] {"\u0061\u2060\u0061"});

    // ÷ 0061 × 2060 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0061\u003A", new String[] {"\u0061\u2060\u0308\u0061"});

    // ÷ 0061 × 2060 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0061\u0027", new String[] {"\u0061\u2060\u0061"});

    // ÷ 0061 × 2060 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0061\u0027", new String[] {"\u0061\u2060\u0308\u0061"});

    // ÷ 0061 × 2060 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0061\u0027\u2060", new String[] {"\u0061\u2060\u0061"});

    // ÷ 0061 × 2060 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u2060\u0308\u0061\u0027\u2060",
        new String[] {"\u0061\u2060\u0308\u0061"});

    // ÷ 0061 × 2060 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [5.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0061\u002C", new String[] {"\u0061\u2060\u0061"});

    // ÷ 0061 × 2060 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0061\u002C", new String[] {"\u0061\u2060\u0308\u0061"});

    // ÷ 0061 × 2060 × 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0031\u003A", new String[] {"\u0061\u2060\u0031"});

    // ÷ 0061 × 2060 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0031\u003A", new String[] {"\u0061\u2060\u0308\u0031"});

    // ÷ 0061 × 2060 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0031\u0027", new String[] {"\u0061\u2060\u0031"});

    // ÷ 0061 × 2060 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0031\u0027", new String[] {"\u0061\u2060\u0308\u0031"});

    // ÷ 0061 × 2060 × 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER
    // (Format_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u2060\u0031\u002C", new String[] {"\u0061\u2060\u0031"});

    // ÷ 0061 × 2060 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0308\u0031\u002C", new String[] {"\u0061\u2060\u0308\u0031"});

    // ÷ 0061 × 2060 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) × [9.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD
    // JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u2060\u0031\u002E\u2060", new String[] {"\u0061\u2060\u0031"});

    // ÷ 0061 × 2060 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [9.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u2060\u0308\u0031\u002E\u2060",
        new String[] {"\u0061\u2060\u0308\u0031"});

    // ÷ 0061 ÷ 003A ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0001", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0001", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\r", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\r", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\n", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\n", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u000B", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u000B", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u3031", new String[] {"\u0061", "\u3031"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u3031", new String[] {"\u0061", "\u3031"});

    // ÷ 0061 × 003A × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) ×
    // [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0041", new String[] {"\u0061\u003A\u0041"});

    // ÷ 0061 × 003A × 0308 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN CAPITAL LETTER A (ALetter)
    // ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u003A\u0308\u0041", new String[] {"\u0061\u003A\u0308\u0041"});

    // ÷ 0061 ÷ 003A ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u003A", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u003A", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u002C", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u002C", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u002E", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u002E", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0030", new String[] {"\u0061", "\u0030"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0030", new String[] {"\u0061", "\u0030"});

    // ÷ 0061 ÷ 003A ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u005F", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u005F", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\uD83C\uDDE6", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\uD83C\uDDE6", new String[] {"\u0061"});

    // ÷ 0061 × 003A × 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON (MidLetter) ×
    // [7.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u05D0", new String[] {"\u0061\u003A\u05D0"});

    // ÷ 0061 × 003A × 0308 × 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u003A\u0308\u05D0", new String[] {"\u0061\u003A\u0308\u05D0"});

    // ÷ 0061 ÷ 003A ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\"", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\"", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0027", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0027", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u231A", new String[] {"\u0061", "\u231A"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u231A", new String[] {"\u0061", "\u231A"});

    // ÷ 0061 ÷ 003A ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0020", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0020", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u00AD", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u00AD", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0300", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0300", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter)
    // × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u200D", new String[] {"\u0061"});

    // ÷ 0061 ÷ 003A × 0308 × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u200D", new String[] {"\u0061"});

    // ÷ 0061 × 003A × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u003A\u0061\u2060", new String[] {"\u0061\u003A\u0061\u2060"});

    // ÷ 0061 × 003A × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u003A\u0308\u0061\u2060",
        new String[] {"\u0061\u003A\u0308\u0061\u2060"});

    // ÷ 0061 × 003A × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0061\u003A", new String[] {"\u0061\u003A\u0061"});

    // ÷ 0061 × 003A × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u003A\u0308\u0061\u003A", new String[] {"\u0061\u003A\u0308\u0061"});

    // ÷ 0061 × 003A × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0061\u0027", new String[] {"\u0061\u003A\u0061"});

    // ÷ 0061 × 003A × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u003A\u0308\u0061\u0027", new String[] {"\u0061\u003A\u0308\u0061"});

    // ÷ 0061 × 003A × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u003A\u0061\u0027\u2060", new String[] {"\u0061\u003A\u0061"});

    // ÷ 0061 × 003A × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [6.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u003A\u0308\u0061\u0027\u2060",
        new String[] {"\u0061\u003A\u0308\u0061"});

    // ÷ 0061 × 003A × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0061\u002C", new String[] {"\u0061\u003A\u0061"});

    // ÷ 0061 × 003A × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u003A\u0308\u0061\u002C", new String[] {"\u0061\u003A\u0308\u0061"});

    // ÷ 0061 ÷ 003A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0031\u003A", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0031\u003A", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0031\u0027", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0031\u0027", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0031\u002C", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0308\u0031\u002C", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD
    // JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u0031\u002E\u2060", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COLON (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u003A\u0308\u0031\u002E\u2060", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0001", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0001", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\r", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\r", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\n", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\n", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u000B", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u000B", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u3031", new String[] {"\u0061", "\u3031"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u3031", new String[] {"\u0061", "\u3031"});

    // ÷ 0061 × 0027 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0041", new String[] {"\u0061\u0027\u0041"});

    // ÷ 0061 × 0027 × 0308 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u0308\u0041", new String[] {"\u0061\u0027\u0308\u0041"});

    // ÷ 0061 ÷ 0027 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u003A", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u003A", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u002C", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u002C", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u002E", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u002E", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0030", new String[] {"\u0061", "\u0030"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0030", new String[] {"\u0061", "\u0030"});

    // ÷ 0061 ÷ 0027 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u005F", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u005F", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\uD83C\uDDE6", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL
    // INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\uD83C\uDDE6", new String[] {"\u0061"});

    // ÷ 0061 × 0027 × 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [7.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u05D0", new String[] {"\u0061\u0027\u05D0"});

    // ÷ 0061 × 0027 × 0308 × 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u0308\u05D0", new String[] {"\u0061\u0027\u0308\u05D0"});

    // ÷ 0061 ÷ 0027 ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\"", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\"", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0027", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0027", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u231A", new String[] {"\u0061", "\u231A"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u231A", new String[] {"\u0061", "\u231A"});

    // ÷ 0061 ÷ 0027 ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0020", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0020", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u00AD", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u00AD", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0300", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0300", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u200D", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 0308 × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u200D", new String[] {"\u0061"});

    // ÷ 0061 × 0027 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u0061\u2060", new String[] {"\u0061\u0027\u0061\u2060"});

    // ÷ 0061 × 0027 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER
    // A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u0308\u0061\u2060",
        new String[] {"\u0061\u0027\u0308\u0061\u2060"});

    // ÷ 0061 × 0027 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0061\u003A", new String[] {"\u0061\u0027\u0061"});

    // ÷ 0061 × 0027 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER
    // A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u0308\u0061\u003A", new String[] {"\u0061\u0027\u0308\u0061"});

    // ÷ 0061 × 0027 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0061\u0027", new String[] {"\u0061\u0027\u0061"});

    // ÷ 0061 × 0027 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER
    // A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u0308\u0061\u0027", new String[] {"\u0061\u0027\u0308\u0061"});

    // ÷ 0061 × 0027 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u0061\u0027\u2060", new String[] {"\u0061\u0027\u0061"});

    // ÷ 0061 × 0027 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [6.0] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u0308\u0061\u0027\u2060",
        new String[] {"\u0061\u0027\u0308\u0061"});

    // ÷ 0061 × 0027 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0061\u002C", new String[] {"\u0061\u0027\u0061"});

    // ÷ 0061 × 0027 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [7.0] LATIN SMALL LETTER
    // A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u0308\u0061\u002C", new String[] {"\u0061\u0027\u0308\u0061"});

    // ÷ 0061 ÷ 0027 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0031\u003A", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0031\u003A", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0031\u0027", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0031\u0027", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0031\u002C", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0308\u0031\u002C", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u0031\u002E\u2060", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u0308\u0031\u002E\u2060", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0001", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0001", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\r", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\r", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\n", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\n", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u000B", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u000B", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u3031", new String[] {"\u0061", "\u3031"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u3031", new String[] {"\u0061", "\u3031"});

    // ÷ 0061 × 0027 × 2060 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0041", new String[] {"\u0061\u0027\u2060\u0041"});

    // ÷ 0061 × 0027 × 2060 × 0308 × 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0308\u0041",
        new String[] {"\u0061\u0027\u2060\u0308\u0041"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u003A", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u003A", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u002C", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u002C", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u002E", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u002E", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0030", new String[] {"\u0061", "\u0030"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0030", new String[] {"\u0061", "\u0030"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u005F", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u005F", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\uD83C\uDDE6", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\uD83C\uDDE6", new String[] {"\u0061"});

    // ÷ 0061 × 0027 × 2060 × 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [7.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u05D0", new String[] {"\u0061\u0027\u2060\u05D0"});

    // ÷ 0061 × 0027 × 2060 × 0308 × 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [7.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0308\u05D0",
        new String[] {"\u0061\u0027\u2060\u0308\u05D0"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\"", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\"", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0027", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0027", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u231A", new String[] {"\u0061", "\u231A"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u231A", new String[] {"\u0061", "\u231A"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0020", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0020", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u00AD", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u00AD", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0300", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u0300", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u200D", new String[] {"\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0308\u200D", new String[] {"\u0061"});

    // ÷ 0061 × 0027 × 2060 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0061\u2060",
        new String[] {"\u0061\u0027\u2060\u0061\u2060"});

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [6.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0308\u0061\u2060",
        new String[] {"\u0061\u0027\u2060\u0308\u0061\u2060"});

    // ÷ 0061 × 0027 × 2060 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0061\u003A", new String[] {"\u0061\u0027\u2060\u0061"});

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [6.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0308\u0061\u003A",
        new String[] {"\u0061\u0027\u2060\u0308\u0061"});

    // ÷ 0061 × 0027 × 2060 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0061\u0027", new String[] {"\u0061\u0027\u2060\u0061"});

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [6.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0308\u0061\u0027",
        new String[] {"\u0061\u0027\u2060\u0308\u0061"});

    // ÷ 0061 × 0027 × 2060 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [6.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0061\u0027\u2060",
        new String[] {"\u0061\u0027\u2060\u0061"});

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter)
    // × [6.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0308\u0061\u0027\u2060",
        new String[] {"\u0061\u0027\u2060\u0308\u0061"});

    // ÷ 0061 × 0027 × 2060 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [6.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [7.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0061\u002C", new String[] {"\u0061\u0027\u2060\u0061"});

    // ÷ 0061 × 0027 × 2060 × 0308 × 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [6.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) × [7.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0027\u2060\u0308\u0061\u002C",
        new String[] {"\u0061\u0027\u2060\u0308\u0061"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0031\u003A", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0308\u0031\u003A", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0031\u0027", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0308\u0031\u0027", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0027\u2060\u0031\u002C", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0308\u0031\u002C", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 2060 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0031\u002E\u2060", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 0027 × 2060 × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u0027\u2060\u0308\u0031\u002E\u2060", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0001", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0001 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0001", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\r", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 000D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\r", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\n", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 000A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\n", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u000B", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 000B ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u000B", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u3031", new String[] {"\u0061", "\u3031"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 3031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u3031", new String[] {"\u0061", "\u3031"});

    // ÷ 0061 ÷ 002C ÷ 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0041", new String[] {"\u0061", "\u0041"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0041 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0041", new String[] {"\u0061", "\u0041"});

    // ÷ 0061 ÷ 002C ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u003A", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u003A", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u002C", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u002C", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u002E", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 002E ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u002E", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0030", new String[] {"\u0061", "\u0030"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0030 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0030", new String[] {"\u0061", "\u0030"});

    // ÷ 0061 ÷ 002C ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u005F", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 005F ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u005F", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\uD83C\uDDE6", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A
    // (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\uD83C\uDDE6", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u05D0", new String[] {"\u0061", "\u05D0"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 05D0 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u05D0", new String[] {"\u0061", "\u05D0"});

    // ÷ 0061 ÷ 002C ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\"", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0022 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\"", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0027", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0027", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u231A", new String[] {"\u0061", "\u231A"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 231A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u231A", new String[] {"\u0061", "\u231A"});

    // ÷ 0061 ÷ 002C ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0020", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0020 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0020", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u00AD", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 × 00AD ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u00AD", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0300", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 × 0300 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0300", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u200D", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C × 0308 × 200D ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u200D", new String[] {"\u0061"});

    // ÷ 0061 ÷ 002C ÷ 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u2060", new String[] {"\u0061", "\u0061\u2060"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u002C\u0308\u0061\u2060", new String[] {"\u0061", "\u0061\u2060"});

    // ÷ 0061 ÷ 002C ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u003A", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0061\u003A", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u0027", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0061\u0027", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u0027\u2060", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u002C\u0308\u0061\u0027\u2060", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0061\u002C", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0061\u002C", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0031\u003A", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0031\u003A", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0031\u0027", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0031\u0027", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0031\u002C", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0308\u0031\u002C", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD
    // JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u0031\u002E\u2060", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COMMA (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric)
    // ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0061\u002C\u0308\u0031\u002E\u2060", new String[] {"\u0061", "\u0031"});

    // ÷ 0031 ÷ 003A ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0001", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0001", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [3.2]
    // <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\r", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\r", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [3.2]
    // <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\n", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\n", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [3.2]
    // <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u000B", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u000B", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u3031", new String[] {"\u0031", "\u3031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u3031", new String[] {"\u0031", "\u3031"});

    // ÷ 0031 ÷ 003A ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0041", new String[] {"\u0031", "\u0041"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0041", new String[] {"\u0031", "\u0041"});

    // ÷ 0031 ÷ 003A ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u003A", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u003A", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u002C", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u002C", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 002E ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u002E", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 002E ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u002E", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0030", new String[] {"\u0031", "\u0030"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0030", new String[] {"\u0031", "\u0030"});

    // ÷ 0031 ÷ 003A ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u005F", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u005F", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\uD83C\uDDE6", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\uD83C\uDDE6", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 05D0 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u05D0", new String[] {"\u0031", "\u05D0"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 05D0 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u05D0", new String[] {"\u0031", "\u05D0"});

    // ÷ 0031 ÷ 003A ÷ 0022 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\"", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0022 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\"", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0027", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0027", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 231A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u231A", new String[] {"\u0031", "\u231A"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 231A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u231A", new String[] {"\u0031", "\u231A"});

    // ÷ 0031 ÷ 003A ÷ 0020 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0020", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0020 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0020", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0]
    // SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u00AD", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u00AD", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0]
    // COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0300", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0300", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 200D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) × [4.0]
    // ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u200D", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A × 0308 × 200D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u200D", new String[] {"\u0031"});

    // ÷ 0031 ÷ 003A ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u2060", new String[] {"\u0031", "\u0061\u2060"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u003A\u0308\u0061\u2060", new String[] {"\u0031", "\u0061\u2060"});

    // ÷ 0031 ÷ 003A ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u003A", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0061\u003A", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 003A ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u0027", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0061\u0027", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 003A ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u0027\u2060", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u003A\u0308\u0061\u0027\u2060", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 003A ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0061\u002C", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter)
    // ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0061\u002C", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 003A ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0031\u003A", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0031\u003A", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 ÷ 003A ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0031\u0027", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0031\u0027", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 ÷ 003A ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0031\u002C", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0308\u0031\u002C", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 ÷ 003A ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER
    // (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u0031\u002E\u2060", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 ÷ 003A × 0308 ÷ 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] DIGIT ONE (Numeric) ÷ [999.0]
    // FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u003A\u0308\u0031\u002E\u2060", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0001", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0001", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\r", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\r", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\n", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\n", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u000B", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u000B", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u3031", new String[] {"\u0031", "\u3031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK
    // (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u3031", new String[] {"\u0031", "\u3031"});

    // ÷ 0031 ÷ 0027 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0041", new String[] {"\u0031", "\u0041"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A
    // (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0041", new String[] {"\u0031", "\u0041"});

    // ÷ 0031 ÷ 0027 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u003A", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u003A", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u002C", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u002C", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 002E ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u002E", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 002E ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u002E", new String[] {"\u0031"});

    // ÷ 0031 × 0027 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE (Single_Quote) ×
    // [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0030", new String[] {"\u0031\u0027\u0030"});

    // ÷ 0031 × 0027 × 0308 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u0027\u0308\u0030", new String[] {"\u0031\u0027\u0308\u0030"});

    // ÷ 0031 ÷ 0027 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u005F", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u005F", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\uD83C\uDDE6", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL
    // LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\uD83C\uDDE6", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 05D0 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u05D0", new String[] {"\u0031", "\u05D0"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF
    // (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u05D0", new String[] {"\u0031", "\u05D0"});

    // ÷ 0031 ÷ 0027 ÷ 0022 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\"", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0022 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK
    // (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\"", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0027", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0027", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 231A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u231A", new String[] {"\u0031", "\u231A"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 231A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u231A", new String[] {"\u0031", "\u231A"});

    // ÷ 0031 ÷ 0027 ÷ 0020 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0020", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0020 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0020", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u00AD", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u00AD", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0300", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0300", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 200D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u200D", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 × 0308 × 200D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u200D", new String[] {"\u0031"});

    // ÷ 0031 ÷ 0027 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u2060", new String[] {"\u0031", "\u0061\u2060"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u0027\u0308\u0061\u2060", new String[] {"\u0031", "\u0061\u2060"});

    // ÷ 0031 ÷ 0027 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u003A", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0061\u003A", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 0027 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u0027", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0061\u0027", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 0027 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ×
    // [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u0027\u2060", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL
    // LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u0027\u0308\u0061\u0027\u2060", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 0027 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0061\u002C", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 0027 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A
    // (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0308\u0061\u002C", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 0027 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE
    // (Single_Quote) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0031\u003A", new String[] {"\u0031\u0027\u0031"});

    // ÷ 0031 × 0027 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u0027\u0308\u0031\u003A", new String[] {"\u0031\u0027\u0308\u0031"});

    // ÷ 0031 × 0027 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE
    // (Single_Quote) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0031\u0027", new String[] {"\u0031\u0027\u0031"});

    // ÷ 0031 × 0027 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u0027\u0308\u0031\u0027", new String[] {"\u0031\u0027\u0308\u0031"});

    // ÷ 0031 × 0027 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE
    // (Single_Quote) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u0027\u0031\u002C", new String[] {"\u0031\u0027\u0031"});

    // ÷ 0031 × 0027 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE
    // (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u0027\u0308\u0031\u002C", new String[] {"\u0031\u0027\u0308\u0031"});

    // ÷ 0031 × 0027 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] APOSTROPHE
    // (Single_Quote) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD
    // JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u0027\u0031\u002E\u2060", new String[] {"\u0031\u0027\u0031"});

    // ÷ 0031 × 0027 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0]
    // APOSTROPHE (Single_Quote) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE
    // (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u0027\u0308\u0031\u002E\u2060",
        new String[] {"\u0031\u0027\u0308\u0031"});

    // ÷ 0031 ÷ 002C ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0001", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0001", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [3.2]
    // <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\r", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\r", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [3.2] <LINE
    // FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\n", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\n", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [3.2] <LINE
    // TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u000B", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u000B", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u3031", new String[] {"\u0031", "\u3031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u3031", new String[] {"\u0031", "\u3031"});

    // ÷ 0031 ÷ 002C ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0041", new String[] {"\u0031", "\u0041"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0041", new String[] {"\u0031", "\u0041"});

    // ÷ 0031 ÷ 002C ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u003A", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u003A", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u002C", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u002C", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 002E ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u002E", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 002E ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u002E", new String[] {"\u0031"});

    // ÷ 0031 × 002C × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [11.0] DIGIT
    // ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0030", new String[] {"\u0031\u002C\u0030"});

    // ÷ 0031 × 002C × 0308 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002C\u0308\u0030", new String[] {"\u0031\u002C\u0308\u0030"});

    // ÷ 0031 ÷ 002C ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LOW
    // LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u005F", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u005F", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\uD83C\uDDE6", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\uD83C\uDDE6", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 05D0 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u05D0", new String[] {"\u0031", "\u05D0"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 05D0 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u05D0", new String[] {"\u0031", "\u05D0"});

    // ÷ 0031 ÷ 002C ÷ 0022 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\"", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0022 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\"", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0027", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0027", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 231A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u231A", new String[] {"\u0031", "\u231A"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 231A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u231A", new String[] {"\u0031", "\u231A"});

    // ÷ 0031 ÷ 002C ÷ 0020 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0020", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0020 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0020", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] SOFT
    // HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u00AD", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u00AD", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0]
    // COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0300", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0300", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 200D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) × [4.0] ZERO
    // WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u200D", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C × 0308 × 200D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ×
    // [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u200D", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002C ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u2060", new String[] {"\u0031", "\u0061\u2060"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum)
    // × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD
    // JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002C\u0308\u0061\u2060", new String[] {"\u0031", "\u0061\u2060"});

    // ÷ 0031 ÷ 002C ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u003A", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum)
    // × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0061\u003A", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002C ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u0027", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum)
    // × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0061\u0027", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002C ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum)
    // ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD
    // JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u0027\u2060", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002C\u0308\u0061\u0027\u2060", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002C ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0061\u002C", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002C × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum)
    // × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0308\u0061\u002C", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 002C × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) ×
    // [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0031\u003A", new String[] {"\u0031\u002C\u0031"});

    // ÷ 0031 × 002C × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum)
    // × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002C\u0308\u0031\u003A", new String[] {"\u0031\u002C\u0308\u0031"});

    // ÷ 0031 × 002C × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) ×
    // [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0031\u0027", new String[] {"\u0031\u002C\u0031"});

    // ÷ 0031 × 002C × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum)
    // × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002C\u0308\u0031\u0027", new String[] {"\u0031\u002C\u0308\u0031"});

    // ÷ 0031 × 002C × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum) ×
    // [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u0031\u002C", new String[] {"\u0031\u002C\u0031"});

    // ÷ 0031 × 002C × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum)
    // × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum)
    // ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002C\u0308\u0031\u002C", new String[] {"\u0031\u002C\u0308\u0031"});

    // ÷ 0031 × 002C × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA (MidNum)
    // × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE)
    // ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002C\u0031\u002E\u2060", new String[] {"\u0031\u002C\u0031"});

    // ÷ 0031 × 002C × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] COMMA
    // (MidNum) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u002C\u0308\u0031\u002E\u2060",
        new String[] {"\u0031\u002C\u0308\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0001", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0001 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // <START OF HEADING> (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0001", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\r", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 000D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2]
    // <CARRIAGE RETURN (CR)> (CR) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\r", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\n", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 000A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2]
    // <LINE FEED (LF)> (LF) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\n", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [3.2] <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u000B", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 000B ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [3.2]
    // <LINE TABULATION> (Newline) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u000B", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u3031", new String[] {"\u0031", "\u3031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 3031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // VERTICAL KANA REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u3031", new String[] {"\u0031", "\u3031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0041", new String[] {"\u0031", "\u0041"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0041 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0041", new String[] {"\u0031", "\u0041"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u003A", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u003A", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u002C", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u002C", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 002E ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u002E", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 002E ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // FULL STOP (MidNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u002E", new String[] {"\u0031"});

    // ÷ 0031 × 002E × 2060 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0030", new String[] {"\u0031\u002E\u2060\u0030"});

    // ÷ 0031 × 002E × 2060 × 0308 × 0030 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [11.0]
    // DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u002E\u2060\u0308\u0030",
        new String[] {"\u0031\u002E\u2060\u0308\u0030"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u005F", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 005F ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u005F", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI)
    // ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\uD83C\uDDE6", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 1F1E6 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // REGIONAL INDICATOR SYMBOL LETTER A (RI) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\uD83C\uDDE6", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 05D0 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u05D0", new String[] {"\u0031", "\u05D0"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 05D0 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u05D0", new String[] {"\u0031", "\u05D0"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0022 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\"", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0022 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // QUOTATION MARK (Double_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\"", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0027", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0027", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 231A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u231A", new String[] {"\u0031", "\u231A"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 231A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // WATCH (ExtPict) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u231A", new String[] {"\u0031", "\u231A"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0020 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0020", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0020 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷ [999.0]
    // SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0020", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) × [4.0] SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u00AD", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 × 00AD ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0]
    // SOFT HYPHEN (Format_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u00AD", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0300", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 × 0300 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0]
    // COMBINING GRAVE ACCENT (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u0300", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 200D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // × [4.0] WORD JOINER (Format_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u200D", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 × 200D ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0]
    // ZERO WIDTH JOINER (ZWJ_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0308\u200D", new String[] {"\u0031"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) × [4.0]
    // WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0061\u2060", new String[] {"\u0031", "\u0061\u2060"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0308\u0061\u2060", new String[] {"\u0031", "\u0061\u2060"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0061\u003A", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0308\u0061\u003A", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0061\u0027", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0308\u0061\u0027", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0061\u0027\u2060", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 ÷ 0027 × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0]
    // FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] APOSTROPHE (Single_Quote) × [4.0] WORD
    // JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0308\u0061\u0027\u2060", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002E × 2060 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u2060\u0061\u002C", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 ÷ 002E × 2060 × 0308 ÷ 0061 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ÷
    // [999.0] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0308\u0061\u002C", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 002E × 2060 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON
    // (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0031\u003A", new String[] {"\u0031\u002E\u2060\u0031"});

    // ÷ 0031 × 002E × 2060 × 0308 × 0031 ÷ 003A ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [11.0] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u002E\u2060\u0308\u0031\u003A",
        new String[] {"\u0031\u002E\u2060\u0308\u0031"});

    // ÷ 0031 × 002E × 2060 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0031\u0027", new String[] {"\u0031\u002E\u2060\u0031"});

    // ÷ 0031 × 002E × 2060 × 0308 × 0031 ÷ 0027 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [11.0] DIGIT ONE (Numeric) ÷ [999.0] APOSTROPHE (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u002E\u2060\u0308\u0031\u0027",
        new String[] {"\u0031\u002E\u2060\u0308\u0031"});

    // ÷ 0031 × 002E × 2060 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL STOP
    // (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA
    // (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0031\u002E\u2060\u0031\u002C", new String[] {"\u0031\u002E\u2060\u0031"});

    // ÷ 0031 × 002E × 2060 × 0308 × 0031 ÷ 002C ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) ×
    // [11.0] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u002E\u2060\u0308\u0031\u002C",
        new String[] {"\u0031\u002E\u2060\u0308\u0031"});

    // ÷ 0031 × 002E × 2060 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL
    // STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u002E\u2060\u0031\u002E\u2060",
        new String[] {"\u0031\u002E\u2060\u0031"});

    // ÷ 0031 × 002E × 2060 × 0308 × 0031 ÷ 002E × 2060 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [12.0]
    // FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE) × [4.0] COMBINING DIAERESIS (Extend_FE)
    // × [11.0] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) × [4.0] WORD JOINER (Format_FE)
    // ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u002E\u2060\u0308\u0031\u002E\u2060",
        new String[] {"\u0031\u002E\u2060\u0308\u0031"});

    // ÷ 000D × 000A ÷ 0061 ÷ 000A ÷ 0308 ÷  #  ÷ [0.2] <CARRIAGE RETURN (CR)> (CR) × [3.0] <LINE
    // FEED (LF)> (LF) ÷ [3.1] LATIN SMALL LETTER A (ALetter) ÷ [3.2] <LINE FEED (LF)> (LF) ÷ [3.1]
    // COMBINING DIAERESIS (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\r\n\u0061\n\u0308", new String[] {"\u0061"});

    // ÷ 0061 × 0308 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] COMBINING DIAERESIS
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0308", new String[] {"\u0061\u0308"});

    // ÷ 0020 × 200D ÷ 0646 ÷  #  ÷ [0.2] SPACE (WSegSpace) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷
    // [999.0] ARABIC LETTER NOON (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0020\u200D\u0646", new String[] {"\u0646"});

    // ÷ 0646 × 200D ÷ 0020 ÷  #  ÷ [0.2] ARABIC LETTER NOON (ALetter) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) ÷ [999.0] SPACE (WSegSpace) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0646\u200D\u0020", new String[] {"\u0646\u200D"});

    // ÷ 0041 × 0041 × 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [5.0] LATIN CAPITAL
    // LETTER A (ALetter) × [5.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u0041\u0041", new String[] {"\u0041\u0041\u0041"});

    // ÷ 0041 × 003A × 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [6.0] COLON (MidLetter)
    // × [7.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u003A\u0041", new String[] {"\u0041\u003A\u0041"});

    // ÷ 0041 ÷ 003A ÷ 003A ÷ 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0041\u003A\u003A\u0041", new String[] {"\u0041", "\u0041"});

    // ÷ 05D0 × 0027 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [7.1] APOSTROPHE
    // (Single_Quote) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\u0027", new String[] {"\u05D0\u0027"});

    // ÷ 05D0 × 0022 × 05D0 ÷  #  ÷ [0.2] HEBREW LETTER ALEF (Hebrew_Letter) × [7.2] QUOTATION MARK
    // (Double_Quote) × [7.3] HEBREW LETTER ALEF (Hebrew_Letter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u05D0\"\u05D0", new String[] {"\u05D0\"\u05D0"});

    // ÷ 0041 × 0030 × 0030 × 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [9.0] DIGIT ZERO
    // (Numeric) × [8.0] DIGIT ZERO (Numeric) × [10.0] LATIN CAPITAL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\u0041\u0030\u0030\u0041", new String[] {"\u0041\u0030\u0030\u0041"});

    // ÷ 0030 × 002C × 0030 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) × [12.0] COMMA (MidNum) × [11.0]
    // DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u002C\u0030", new String[] {"\u0030\u002C\u0030"});

    // ÷ 0030 ÷ 002C ÷ 002C ÷ 0030 ÷  #  ÷ [0.2] DIGIT ZERO (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ZERO (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0030\u002C\u002C\u0030", new String[] {"\u0030", "\u0030"});

    // ÷ 3031 × 3031 ÷  #  ÷ [0.2] VERTICAL KANA REPEAT MARK (Katakana) × [13.0] VERTICAL KANA
    // REPEAT MARK (Katakana) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u3031\u3031", new String[] {"\u3031\u3031"});

    // ÷ 0041 × 005F × 0030 × 005F × 3031 × 005F ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ZERO (Numeric) × [13.1] LOW LINE (ExtendNumLet)
    // × [13.2] VERTICAL KANA REPEAT MARK (Katakana) × [13.1] LOW LINE (ExtendNumLet) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0041\u005F\u0030\u005F\u3031\u005F",
        new String[] {"\u0041\u005F\u0030\u005F\u3031\u005F"});

    // ÷ 0041 × 005F × 005F × 0041 ÷  #  ÷ [0.2] LATIN CAPITAL LETTER A (ALetter) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN CAPITAL LETTER A (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer, "\u0041\u005F\u005F\u0041", new String[] {"\u0041\u005F\u005F\u0041"});

    // ÷ 1F1E6 × 1F1E7 ÷ 1F1E8 ÷ 0062 ÷  #  ÷ [0.2] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [15.0]
    // REGIONAL INDICATOR SYMBOL LETTER B (RI) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER C (RI) ÷
    // [999.0] LATIN SMALL LETTER B (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\uD83C\uDDE6\uD83C\uDDE7\uD83C\uDDE8\u0062",
        new String[] {"\uD83C\uDDE6\uD83C\uDDE7", "\u0062"});

    // ÷ 0061 ÷ 1F1E6 × 1F1E7 ÷ 1F1E8 ÷ 0062 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0]
    // REGIONAL INDICATOR SYMBOL LETTER A (RI) × [16.0] REGIONAL INDICATOR SYMBOL LETTER B (RI) ÷
    // [999.0] REGIONAL INDICATOR SYMBOL LETTER C (RI) ÷ [999.0] LATIN SMALL LETTER B (ALetter) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\uD83C\uDDE6\uD83C\uDDE7\uD83C\uDDE8\u0062",
        new String[] {"\u0061", "\uD83C\uDDE6\uD83C\uDDE7", "\u0062"});

    // ÷ 0061 ÷ 1F1E6 × 1F1E7 × 200D ÷ 1F1E8 ÷ 0062 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [16.0] REGIONAL INDICATOR SYMBOL LETTER B
    // (RI) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER C (RI) ÷
    // [999.0] LATIN SMALL LETTER B (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\uD83C\uDDE6\uD83C\uDDE7\u200D\uD83C\uDDE8\u0062",
        new String[] {"\u0061", "\uD83C\uDDE6\uD83C\uDDE7\u200D", "\u0062"});

    // ÷ 0061 ÷ 1F1E6 × 200D × 1F1E7 ÷ 1F1E8 ÷ 0062 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) × [16.0]
    // REGIONAL INDICATOR SYMBOL LETTER B (RI) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER C (RI) ÷
    // [999.0] LATIN SMALL LETTER B (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\uD83C\uDDE6\u200D\uD83C\uDDE7\uD83C\uDDE8\u0062",
        new String[] {"\u0061", "\uD83C\uDDE6\u200D\uD83C\uDDE7", "\u0062"});

    // ÷ 0061 ÷ 1F1E6 × 1F1E7 ÷ 1F1E8 × 1F1E9 ÷ 0062 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷
    // [999.0] REGIONAL INDICATOR SYMBOL LETTER A (RI) × [16.0] REGIONAL INDICATOR SYMBOL LETTER B
    // (RI) ÷ [999.0] REGIONAL INDICATOR SYMBOL LETTER C (RI) × [16.0] REGIONAL INDICATOR SYMBOL
    // LETTER D (RI) ÷ [999.0] LATIN SMALL LETTER B (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\uD83C\uDDE6\uD83C\uDDE7\uD83C\uDDE8\uD83C\uDDE9\u0062",
        new String[] {"\u0061", "\uD83C\uDDE6\uD83C\uDDE7", "\uD83C\uDDE8\uD83C\uDDE9", "\u0062"});

    // ÷ 1F476 × 1F3FF ÷ 1F476 ÷  #  ÷ [0.2] BABY (ExtPict) × [4.0] EMOJI MODIFIER FITZPATRICK
    // TYPE-6 (Extend_FE) ÷ [999.0] BABY (ExtPict) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\uD83D\uDC76\uD83C\uDFFF\uD83D\uDC76",
        new String[] {"\uD83D\uDC76\uD83C\uDFFF", "\uD83D\uDC76"});

    // ÷ 1F6D1 × 200D × 1F6D1 ÷  #  ÷ [0.2] OCTAGONAL SIGN (ExtPict) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) × [3.3] OCTAGONAL SIGN (ExtPict) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\uD83D\uDED1\u200D\uD83D\uDED1",
        new String[] {"\uD83D\uDED1\u200D\uD83D\uDED1"});

    // ÷ 0061 × 200D × 1F6D1 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) × [3.3] OCTAGONAL SIGN (ExtPict) ÷ [0.3]
    // Skipping this test because it conflicts with TR#51 v11.0 rules.

    // ÷ 2701 × 200D × 2701 ÷  #  ÷ [0.2] UPPER BLADE SCISSORS (Other) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) × [3.3] UPPER BLADE SCISSORS (Other) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u2701\u200D\u2701", new String[] {"\u2701\u200D\u2701"});

    // ÷ 0061 × 200D × 2701 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0] ZERO WIDTH JOINER
    // (ZWJ_FE) × [3.3] UPPER BLADE SCISSORS (Other) ÷ [0.3]
    // Skipping this test because it conflicts with TR#51 v11.0 rules.

    // ÷ 1F476 × 1F3FF × 0308 × 200D × 1F476 × 1F3FF ÷  #  ÷ [0.2] BABY (ExtPict) × [4.0] EMOJI
    // MODIFIER FITZPATRICK TYPE-6 (Extend_FE) × [4.0] COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO
    // WIDTH JOINER (ZWJ_FE) × [3.3] BABY (ExtPict) × [4.0] EMOJI MODIFIER FITZPATRICK TYPE-6
    // (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\uD83D\uDC76\uD83C\uDFFF\u0308\u200D\uD83D\uDC76\uD83C\uDFFF",
        new String[] {"\uD83D\uDC76\uD83C\uDFFF\u0308\u200D\uD83D\uDC76\uD83C\uDFFF"});

    // ÷ 1F6D1 × 1F3FF ÷  #  ÷ [0.2] OCTAGONAL SIGN (ExtPict) × [4.0] EMOJI MODIFIER FITZPATRICK
    // TYPE-6 (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer, "\uD83D\uDED1\uD83C\uDFFF", new String[] {"\uD83D\uDED1\uD83C\uDFFF"});

    // ÷ 200D × 1F6D1 × 1F3FF ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [3.3] OCTAGONAL SIGN
    // (ExtPict) × [4.0] EMOJI MODIFIER FITZPATRICK TYPE-6 (Extend_FE) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u200D\uD83D\uDED1\uD83C\uDFFF",
        new String[] {"\u200D\uD83D\uDED1\uD83C\uDFFF"});

    // ÷ 200D × 1F6D1 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [3.3] OCTAGONAL SIGN (ExtPict) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\uD83D\uDED1", new String[] {"\u200D\uD83D\uDED1"});

    // ÷ 200D × 1F6D1 ÷  #  ÷ [0.2] ZERO WIDTH JOINER (ZWJ_FE) × [3.3] OCTAGONAL SIGN (ExtPict) ÷
    // [0.3]
    assertAnalyzesTo(analyzer, "\u200D\uD83D\uDED1", new String[] {"\u200D\uD83D\uDED1"});

    // ÷ 1F6D1 ÷ 1F6D1 ÷  #  ÷ [0.2] OCTAGONAL SIGN (ExtPict) ÷ [999.0] OCTAGONAL SIGN (ExtPict) ÷
    // [0.3]
    assertAnalyzesTo(
        analyzer, "\uD83D\uDED1\uD83D\uDED1", new String[] {"\uD83D\uDED1", "\uD83D\uDED1"});

    // ÷ 0061 × 0308 × 200D × 0308 × 0062 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) × [4.0]
    // COMBINING DIAERESIS (Extend_FE) × [4.0] ZERO WIDTH JOINER (ZWJ_FE) × [4.0] COMBINING
    // DIAERESIS (Extend_FE) × [5.0] LATIN SMALL LETTER B (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u0308\u200D\u0308\u0062",
        new String[] {"\u0061\u0308\u200D\u0308\u0062"});

    // ÷ 0061 ÷ 0020 × 0020 ÷ 0062 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] SPACE
    // (WSegSpace) × [3.4] SPACE (WSegSpace) ÷ [999.0] LATIN SMALL LETTER B (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u0020\u0020\u0062", new String[] {"\u0061", "\u0062"});

    // ÷ 0031 ÷ 003A ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u003A\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 003A ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u003A\u003A\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 003A ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u003A\u003A\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 003A ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u003A\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 003A ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u003A\u003A\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 003A ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u003A\u003A\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0031 ÷ 003A ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u002E\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 003A ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u003A\u002E\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 003A ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u003A\u002E\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 003A ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u002E\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 003A ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u003A\u002E\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 003A ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u003A\u002E\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0031 ÷ 003A ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u002C\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 003A ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u003A\u002C\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 003A ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u003A\u002C\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 003A ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u003A\u002C\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 003A ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u003A\u002C\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 003A ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0]
    // COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u003A\u002C\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0031 ÷ 002E ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u003A\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 002E ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002E\u003A\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 002E ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002E\u003A\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 002E ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u003A\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 002E ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002E\u003A\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 002E ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002E\u003A\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0031 ÷ 002E ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u002E\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 002E ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] FULL
    // STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002E\u002E\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 002E ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002E\u002E\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 002E ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u002E\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 002E ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] FULL
    // STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002E\u002E\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 002E ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002E\u002E\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0031 ÷ 002E ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u002C\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 002E ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002E\u002C\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 002E ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002E\u002C\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 002E ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet)
    // ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002E\u002C\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 002E ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002E\u002C\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 002E ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002E\u002C\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0031 ÷ 002C ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u003A\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 002C ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002C\u003A\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 002C ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002C\u003A\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 002C ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u003A\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 002C ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002C\u003A\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 002C ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002C\u003A\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0031 ÷ 002C ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u002E\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 002C ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002C\u002E\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 002C ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002C\u002E\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 002C ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u002E\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 002C ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002C\u002E\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 002C ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002C\u002E\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0031 ÷ 002C ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u002C\u0031", new String[] {"\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0031 ÷ 002C ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] COMMA (MidNum)
    // ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002C\u002C\u0031",
        new String[] {"\u0031\u005F\u0031", "\u0031"});

    // ÷ 0031 × 005F × 0061 ÷ 002C ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002C\u002C\u0031",
        new String[] {"\u0031\u005F\u0061", "\u0031"});

    // ÷ 0031 ÷ 002C ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0031\u002C\u002C\u0061", new String[] {"\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0031 ÷ 002C ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷ [999.0] COMMA (MidNum)
    // ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0031\u002C\u002C\u0061",
        new String[] {"\u0031\u005F\u0031", "\u0061"});

    // ÷ 0031 × 005F × 0061 ÷ 002C ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] DIGIT ONE (Numeric) × [13.1] LOW LINE
    // (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0]
    // COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0031\u005F\u0061\u002C\u002C\u0061",
        new String[] {"\u0031\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 003A ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u003A\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 003A ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u003A\u003A\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 003A ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u003A\u003A\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u003A\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 003A ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u003A\u003A\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 003A ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u003A\u003A\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 003A ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u002E\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 003A ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u003A\u002E\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 003A ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u003A\u002E\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u002E\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 003A ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u003A\u002E\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 003A ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u003A\u002E\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 003A ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u002C\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 003A ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u003A\u002C\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 003A ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u003A\u002C\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 003A ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u003A\u002C\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 003A ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COLON (MidLetter) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u003A\u002C\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 003A ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COLON
    // (MidLetter) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u003A\u002C\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 002E ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002E\u003A\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 002E ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002E\u003A\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 002E ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002E\u003A\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 002E ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002E\u003A\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 002E ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002E\u003A\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 002E ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002E\u003A\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 002E ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002E\u002E\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 002E ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002E\u002E\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 002E ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002E\u002E\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 002E ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002E\u002E\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 002E ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002E\u002E\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 002E ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002E\u002E\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 002E ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002E\u002C\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 002E ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002E\u002C\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 002E ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002E\u002C\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 002E ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002E\u002C\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 002E ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] FULL STOP (MidNumLet) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002E\u002C\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 002E ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] FULL STOP
    // (MidNumLet) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002E\u002C\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u003A\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 002C ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002C\u003A\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 002C ÷ 003A ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] COLON (MidLetter) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002C\u003A\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u003A\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 002C ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002C\u003A\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 002C ÷ 003A ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] COLON (MidLetter) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002C\u003A\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u002E\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 002C ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002C\u002E\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 002C ÷ 002E ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002C\u002E\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u002E\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 002C ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002C\u002E\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 002C ÷ 002E ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] FULL STOP (MidNumLet) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002C\u002E\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});

    // ÷ 0061 ÷ 002C ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u002C\u0031", new String[] {"\u0061", "\u0031"});

    // ÷ 0061 × 005F × 0031 ÷ 002C ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002C\u002C\u0031",
        new String[] {"\u0061\u005F\u0031", "\u0031"});

    // ÷ 0061 × 005F × 0061 ÷ 002C ÷ 002C ÷ 0031 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] COMMA (MidNum) ÷ [999.0] DIGIT ONE (Numeric) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002C\u002C\u0031",
        new String[] {"\u0061\u005F\u0061", "\u0031"});

    // ÷ 0061 ÷ 002C ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(analyzer, "\u0061\u002C\u002C\u0061", new String[] {"\u0061", "\u0061"});

    // ÷ 0061 × 005F × 0031 ÷ 002C ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] DIGIT ONE (Numeric) ÷ [999.0] COMMA (MidNum) ÷
    // [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0031\u002C\u002C\u0061",
        new String[] {"\u0061\u005F\u0031", "\u0061"});

    // ÷ 0061 × 005F × 0061 ÷ 002C ÷ 002C ÷ 0061 ÷  #  ÷ [0.2] LATIN SMALL LETTER A (ALetter) ×
    // [13.1] LOW LINE (ExtendNumLet) × [13.2] LATIN SMALL LETTER A (ALetter) ÷ [999.0] COMMA
    // (MidNum) ÷ [999.0] COMMA (MidNum) ÷ [999.0] LATIN SMALL LETTER A (ALetter) ÷ [0.3]
    assertAnalyzesTo(
        analyzer,
        "\u0061\u005F\u0061\u002C\u002C\u0061",
        new String[] {"\u0061\u005F\u0061", "\u0061"});
  }
}
