# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Reporting functions for MAAS test."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = [
    "create_launchpad_blob",
    "upload_blob",
    ]


from datetime import datetime
import email.mime
from io import BytesIO
import logging
import os
from urllib2 import (
    build_opener,
    HTTPSHandler,
    ProxyHandler,
    Request,
    )

from apiclient import multipart
from maastest.utils import DEFAULT_LOG_DIR


# Used to disable proxies when opening the connection to
# Launchpad (see the XXX re bug 1267443 below).
NO_PROXY_HANDLER = ProxyHandler(proxies={})


# This is largely cargo-culted from apiclient.multipart because
# Launchpad only pays attention to the first Content-Disposition header
# on an attachment, and if we want the content to actually be attached
# to the bug rather than lost in the ether we have to make it
# Content-Disposition: attachment. By default,
# apiclient.multipart.make_file_payload() gives the payload a
# Content-Disposition of "form-data".
def make_file_payload(name, content):
    payload = email.mime.application.MIMEApplication(content)
    payload.add_header(
        "Content-Disposition", "attachment", name=name, filename=name)
    names = name, getattr(content, "name", None)
    payload.set_type(multipart.get_content_type(*names))
    return payload


def create_launchpad_blob(test_results, test_succeeded):
    """Create an RFC822-formatted blob of data to upload to Launchpad.
    """
    launchpad_form_data = multipart.build_multipart_message([])
    launchpad_form_data['private'] = 'yes'
    launchpad_form_data['subscribers'] = "private-canonical-maas"

    if test_succeeded:
        launchpad_form_data['subject'] = "maas-test success"
        launchpad_form_data['tags'] = 'success'
    else:
        launchpad_form_data['subject'] = "maas-test failure"
        launchpad_form_data['tags'] = 'failure'

    results_payload = make_file_payload(
        'maas-test.log', test_results.encode('utf-8', 'replace'))
    launchpad_form_data.attach(results_payload)

    return launchpad_form_data


# And this is largely cargo-culted from apport.crashdb_impl.launchpad,
# because it does all the form-submission dance that we need to do to
# make this actually work.
def build_upload_mime_multipart(blob):
    launchpad_form_data = email.mime.multipart.MIMEMultipart()

    submit = email.mime.Text.MIMEText('1')
    submit.add_header(
        'Content-Disposition', 'form-data; name="FORM_SUBMIT"')
    launchpad_form_data.attach(submit)

    form_blob_field = email.mime.Base.MIMEBase('application', 'octet-stream')
    form_blob_field.add_header(
        'Content-Disposition', 'form-data; name="field.blob"; filename="x"')
    form_blob_field.set_payload(blob.as_string().encode('ascii'))
    launchpad_form_data.attach(form_blob_field)

    return launchpad_form_data


def build_upload_request(url, launchpad_form_data):
    # These next three lines are cargo-culted from
    # apport.crashdb_impl.launchpad. Launchpad won't recognise the blob
    # as something it can parse if we supress the leading headers.
    data_flat = BytesIO()
    generator = email.generator.Generator(data_flat, mangle_from_=False)
    generator.flatten(launchpad_form_data)

    # do the request; we need to explicitly set the content type here, as it
    # defaults to x-www-form-urlencoded
    request = Request(url, data_flat.getvalue())
    request.add_header(
        'Content-Type',
        'multipart/form-data; boundary=' + launchpad_form_data.get_boundary())

    return request


def upload_blob(blob, hostname='launchpad.net'):
    """Upload blob (file-like object) to Launchpad.

    :param blob:
    """
    token = None
    url = 'https://%s/+storeblob' % hostname
    form = build_upload_mime_multipart(blob)
    request = build_upload_request(url, form)
    # XXX 2014-01-09 gmb bug=1267443:
    #     We explicitly disable proxies here because connecting to
    #     Launchpad fails if https_proxy is set in the environment (see
    #     linked bug).
    opener = build_opener(HTTPSHandler, NO_PROXY_HANDLER)
    try:
        result = opener.open(request)
    except Exception:
        logging.exception("Unable to connect to Launchpad.")
        return None
    else:
        token = result.info().get('X-Launchpad-Blob-Token')
        return token


def write_test_results(results, log_file_name=None, log_dir=None, now=None):
    """Write maas-test results to a file.

    :param results: The results from maas-test.
    :param log_file_name: The filename to which to write the results
        (used for testing only).
    :param log_dir: The directory under which to create `log_file_name`.
    :param now: The current date/time (used for testing only).
    :type now: datetime.datetime.
    """
    if now is None:
        now = datetime.utcnow()

    if log_dir is None:
        log_dir = DEFAULT_LOG_DIR
    if not os.path.exists(log_dir):
        os.makedirs(log_dir)

    if log_file_name is None:
        log_file_name = (
            "maas-test.%s.log" % now.strftime("%Y-%m-%d_%H:%M:%S"))

    log_file_path = os.path.join(log_dir, log_file_name)
    with open(log_file_path, 'w') as log_file:
        log_file.write(results)

    logging.info("Test log saved to %s." % log_file_path)


def report_test_results(results, test_succeeded):
    """Handle the reporting of maas-test results.

    :param results: The results to report.
    :param test_succeeded: True if maas-test passed, False otherwise.
    """
    blob = create_launchpad_blob(results, test_succeeded)
    blob_token = upload_blob(blob)
    if blob_token is not None:
        logging.info("Test results uploaded to Launchpad.")
        logging.info(
            "Visit https://bugs.launchpad.net/maas-test-reports/+filebug/%s "
            "to file a bug and complete the maas-test reporting process."
            % blob_token)
    else:
        logging.info("Unable to store test results on Launchpad.")
