#include "video_encoder.h"

#include <assert.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>
#include <string>
#include <thread>

extern "C" {
#include <libavutil/mem.h>
}

#include "audio_encoder.h"
#ifdef HAVE_AV1
#include "av1_encoder.h"
#endif
#include "defs.h"
#include "shared/ffmpeg_raii.h"
#include "flags.h"
#include "shared/httpd.h"
#include "shared/mux.h"
#include "quicksync_encoder.h"
#include "shared/timebase.h"
#include "x264_encoder.h"

class RefCountedFrame;

using namespace std;
using namespace movit;

namespace {

string generate_local_dump_filename(int frame)
{
	time_t now = time(NULL);
	tm now_tm;
	localtime_r(&now, &now_tm);

	char timestamp[64];
	strftime(timestamp, sizeof(timestamp), "%F-%H%M%S%z", &now_tm);

	// Use the frame number to disambiguate between two cuts starting
	// on the same second.
	char filename[256];
	snprintf(filename, sizeof(filename), "%s/%s%s-f%02d%s",
		global_flags.recording_dir.c_str(),
		LOCAL_DUMP_PREFIX, timestamp, frame % 100, LOCAL_DUMP_SUFFIX);
	return filename;
}

}  // namespace

VideoEncoder::VideoEncoder(ResourcePool *resource_pool, QSurface *surface, const std::string &va_display, int width, int height, HTTPD *httpd, DiskSpaceEstimator *disk_space_estimator)
	: resource_pool(resource_pool), surface(surface), va_display(va_display), width(width), height(height), httpd(httpd), disk_space_estimator(disk_space_estimator)
{
	oformat = av_guess_format(global_flags.stream_mux_name.c_str(), nullptr, nullptr);
	assert(oformat != nullptr);
	if (global_flags.stream_audio_codec_name.empty()) {
		stream_audio_encoder.reset(new AudioEncoder(AUDIO_OUTPUT_CODEC_NAME, DEFAULT_AUDIO_OUTPUT_BIT_RATE, oformat));
	} else {
		stream_audio_encoder.reset(new AudioEncoder(global_flags.stream_audio_codec_name, global_flags.stream_audio_codec_bitrate, oformat));
	}
	if (global_flags.x264_video_to_http || global_flags.x264_video_to_disk) {
		x264_encoder.reset(new X264Encoder(oformat, /*use_separate_disk_params=*/false));
	}
	VideoCodecInterface *http_encoder = x264_encoder.get();
	VideoCodecInterface *disk_encoder = x264_encoder.get();
#ifdef HAVE_AV1
	if (global_flags.av1_video_to_http) {
		av1_encoder.reset(new AV1Encoder(oformat));
		http_encoder = av1_encoder.get();
	}
#endif
	if (global_flags.x264_separate_disk_encode) {
		x264_disk_encoder.reset(new X264Encoder(oformat, /*use_separate_disk_params=*/true));
		disk_encoder = x264_disk_encoder.get();
	}

	string filename = generate_local_dump_filename(/*frame=*/0);
	quicksync_encoder.reset(new QuickSyncEncoder(filename, resource_pool, surface, va_display, width, height, oformat, http_encoder, disk_encoder, disk_space_estimator));

	open_output_stream();
	stream_audio_encoder->add_mux(http_mux.get());
	quicksync_encoder->set_http_mux(http_mux.get());
	if (global_flags.x264_video_to_http) {
		x264_encoder->add_mux(http_mux.get());
	}
#ifdef HAVE_AV1
	if (global_flags.av1_video_to_http) {
		av1_encoder->add_mux(http_mux.get());
	}
#endif
}

VideoEncoder::~VideoEncoder()
{
	quicksync_encoder->shutdown();
	x264_encoder.reset(nullptr);
	x264_disk_encoder.reset(nullptr);
	quicksync_encoder->close_file();
	quicksync_encoder.reset(nullptr);
	while (quicksync_encoders_in_shutdown.load() > 0) {
		usleep(10000);
	}
}

void VideoEncoder::do_cut(int frame)
{
	string filename = generate_local_dump_filename(frame);
	printf("Starting new recording: %s\n", filename.c_str());

	// Do the shutdown of the old encoder in a separate thread, since it can
	// take some time (it needs to wait for all the frames in the queue to be
	// done encoding, for one) and we are running on the main mixer thread.
	// However, since this means both encoders could be sending packets at
	// the same time, it means pts could come out of order to the stream mux,
	// and we need to plug it until the shutdown is complete.
	http_mux->plug();
	lock(qs_mu, qs_audio_mu);
	lock_guard<mutex> lock1(qs_mu, adopt_lock), lock2(qs_audio_mu, adopt_lock);
	QuickSyncEncoder *old_encoder = quicksync_encoder.release();  // When we go C++14, we can use move capture instead.
	X264Encoder *old_x264_encoder = nullptr;
	X264Encoder *old_x264_disk_encoder = nullptr;
	if (global_flags.x264_video_to_disk) {
		old_x264_encoder = x264_encoder.release();
	}
	if (global_flags.x264_separate_disk_encode) {
		old_x264_disk_encoder = x264_disk_encoder.release();
	}
	thread([old_encoder, old_x264_encoder, old_x264_disk_encoder, this]{
		old_encoder->shutdown();
		delete old_x264_encoder;
		delete old_x264_disk_encoder;
		old_encoder->close_file();
		http_mux->unplug();

		// We cannot delete the encoder here, as this thread has no OpenGL context.
		// We'll deal with it in begin_frame().
		lock_guard<mutex> lock(qs_mu);
		qs_needing_cleanup.emplace_back(old_encoder);
	}).detach();

	if (global_flags.x264_video_to_disk) {
		x264_encoder.reset(new X264Encoder(oformat, /*use_separate_disk_params=*/false));
		assert(global_flags.x264_video_to_http);
		if (global_flags.x264_video_to_http) {
			x264_encoder->add_mux(http_mux.get());
		}
		if (overriding_bitrate != 0) {
			x264_encoder->change_bitrate(overriding_bitrate);
		}
	}
	X264Encoder *http_encoder = x264_encoder.get();
	X264Encoder *disk_encoder = x264_encoder.get();
	if (global_flags.x264_separate_disk_encode) {
		x264_disk_encoder.reset(new X264Encoder(oformat, /*use_separate_disk_params=*/true));
		disk_encoder = x264_disk_encoder.get();
	}

	quicksync_encoder.reset(new QuickSyncEncoder(filename, resource_pool, surface, va_display, width, height, oformat, http_encoder, disk_encoder, disk_space_estimator));
	quicksync_encoder->set_http_mux(http_mux.get());
}

void VideoEncoder::change_x264_bitrate(unsigned rate_kbit)
{
	overriding_bitrate = rate_kbit;
	x264_encoder->change_bitrate(rate_kbit);
}

void VideoEncoder::add_audio(int64_t pts, std::vector<float> audio)
{
	// Take only qs_audio_mu, since add_audio() is thread safe
	// (we can only conflict with do_cut(), which takes qs_audio_mu)
	// and we don't want to contend with begin_frame().
	{
		lock_guard<mutex> lock(qs_audio_mu);
		quicksync_encoder->add_audio(pts, audio);
	}
	stream_audio_encoder->encode_audio(audio, pts + quicksync_encoder->global_delay());
}

bool VideoEncoder::is_zerocopy() const
{
	// Explicitly do _not_ take qs_mu; this is called from the mixer,
	// and qs_mu might be contended. is_zerocopy() is thread safe
	// and never called in parallel with do_cut() (both happen only
	// from the mixer thread).
	return quicksync_encoder->is_zerocopy();
}

bool VideoEncoder::begin_frame(int64_t pts, int64_t duration, movit::YCbCrLumaCoefficients ycbcr_coefficients, const std::vector<RefCountedFrame> &input_frames, GLuint *y_tex, GLuint *cbcr_tex)
{
	lock_guard<mutex> lock(qs_mu);
	qs_needing_cleanup.clear();  // Since we have an OpenGL context here, and are called regularly.
	return quicksync_encoder->begin_frame(pts, duration, ycbcr_coefficients, input_frames, y_tex, cbcr_tex);
}

RefCountedGLsync VideoEncoder::end_frame()
{
	lock_guard<mutex> lock(qs_mu);
	return quicksync_encoder->end_frame();
}

void VideoEncoder::open_output_stream()
{
	AVFormatContext *avctx = avformat_alloc_context();
	avctx->oformat = oformat;

	uint8_t *buf = (uint8_t *)av_malloc(MUX_BUFFER_SIZE);
	avctx->pb = avio_alloc_context(buf, MUX_BUFFER_SIZE, 1, this, nullptr, nullptr, nullptr);
	avctx->pb->write_data_type = &VideoEncoder::write_packet2_thunk;
	avctx->pb->ignore_boundary_point = 1;

	Mux::Codec video_codec;
	if (global_flags.av1_video_to_http) {
		video_codec = Mux::CODEC_AV1;
	} else {
		video_codec = Mux::CODEC_H264;
	}

	avctx->flags = AVFMT_FLAG_CUSTOM_IO;

	string video_extradata;
	if (global_flags.x264_video_to_http) {
		video_extradata = x264_encoder->get_global_headers();
#ifdef HAVE_AV1
	} else if (global_flags.av1_video_to_http) {
		video_extradata = av1_encoder->get_global_headers();
#endif
	}

	http_mux.reset(new Mux(avctx, width, height, video_codec, video_extradata, stream_audio_encoder->get_codec_parameters().get(),
		get_color_space(global_flags.ycbcr_rec709_coefficients), COARSE_TIMEBASE,
		/*write_callback=*/nullptr, Mux::WRITE_FOREGROUND, { &http_mux_metrics }));
	http_mux_metrics.init({{ "destination", "http" }});
}

int VideoEncoder::write_packet2_thunk(void *opaque, uint8_t *buf, int buf_size, AVIODataMarkerType type, int64_t time)
{
	VideoEncoder *video_encoder = (VideoEncoder *)opaque;
	return video_encoder->write_packet2(buf, buf_size, type, time);
}

int VideoEncoder::write_packet2(uint8_t *buf, int buf_size, AVIODataMarkerType type, int64_t time)
{
	if (type == AVIO_DATA_MARKER_SYNC_POINT || type == AVIO_DATA_MARKER_BOUNDARY_POINT) {
		seen_sync_markers = true;
	} else if (type == AVIO_DATA_MARKER_UNKNOWN && !seen_sync_markers) {
		// We don't know if this is a keyframe or not (the muxer could
		// avoid marking it), so we just have to make the best of it.
		type = AVIO_DATA_MARKER_SYNC_POINT;
	}

	if (type == AVIO_DATA_MARKER_HEADER) {
		http_mux_header.append((char *)buf, buf_size);
		httpd->set_header(HTTPD::StreamID{ HTTPD::MAIN_STREAM, 0 }, http_mux_header);
	} else {
		httpd->add_data(HTTPD::StreamID{ HTTPD::MAIN_STREAM, 0 }, (char *)buf, buf_size, type == AVIO_DATA_MARKER_SYNC_POINT, time, AVRational{ AV_TIME_BASE, 1 });
	}
	return buf_size;
}

