/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.gsf.testrunner.api;

import java.awt.EventQueue;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Logger;
import org.openide.nodes.AbstractNode;
import org.openide.nodes.Children;
import org.openide.util.NbBundle;
import org.openide.util.actions.SystemAction;

/**
 *
 * @author Marian Petras, Erno Mononen
 */
final class RootNode extends AbstractNode {


    /** constant meaning "information about passed tests not displayed" */
    static final int ALL_PASSED_ABSENT = 0;
    /** constant meaning "information about some passed tests not displayed" */
    static final int SOME_PASSED_ABSENT = 1;
    /** constant meaning "information about all passed tests displayed */
    static final int ALL_PASSED_DISPLAYED = 2;
    /**
     */
    private final RootNodeChildren children;
    /**
     */
    private volatile int filterMask;
    /** */
    private volatile String message;
    private volatile int totalTests = 0;
    private volatile int failures = 0;
    private volatile int errors = 0;
    private volatile int pending = 0;
    private volatile long elapsedTimeMillis = 0;
    private volatile int detectedPassedTests = 0;
    private boolean sessionFinished;
    private final TestSession session;

    private static final Logger LOGGER = Logger.getLogger(RootNode.class.getName());

    /**
     * Creates a new instance of RootNode
     */
    public RootNode(TestSession session, int filterMask) {
        super(new RootNodeChildren(session, filterMask));
        this.session = session;
        this.filterMask = filterMask;
        children = (RootNodeChildren) getChildren();
        setName(NbBundle.getMessage(RootNode.class, "MSG_RunningTests"));

        setIconBaseWithExtension(
                "org/netbeans/modules/gsf/testrunner/resources/empty.gif");     //NOI18N

    }

    /**
     */
    void displayMessage(final String msg) {
        assert EventQueue.isDispatchThread();

        /* Called from the EventDispatch thread */

        this.message = msg;
        updateDisplayName();
    }

    /**
     * Updates the display when the session is finished.
     * 
     * @param  msg  optional message to be displayed (e.g. notice that
     *              the sessions has been interrupted); or {@code null}
     */
    void displayMessageSessionFinished(final String msg) {
        sessionFinished = true;
        displayMessage(msg);
    }

    /**
     * Displays a message that a given test suite is running.
     *
     * @param  suiteName  name of the running test suite,
     *                    or {@code ANONYMOUS_SUITE} for anonymous suites
     *
     * @see  ResultDisplayHandler#ANONYMOUS_SUITE
     */
    void displaySuiteRunning(final String suiteName) {
        assert EventQueue.isDispatchThread();

        /* Called from the EventDispatch thread */

        children.displaySuiteRunning(suiteName);
    }

    /**
     * Displays a message that a given test suite is running.
     *
     * @param  suite  running test suite,
     *                    or {@code ANONYMOUS_TEST_SUITE} for anonymous suites
     *
     * @see  ResultDisplayHandler#ANONYMOUS_TEST_SUITE
     */
    void displaySuiteRunning(final TestSuite suite) {
        assert EventQueue.isDispatchThread();

        /* Called from the EventDispatch thread */

        children.displaySuiteRunning(suite);
    }

    /**
     */
    public TestsuiteNode displayReport(final Report report) {
        assert EventQueue.isDispatchThread();

        /* Called from the EventDispatch thread */

        TestsuiteNode suiteNode = children.displayReport(report);
        updateStatistics();
        updateDisplayName();
        return suiteNode;
    }

    /**
     */
    void displayReports(final Collection<Report> reports) {
        assert EventQueue.isDispatchThread();

        /* Called from the EventDispatch thread */

        children.displayReports(reports);
        updateStatistics();
        updateDisplayName();
    }

    /**
     */
    private synchronized void updateStatistics() {
        totalTests = 0;
        failures = 0;
        errors = 0;
        pending = 0;
        detectedPassedTests = 0;
        elapsedTimeMillis = 0;
        for(Report rep: children.getReports()){
            totalTests += rep.getTotalTests();
            failures += rep.getFailures();
            errors += rep.getErrors();
            pending += rep.getPending();
            detectedPassedTests += rep.getDetectedPassedTests();
            elapsedTimeMillis += rep.getElapsedTimeMillis();
        }
    }

    float getPassedPercentage() {
        return (float) detectedPassedTests / totalTests * 100;
    }

    /**
     */
    void setFilterMask(final int filterMask) {
        assert EventQueue.isDispatchThread();

        if (filterMask == this.filterMask) {
            return;
        }
        this.filterMask = filterMask;

        Children children = getChildren();
        if (children != Children.LEAF) {
            ((RootNodeChildren) children).setFilterMask(filterMask);
        }
    }

    /**
     */
    private void updateDisplayName() {
        assert EventQueue.isDispatchThread();

        /* Called from the EventDispatch thread */

        final Class bundleRefClass = getClass();
        String msg;

        if (totalTests == 0) {
            if (sessionFinished) {
                msg = NbBundle.getMessage(bundleRefClass,
                        "MSG_TestsInfoNoTests");      //NOI18N

            } else {
                msg = null;
            }
        } else if (failures == 0 && errors == 0 && pending == 0) {
            msg = NbBundle.getMessage(bundleRefClass,
                    "MSG_TestsInfoAllOK", //NOI18N
                    new Integer(totalTests));
        } else {
            
            String passedTestsInfo = NbBundle.getMessage(
                    bundleRefClass,
                    "MSG_PassedTestsInfo", //NOI18N
                    new Integer(totalTests - failures - errors - pending));
            
            String pendingTestsInfo = (pending == 0)
                    ? null
                    : NbBundle.getMessage(
                    bundleRefClass,
                    "MSG_PendingTestsInfo", //NOI18N
                    new Integer(errors));

            String failedTestsInfo = (failures == 0)
                    ? null
                    : NbBundle.getMessage(
                    bundleRefClass,
                    "MSG_FailedTestsInfo", //NOI18N
                    new Integer(failures));
            String errorTestsInfo = (errors == 0)
                    ? null
                    : NbBundle.getMessage(
                    bundleRefClass,
                    "MSG_ErrorTestsInfo", //NOI18N
                    new Integer(errors));
            
            msg = constructMessage(passedTestsInfo, pendingTestsInfo, failedTestsInfo, errorTestsInfo);
            
        }

        if (totalTests != 0) {
            assert msg != null;
            final int successDisplayedLevel = getSuccessDisplayedLevel();
            switch (successDisplayedLevel) {
                case SOME_PASSED_ABSENT:
                    msg += ' ';
                    msg += NbBundle.getMessage(
                            bundleRefClass,
                            "MSG_SomePassedNotDisplayed");  //NOI18N

                    break;
                case ALL_PASSED_ABSENT:
                    msg += ' ';
                    msg += NbBundle.getMessage(
                            bundleRefClass,
                            "MSG_PassedNotDisplayed");      //NOI18N

                    break;
                case ALL_PASSED_DISPLAYED:
                    break;
                default:
                    assert false;
                    break;
            }
        }

        if (msg != null) {
            msg += NbBundle.getMessage(bundleRefClass, "MSG_TestSuiteElapsedTime", new Double(elapsedTimeMillis / 1000d));
        }

        if (this.message != null) {
            if (msg == null) {
                msg = this.message;
            } else {
                msg = msg + ' ' + message;
            }
        }

        // #143508
        LOGGER.fine("Setting display name to: '" + msg + "'. Total tests run: " + totalTests + ". Session finished: " + sessionFinished);

        setDisplayName(msg);
    }
    
    String constructMessage(String... subMessages) {
        List<String> messageList = new ArrayList<String>();
        for (String msg : subMessages) {
            if (msg != null) {
                messageList.add(msg);
            }
        }
        int size = messageList.size();
        String key = "MSG_TestResultSummary" + (size - 1);
        return NbBundle.getMessage(RootNode.class, key, messageList.toArray(new String[size]));
    }

    /**
     * Returns information whether information about passed tests is displayed.
     *
     * @return  one of constants <code>ALL_PASSED_DISPLAYED</code>,
     *                           <code>SOME_PASSED_ABSENT</code>,
     *                           <code>ALL_PASSED_ABSENT</code>
     */
    int getSuccessDisplayedLevel() {
        int reportedPassedTestsCount = totalTests - failures - errors;
        if (detectedPassedTests >= reportedPassedTestsCount) {
            return ALL_PASSED_DISPLAYED;
        } else if (detectedPassedTests == 0) {
            return ALL_PASSED_ABSENT;
        } else {
            return SOME_PASSED_ABSENT;
        }
    }

    @Override
    public SystemAction[] getActions(boolean context) {
        return new SystemAction[0];
    }
}
