/***************************************************************************
 *   Copyright (C) 2005-2008 by Eugene V. Lyubimkin aka jackyf             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License                  *
 *   (version 3 or above) as published by the Free Software Foundation.    *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU GPL                        *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA               *
 ***************************************************************************/
#include <QPixmap>
#include <QLabel>
#include <QGridLayout>
#include <QtDebug>
#include <QPainter>
#include <QPaintEvent>

#include "KeyboardWidget.hpp"
#include "Setup.hpp"

KeyboardWidget::KeyboardWidget(const Layout& layout, QWidget* parent)
	: QFrame(parent), layout(layout)
{
	symbolToMark = nullChar;
	// reading the image
	this->keyboardImage = new QPixmap(pathToKeyboardImage);
	if (keyboardImage->isNull())
	{
		qCritical() << "Cannot load the keyboard image at" << qPrintable(pathToKeyboardImage);
	}

	const int imageMargin = 2;
	//this->setMargin(imageMargin);
	this->setFrameStyle(QFrame::StyledPanel);
	this->setFixedSize(keyboardImage->width() + imageMargin, keyboardImage->height() + imageMargin);
	this->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

	this->drawLettersOnPixmap();
}

void KeyboardWidget::markSymbol(QChar symbol)
{
	//qDebug() << "Symbol to mark:" << symbol;
	symbolToMark = symbol;
	this->update();
}

KeyboardWidget::~KeyboardWidget()
{
	delete this->keyboardImage;
}


// some common constants
// ---------------------------
const size_t rowCount = 4;

const QPoint rowStartPoints[rowCount] = {
	QPoint(14, 5),
	QPoint(69, 41),
	QPoint(81, 77),
	QPoint(101, 113)
};

const size_t cellSpacing = 41;
// ----------------------------

void KeyboardWidget::drawLettersOnPixmap()
{
	const int fontSize = 13;

	const size_t bigCharTopDisplace = 19;
	const size_t bigCharLeftDisplace = 18;
	const size_t smallCharTopDisplace = 26;
	const size_t smallCharLeftDisplace = 5;

	QPainter painter(this->keyboardImage);
	painter.setFont(QFont("Monospace", fontSize));

	for (size_t row = 0; row < rowCount; ++row)
	{
		int indexCount = layout.charsInRow(row);
		for (int index = 0; index < indexCount; ++index)
		{
			int drawX = rowStartPoints[row].x() + cellSpacing*index;
			int drawY = rowStartPoints[row].y();

			painter.drawText(drawX + bigCharLeftDisplace, drawY + bigCharTopDisplace, QString(layout.getBigChar(row, index)));
			painter.drawText(drawX + smallCharLeftDisplace, drawY + smallCharTopDisplace, QString(layout.getSmallChar(row, index)));
		}
	}
}

void KeyboardWidget::paintEvent(QPaintEvent* event)
{
	const size_t cellSize = 32;

	const QPen defaultPen = QPen(Qt::black);
	const QColor bigMarkColor = Qt::red;
	const QColor smallMarkColor = Qt::blue;
	const QColor spaceMarkColor = Qt::green;
	const int penWidth = 3;

	const QRect spaceRect = QRect(QPoint(162, 151), QSize(244, 25));

	QPainter painter(this);
	painter.drawPixmap(0, 0, *keyboardImage);
	painter.setBrush(QBrush(QColor(0, 0, 0, 0)));

	for (size_t row = 0; row < rowCount; ++row)
	{
		int indexCount = layout.charsInRow(row);
		for (int index = 0; index < indexCount; ++index)
		{
			int drawX = rowStartPoints[row].x() + cellSpacing*index;
			int drawY = rowStartPoints[row].y();

			/*
			qDebug() << "symbolToMark:" << symbolToMark << ", big symbol:"
				<< layout.getBigChar(row, index) << ",small symbol:" << layout.getSmallChar(row, index);
			*/
			QColor markColor;
			bool doMarking = false;
			if (symbolToMark == spaceChar)
			{
				QPen markPen(spaceMarkColor);
				markPen.setWidth(penWidth);
				painter.setPen(markPen);
				painter.drawRect(spaceRect);
				painter.setPen(defaultPen);
			}
			else if (symbolToMark == layout.getBigChar(row, index))
			{
				//qDebug("Marking big symbol");
				markColor = bigMarkColor;
				doMarking = true;
			}
			else if (symbolToMark == layout.getSmallChar(row, index))
			{
				//qDebug("Marking small symbol");
				markColor = smallMarkColor;
				doMarking = true;
			}
			if (doMarking)
			{
				QPen markPen(markColor);
				markPen.setWidth(penWidth);
				painter.setPen(markPen);
				painter.drawRect(drawX, drawY, cellSize, cellSize);
				painter.setPen(defaultPen);
			}
		}
	}

	painter.end();
	QFrame::paintEvent(event);
	event->accept();
}

