/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isnan = require( './../../../../base/assert/is-nan' );
var isInfinite = require( './../../../../base/assert/is-infinite' );
var pow = require( './../../../../base/special/pow' );
var floor = require( './../../../../base/special/floor' );
var ceil = require( './../../../../base/special/ceil' );
var log10 = require( './../../../../base/special/log10' );
var MAX_EXP = require( '@stdlib/constants/float64/max-base10-exponent' );
var MIN_EXP_SUBNORMAL = require( '@stdlib/constants/float64/min-base10-exponent-subnormal' );


// VARIABLES //

// 10^308:
var HUGE = 1.0e308;

// 10^-323
var TINY = 1.0e-323;


// MAIN //

/**
* Rounds a numeric value to the nearest power of `10` on a linear scale.
*
* @param {number} x - input value
* @returns {number} rounded value
*
* @example
* var v = round10( 3.141592653589793 );
* // returns 1.0
*
* @example
* var v = round10( 13.0 );
* // returns 10.0
*
* @example
* var v = round10( -0.314 );
* // returns -0.1
*/
function round10( x ) {
	var sign;
	var half;
	var p1;
	var p2;
	var y1;
	var y2;
	var p;
	if (
		isnan( x ) ||
		isInfinite( x ) ||
		x === 0.0
	) {
		return x;
	}
	if ( x < 0 ) {
		x = -x;
		sign = -1.0;
	} else {
		sign = 1.0;
	}
	// Solve the equation `10^p = x` for `p`:
	p = log10( x );

	// Find the previous and next integer powers:
	p1 = floor( p );
	p2 = ceil( p );

	// Handle tiny:
	if ( p1 === MIN_EXP_SUBNORMAL ) {
		return sign * TINY;
	}
	// Handle huge:
	if ( p1 === MAX_EXP ) {
		return sign * HUGE;
	}
	// Compute previous and next powers of 10:
	y1 = pow( 10.0, p1 );
	y2 = pow( 10.0, p2 );

	// Find the closest power of 10:
	half = ( y2 - y1 ) / 2.0;
	if ( y1+half > x ) {
		return sign * y1;
	}
	return sign * y2;
}


// EXPORTS //

module.exports = round10;
