/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*
* ## Notice
*
* The original C code, long comment, copyright, license, and constants are from [Cephes]{@link http://www.netlib.org/cephes}. The implementation follows the original, but has been modified for JavaScript.
*
* ```text
* Copyright 1984, 1995, 2000 by Stephen L. Moshier
*
* Some software in this archive may be from the book _Methods and Programs for Mathematical Functions_ (Prentice-Hall or Simon & Schuster International, 1989) or from the Cephes Mathematical Library, a commercial product. In either event, it is copyrighted by the author. What you see here may be used freely but it comes with no support or guarantee.
*
* Stephen L. Moshier
* moshier@na-net.ornl.gov
* ```
*/

'use strict';

// MODULES //

var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var abs = require( './../../../../base/special/abs' );
var exp = require( './../../../../base/special/exp' );
var LN2 = require( '@stdlib/constants/float64/ln-two' );
var rateval = require( './rational_pq.js' );


// VARIABLES //

// ln(2^1024)
var MAXLOG = 7.09782712893383996843e2;

// ln(2^-1022)
var MINLOG = -7.08396418532264106224e2;

var POS_OVERFLOW = MAXLOG + LN2;
var NEG_OVERFLOW = MINLOG - LN2;

var LARGE = MAXLOG - LN2;


// MAIN //

/**
* Computes the hyperbolic sine of a number.
*
* ## Method
*
* The range is partitioned into two segments. If \\( |x| \le 1 \\), we use a rational function of the form
*
* ```tex
* x + x^3 \frac{\mathrm{P}(x)}{\mathrm{Q}(x)}
* ```
*
* Otherwise, the calculation is
*
* ```tex
* \operatorname{sinh}(x) = \frac{ e^x - e^{-x} }{2}.
* ```
*
* ## Notes
*
* -   Relative error:
*
*     | arithmetic | domain   | # trials | peak    | rms     |
*     |:----------:|:--------:|:--------:|:-------:|:-------:|
*     | DEC        | +- 88    | 50000    | 4.0e-17 | 7.7e-18 |
*     | IEEE       | +-MAXLOG | 30000    | 2.6e-16 | 5.7e-17 |
*
*
* @param {number} x - input value (in radians)
* @returns {number} hyperbolic sine
*
* @example
* var v = sinh( 0.0 );
* // returns 0.0
*
* @example
* var v = sinh( 2.0 );
* // returns ~3.627
*
* @example
* var v = sinh( -2.0 );
* // returns ~-3.627
*
* @example
* var v = sinh( NaN );
* // returns NaN
*/
function sinh( x ) {
	var a;
	if ( x === 0.0 ) {
		return x; // handles `+-0`
	}
	a = abs( x );
	if ( x > POS_OVERFLOW || x < NEG_OVERFLOW ) {
		return ( x > 0.0 ) ? PINF : NINF;
	}
	if ( a > 1.0 ) {
		if ( a >= LARGE ) {
			a = exp( 0.5*a );
			a *= 0.5 * a;
			if ( x < 0.0 ) {
				a = -a;
			}
			return a;
		}
		a = exp( a );
		a = (0.5*a) - (0.5/a);
		if ( x < 0.0 ) {
			a = -a;
		}
		return a;
	}
	a *= a;
	return x + ( x*a*rateval( a ) );
}


// EXPORTS //

module.exports = sinh;
