<!--

@license Apache-2.0

Copyright (c) 2020 The Stdlib Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

-->

# Cumulative Distribution Function

> Wilcoxon signed rank test statistic [cumulative distribution function][cdf].

<section class="intro">

</section>

<!-- /.intro -->

<section class="usage">

## Usage

```javascript
var cdf = require( '@stdlib/stats/base/dists/signrank/cdf' );
```

#### cdf( x, n )

Evaluates the [cumulative distribution function][cdf] of the Wilcoxon signed rank test statistic with `n` observations.

```javascript
var y = cdf( 7.0, 9 );
// returns ~0.037

y = cdf( 7.0, 6 );
// returns ~0.281

y = cdf( -1.0, 40 );
// returns 0.0
```

If provided `NaN` as any argument, the function returns `NaN`.

```javascript
var y = cdf( NaN, 8 );
// returns NaN

y = cdf( 0.0, NaN );
// returns NaN
```

If provided `x < 0`, the function returns `NaN`.

```javascript
var y = cdf( 2.0, -1.0 );
// returns NaN
```

If not provided a positive integer for `n`, the function returns `NaN`.

```javascript
var y = cdf( 2.0, 0 );
// returns NaN

y = cdf( 2.0, -2 );
// returns NaN

y = cdf( 2.0, 8.9 );
// returns NaN
```

#### cdf.factory( n )

Returns a function for evaluating the [cumulative distribution function][cdf] of the Wilcoxon signed rank test statistic with `n` observations.

```javascript
var mycdf = cdf.factory( 8 );
var y = mycdf( 3.9 );
// returns ~0.027

y = mycdf( 17.0 );
// returns ~0.473
```

</section>

<!-- /.usage -->

<section class="examples">

## Examples

<!-- eslint no-undef: "error" -->

```javascript
var ceil = require( '@stdlib/math/base/special/ceil' );
var randu = require( '@stdlib/random/base/randu' );
var cdf = require( '@stdlib/stats/base/dists/signrank/cdf' );

var n;
var x;
var y;
var i;

for ( i = 0; i < 10; i++ ) {
    x = randu() * 30.0;
    n = ceil( randu() * 30.0 );
    y = cdf( x, n );
    console.log( 'x: %d, n: %d, F(x;n): %d', x.toFixed( 4 ), n.toFixed( 4 ), y.toFixed( 4 ) );
}
```

</section>

<!-- /.examples -->

<section class="links">

[cdf]: https://en.wikipedia.org/wiki/Cumulative_distribution_function

</section>

<!-- /.links -->
