/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var EventEmitter = require( 'events' ).EventEmitter;
var tape = require( 'tape' );
var proxyquire = require( 'proxyquire' );
var nextTick = require( './../../next-tick' );
var worker = require( './../lib/node/worker/worker.js' );


// FIXTURES //

function Proc() {
	EventEmitter.call( this );
	this.env = {
		'WORKER_CMD': 'node'
	};
	return this;
}

Proc.prototype = Object.create( EventEmitter.prototype );
Proc.prototype.constructor = Proc;

Proc.prototype.send = function send() {};


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof worker, 'function', 'main export is a function' );
	t.end();
});

tape( 'if the worker receives a `close` message, the worker closes child processes', function test( t ) {
	var worker;
	var proc;

	proc = new Proc();

	worker = proxyquire( './../lib/node/worker/worker.js', {
		'./close.js': closeProcess,
		'./process.js': proc
	});
	worker();
	proc.emit( 'message', 'close' );

	function closeProcess() {
		t.ok( true, 'calls fcn to close child processes' );
		t.end();
	}
});

tape( 'if the `WORKER_ORDERED` environment variable is set, the worker should "exec" a script', function test( t ) {
	var worker;
	var proc;

	proc = new Proc();
	proc.env.WORKER_ORDERED = '1';

	worker = proxyquire( './../lib/node/worker/worker.js', {
		'./process.js': proc,
		'./exec.js': exec,
		'./children.js': {}
	});
	worker();
	proc.emit( 'message', '/a/b/c.js' );

	t.end();

	function exec() {
		t.ok( true, 'calls an exec script' );
		return {
			'pid': 1234
		};
	}
});

tape( 'if the `WORKER_ORDERED` environment variable is not set, the worker should "spawn" a script', function test( t ) {
	var worker;
	var proc;

	proc = new Proc();
	delete proc.env.WORKER_ORDERED;

	worker = proxyquire( './../lib/node/worker/worker.js', {
		'./process.js': proc,
		'./spawn.js': spawn,
		'./children.js': {}
	});
	worker();
	proc.emit( 'message', '/a/b/c.js' );

	t.end();

	function spawn() {
		t.ok( true, 'calls a spawn script' );
		return {
			'pid': 12345
		};
	}
});

tape( 'the process emits an error and closes itself if an error is encountered when spawning a script', function test( t ) {
	var worker;
	var proc;
	var flg;

	proc = new Proc();
	delete proc.env.WORKER_ORDERED;

	worker = proxyquire( './../lib/node/worker/worker.js', {
		'./close.js': closeProcess,
		'./process.js': proc,
		'./spawn.js': spawn,
		'./children.js': {}
	});
	worker();

	proc.on( 'error', onError );
	proc.emit( 'message', '/a/b/c.js' );

	function spawn( cmd, msg, clbk ) {
		nextTick( onTick );
		return {
			'pid': 12345
		};
		function onTick() {
			clbk( new Error( 'beep' ) );
		}
	}

	function closeProcess() {
		t.ok( true, 'calls fcn to close a process' );
		flg = true;
	}

	function onError( error ) {
		t.ok( error, 'emits an error' );
		if ( !flg ) {
			t.ok( false, 'did not call fcn to close a process' );
		}
		t.end();
	}
});

tape( 'the process emits an error and closes itself if an error is encountered when executing a script', function test( t ) {
	var worker;
	var proc;
	var flg;

	proc = new Proc();
	proc.env.WORKER_ORDERED = '1';

	worker = proxyquire( './../lib/node/worker/worker.js', {
		'./close.js': closeProcess,
		'./process.js': proc,
		'./exec.js': exec,
		'./children.js': {}
	});
	worker();

	proc.on( 'error', onError );
	proc.emit( 'message', '/a/b/c.js' );

	function exec( cmd, msg, clbk ) {
		nextTick( onTick );
		return {
			'pid': 12345
		};
		function onTick() {
			clbk( new Error( 'beep' ) );
		}
	}

	function closeProcess() {
		t.ok( true, 'calls fcn to close a process' );
		flg = true;
	}

	function onError( error ) {
		t.ok( error, 'emits an error' );
		if ( !flg ) {
			t.ok( false, 'did not call fcn to close a process' );
		}
		t.end();
	}
});

tape( 'the process sends a message to the parent process containing the script filepath as an identifier, informing the parent process that the script has finished executing', function test( t ) {
	var worker;
	var proc;
	var msg;

	proc = new Proc();
	delete proc.env.WORKER_ORDERED;

	proc.send = send;

	worker = proxyquire( './../lib/node/worker/worker.js', {
		'./process.js': proc,
		'./spawn.js': spawn,
		'./children.js': {}
	});
	worker();

	msg = '/a/b/c.js';
	proc.emit( 'message', msg );

	function spawn( cmd, msg, clbk ) {
		nextTick( onTick );
		return {
			'pid': 12345
		};
		function onTick() {
			clbk( null, 12345, msg );
		}
	}

	function send( val ) {
		t.equal( val, msg, 'sends file path to parent process' );
		t.end();
	}
});
