/*###############################################################################
# Linux Management Providers (LMP), CPU provider package
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr ,desmons_frederic@yahoo.fr>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by 2009 Rakhimov Rustam, TUIT <rusyasoft@gmail.com>
# Modified by 2009 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr>
###############################################################################*/

#include "OpenDRIM_ProcessorCoreAccess.h"

vector<_processor_topology> _previous_cpu_info_processors_information;

int CPU_OpenDRIM_ProcessorCore_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	CF_assert(CPU_getProcessors("/proc/cpuinfo", _previous_cpu_info_processors_information, errorMessage));
	CF_assert(CPU_getLoadAverages(_previous_cpu_info_processors_information, _previous_cpu_info_processors_information, errorMessage));
	_L_;
	return OK;
}

int CPU_OpenDRIM_ProcessorCore_unload(string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int CPU_OpenDRIM_ProcessorCore_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_ProcessorCore>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	// get info from SMBIOS
	vector<_processor_information> SMBIOS_processors_information;
	int errorCode = SMBIOS_getProcessorsInformation(SMBIOS_processors_information, errorMessage);
	// if we cannot get info from SMBIOS, we'll just use info from cpuinfo
	bool use_SMBIOS = false;
	if (errorCode == OK)
		use_SMBIOS = true;
	
	// get info from cpuinfo
	vector<_processor_topology> cpu_info_processors_information;
	CF_assert(CPU_getProcessors("/proc/cpuinfo", cpu_info_processors_information, errorMessage));
	CF_assert(CPU_getLoadAverages(cpu_info_processors_information, _previous_cpu_info_processors_information, errorMessage));
	_previous_cpu_info_processors_information = cpu_info_processors_information;
	
	// different processors count...
	if (cpu_info_processors_information.size() != SMBIOS_processors_information.size())
		use_SMBIOS = false;
	
	for (unsigned int i = 0; i < cpu_info_processors_information.size(); i++) {
		_processor_information SMBIOS_processor_information;
		if (use_SMBIOS)
			SMBIOS_processor_information = SMBIOS_processors_information[i];
		for (unsigned int j = 0; j < cpu_info_processors_information[i].processor_cores.size(); j++) {
			OpenDRIM_ProcessorCore instance;
			instance.setInstanceID(CF_intToStr(i) + ":" + CF_intToStr(j));
			if (discriminant == "ei") {
				CF_assert(CPU_OpenDRIM_ProcessorCore_populate(instance, cpu_info_processors_information[i], cpu_info_processors_information[i].processor_cores[j], SMBIOS_processor_information, use_SMBIOS, errorMessage));
			}
			result.push_back(instance);
		}
	}
	_L_;
	return OK;
}

int CPU_OpenDRIM_ProcessorCore_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_ProcessorCore& instance, const char** properties, string& errorMessage) {
	_E_;
	string InstanceID;
	instance.getInstanceID(InstanceID);
	vector<string> InstanceID_elements;
	CF_splitText(InstanceID_elements, InstanceID, ':');
	if (InstanceID_elements.size() != 2)
		return NOT_FOUND;
	if (!CF_isNumber(InstanceID_elements[0]) || !CF_isNumber(InstanceID_elements[1]))
		return NOT_FOUND;
	unsigned int processor_id = atoll(InstanceID_elements[0].c_str());
	unsigned int core_id = atoll(InstanceID_elements[1].c_str());
	
	// get info from cpuinfo
	vector<_processor_topology> cpu_info_processors_information;
	CF_assert(CPU_getProcessors("/proc/cpuinfo", cpu_info_processors_information, errorMessage));
	CF_assert(CPU_getLoadAverages(cpu_info_processors_information, _previous_cpu_info_processors_information, errorMessage));
	_previous_cpu_info_processors_information = cpu_info_processors_information;
	
	// does this core exists ?
	if (processor_id >= cpu_info_processors_information.size())
		return NOT_FOUND;
	if (core_id >= cpu_info_processors_information[processor_id].processor_cores.size())
		return NOT_FOUND;
	
	// get info from SMBIOS
	vector<_processor_information> SMBIOS_processors_information;
	int errorCode = SMBIOS_getProcessorsInformation(SMBIOS_processors_information, errorMessage);
	// if we cannot get info from SMBIOS, we'll just use info from cpuinfo
	bool use_SMBIOS = false;
	if (errorCode == OK)
		use_SMBIOS = true;
	
	_processor_information SMBIOS_processor_information;
	if (use_SMBIOS)
		SMBIOS_processor_information = SMBIOS_processors_information[processor_id];
	
	// populate the instance
	CF_assert(CPU_OpenDRIM_ProcessorCore_populate(instance, cpu_info_processors_information[processor_id], cpu_info_processors_information[processor_id].processor_cores[core_id], SMBIOS_processor_information, use_SMBIOS, errorMessage));
	_L_;
	return OK;
}

int CPU_OpenDRIM_ProcessorCore_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_ProcessorCore& newInstance, const OpenDRIM_ProcessorCore& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_ProcessorCore_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_ProcessorCore& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_ProcessorCore_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_ProcessorCore& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_ProcessorCore_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_ProcessorCore& instance, unsigned int& returnValue, const OpenDRIM_ProcessorCore_RequestStateChange_In& in, OpenDRIM_ProcessorCore_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int CPU_OpenDRIM_ProcessorCore_populate(OpenDRIM_ProcessorCore& instance, const _processor_topology& cpu_info_processor_information, const _processor_core& processor_core, const _processor_information& SMBIOS_processor_information, bool use_SMBIOS, string& errorMessage) {
	_E_;
	
	/*
	 * Properties to fill from profile
	 * + Mandatory:
	 * [X] InstanceID
	 * [X] CoreEnabledState
	 * [X] EnabledState
	 * [X] RequestedState
	 * [X] OperationalStatus
	 * [X] HealthState
	 * [X] ElementName
	 */
	
	vector<unsigned short> OperationalStatus;
	int core_num=atoi(instance.InstanceID.substr(instance.InstanceID.find(":")+1).c_str());
	
	// default
	// we don't support RequestedStateChange()
	instance.setRequestedState(12);    // Not Applicable
	instance.setEnabledState(5);       // Not Applicable
	
	if(SMBIOS_processor_information.core_enabled == 0)
		instance.setCoreEnabledState(0); // unknown
	else if(core_num < SMBIOS_processor_information.core_enabled)
		instance.setCoreEnabledState(2); // Core Enabled
	else
	instance.setCoreEnabledState(3);   // Core Disabled
	
	OperationalStatus.push_back(0);    // unknown
	instance.setOperationalStatus(OperationalStatus);
	instance.setHealthState(0);        // unknown
	
	// set info from SMBIOS
	if (use_SMBIOS)
		instance.setMaxClockSpeed(SMBIOS_processor_information.max_speed);
	else
		instance.setMaxClockSpeed(0);
	
	// set /proc/cpuinfo info
	instance.setElementName(cpu_info_processor_information.name + " core");
	instance.setCurrentClockSpeed(processor_core.current_freq/1000);
	
	// load percentage
	instance.setLoadPercentage(processor_core.load_average);
	_L_;
	return OK;
}

