////////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_APPLY_MODIFIER_ACTION_H
#define __OVITO_APPLY_MODIFIER_ACTION_H

#include <core/Core.h>
#include <core/reference/RefTargetListener.h>

#include "Action.h"

namespace Core {

class SelectionSet;		// defined in SelectionSet.h

/**
 * \brief An action that applies a Modifier to the selected scene nodes.
 *
 * \author Alexander Stukowski
 * \sa Modifier
 */
class CORE_DLLEXPORT ApplyModifierAction : public Action
{
private:

	/// \brief Default constructor. Should not be used.
	ApplyModifierAction() : Action() { OVITO_ASSERT(false); }

public:

	/// \brief Initializes the action object.
	/// \param modifierClass The Modifier sub-class that should be applied by this action.
	ApplyModifierAction(PluginClassDescriptor* modifierClass);

protected:

	/// This is called by the system after the action object has been bound to its ActionProxy.
	virtual void initializeAction(ActionProxy* proxy);

protected Q_SLOTS:

	/// Is called when the user has triggered the action's state.
	void onActionTriggered(bool checked);

	/// This is called after all changes to the selection set have been completed.
	void onSelectionChanged(SelectionSet* newSelection);

	/// This is called by the RefTargetListener that listens to notification messages sent by the
	/// current selection set.
	void onSelectionSetMessage(RefTargetMessage* msg);

	/// Updates the enabled/disable state of the modifier action based on the currently selected objects.
	void updateActionState();

Q_SIGNALS:

	/// This signal is only used internaly to call the updateActionState() method in an asynchronous way.
	void updateActionStateSignal();

private:

	/// The type of modifier that is created by this action.
	PluginClassDescriptor* _modifierClass;

	/// The type of input objects to which the modifier can be applied.
	/// If this list is empty then the modifier can be applied to every object.
	QVector<PluginClassDescriptor*> _inputObjectClasses;

	/// This listens to notification message sent by the current selection set.
	RefTargetListener selectionSetListener;

	/// Indicates that the state of the action needs to be updated.
	bool _needsUpdate;

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(ApplyModifierAction)
};

}; // End of namespace


#endif // __OVITO_APPLY_MODIFIER_ACTION_H
