///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <scripting/Scripting.h>

namespace Scripting {

using namespace boost::python;

void ExportLinAlg();
void ExportTriMesh();
void ExportViewport();
void ExportData();
void ExportScene();
void ExportReference();
void ExportPlugins();
void ExportAnimation();
void ExportRendering();
void ExportActions();

struct QString_to_python_str {
	static PyObject* convert(const QString& s)	{
		return boost::python::incref(boost::python::object(std::string(s.toLocal8Bit().constData())).ptr());
	}
};

struct QString_from_python_str
{
	QString_from_python_str() {
		boost::python::converter::registry::push_back(
			&convertible,
			&construct,
			boost::python::type_id<QString>());
	}

	static void* convertible(PyObject* obj_ptr)	{
		if (!PyString_Check(obj_ptr)) return 0;
		return obj_ptr;
	}

	static void construct(PyObject* obj_ptr, boost::python::converter::rvalue_from_python_stage1_data* data) {
		const char* value = PyString_AsString(obj_ptr);
		if (value == 0) boost::python::throw_error_already_set();
		void* storage = (
			(boost::python::converter::rvalue_from_python_storage<QString>*)
			data)->storage.bytes;
		new (storage) QString(value);
		data->convertible = storage;
	}
};

void ExceptionTranslator(const Exception& ex) {
	QString s = ex.message();
	for(int i=1; i<ex.messages().size(); i++) {
		s += QChar('\n');
		s += ex.messages()[i];
	}
	PyErr_SetString(PyExc_RuntimeError, s.toLocal8Bit().constData());
}

BOOST_PYTHON_MODULE(Scripting)
{
	register_exception_translator<Exception>(&ExceptionTranslator);

	boost::python::to_python_converter<
		QString,
		QString_to_python_str>();
	QString_from_python_str();

	ExportLinAlg();
	ExportTriMesh();
	ExportPlugins();
	ExportReference();
	ExportViewport();
	ExportAnimation();
	ExportScene();
	ExportRendering();
	ExportData();
	ExportActions();
}

};
