/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment text example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <pgm/pgm.h>

PgmDrawable *txt1, *txt2, *txt3;
PgmCanvas *canvas;

static void
configure_event_cb (PgmViewport *viewport,
                    PgmEventConfigure *event,
                    gpointer data)
{
  pgm_canvas_regenerate (canvas);
}

static void
key_press_event_cb (PgmViewport *viewport,
                    PgmEventKey *event,
                    gpointer data)
{
  /* Switch monoline/multiline */
  if (event->keyval == PGM_m)
    {
      static gboolean multiline = TRUE;

      multiline = !multiline;
      pgm_text_set_multiline (PGM_TEXT (txt1), multiline);
      pgm_text_set_multiline (PGM_TEXT (txt2), multiline);
      pgm_text_set_multiline (PGM_TEXT (txt3), multiline);
    }

  /* Change ellipsizing */
  else if (event->keyval == PGM_e)
    {
      static PgmTextEllipsize ellipsize = PGM_TEXT_ELLIPSIZE_NONE;

      if (ellipsize == PGM_TEXT_ELLIPSIZE_NONE)
        ellipsize = PGM_TEXT_ELLIPSIZE_END;
      else
        ellipsize = PGM_TEXT_ELLIPSIZE_NONE;

      pgm_text_set_ellipsize (PGM_TEXT (txt1), ellipsize);
      pgm_text_set_ellipsize (PGM_TEXT (txt2), ellipsize);
      pgm_text_set_ellipsize (PGM_TEXT (txt3), ellipsize);
    }

  /* Change alignment */
  else if (event->keyval == PGM_a)
    {
      static PgmTextAlignment alignment = PGM_TEXT_ALIGN_LEFT;

      if (alignment == PGM_TEXT_ALIGN_LEFT)
        alignment = PGM_TEXT_ALIGN_CENTER;
      else if (alignment == PGM_TEXT_ALIGN_CENTER)
        alignment = PGM_TEXT_ALIGN_RIGHT;
      else
        alignment = PGM_TEXT_ALIGN_LEFT;

      pgm_text_set_alignment (PGM_TEXT (txt1), alignment);
      pgm_text_set_alignment (PGM_TEXT (txt2), alignment);
      pgm_text_set_alignment (PGM_TEXT (txt3), alignment);
    }

  /* Change shadow casting state */
  else if (event->keyval == PGM_s)
    {
      static gboolean cast_shadow = FALSE;

      cast_shadow = !cast_shadow;

      pgm_text_set_cast_shadow (PGM_TEXT (txt1), cast_shadow);
      pgm_text_set_cast_shadow (PGM_TEXT (txt2), cast_shadow);
      pgm_text_set_cast_shadow (PGM_TEXT (txt3), cast_shadow);
    }

  /* Change shadow casting state */
  else if (event->keyval == PGM_p)
    {
      static PgmTextShadowPosition shadow_position = PGM_TEXT_SHADOW_NORTH;

      if (shadow_position == PGM_TEXT_SHADOW_NORTH)
        shadow_position = PGM_TEXT_SHADOW_NORTH_EAST;
      else if (shadow_position == PGM_TEXT_SHADOW_NORTH_EAST)
        shadow_position = PGM_TEXT_SHADOW_EAST;
      else if (shadow_position == PGM_TEXT_SHADOW_EAST)
        shadow_position = PGM_TEXT_SHADOW_SOUTH_EAST;
      else if (shadow_position == PGM_TEXT_SHADOW_SOUTH_EAST)
        shadow_position = PGM_TEXT_SHADOW_SOUTH;
      else if (shadow_position == PGM_TEXT_SHADOW_SOUTH)
        shadow_position = PGM_TEXT_SHADOW_SOUTH_WEST;
      else if (shadow_position == PGM_TEXT_SHADOW_SOUTH_WEST)
        shadow_position = PGM_TEXT_SHADOW_WEST;
      else if (shadow_position == PGM_TEXT_SHADOW_WEST)
        shadow_position = PGM_TEXT_SHADOW_NORTH_WEST;
      else if (shadow_position == PGM_TEXT_SHADOW_NORTH_WEST)
        shadow_position = PGM_TEXT_SHADOW_NORTH;

      pgm_text_set_shadow_position (PGM_TEXT (txt1), shadow_position);
      pgm_text_set_shadow_position (PGM_TEXT (txt2), shadow_position);
      pgm_text_set_shadow_position (PGM_TEXT (txt3), shadow_position);
    }

  /* Change fullscreen state */
  else if (event->keyval == PGM_f)
    {
      static gboolean fullscreen = FALSE;

      fullscreen = !fullscreen;
      pgm_viewport_set_fullscreen (viewport, fullscreen);
    }
}

int
main (int argc, char *argv[])
{
  static const gchar *label =
    "Pigment is a 3D scene graph library designed to easily create rich "
    "application user interfaces";
  PgmViewport *viewport = NULL;

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_title (viewport, "Text");

  /* Text creation */
  txt1 = pgm_text_new (NULL);
  pgm_text_set_markup (PGM_TEXT (txt1), label);
  pgm_text_set_font_height (PGM_TEXT (txt1), 0.4f);
  pgm_drawable_set_size (txt1, 1.0f, 2.5f);
  pgm_drawable_set_position (txt1, 0.25f, 0.25f, 0.0f);
  pgm_drawable_set_fg_color (txt1, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (txt1, 120, 120, 120, 255);
  pgm_drawable_show (txt1);

  /* Text creation */
  txt2 = pgm_text_new (NULL);
  pgm_text_set_markup (PGM_TEXT (txt2), label);
  pgm_text_set_font_height (PGM_TEXT (txt2), 0.25f);
  pgm_drawable_set_size (txt2, 1.0f, 2.5f);
  pgm_drawable_set_position (txt2, 1.5f, 0.25f, 0.0f);
  pgm_drawable_set_fg_color (txt2, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (txt2, 120, 120, 120, 255);
  pgm_drawable_show (txt2);

  /* Text creation */
  txt3 = pgm_text_new (NULL);
  pgm_text_set_markup (PGM_TEXT (txt3), label);
  pgm_text_set_font_height (PGM_TEXT (txt3), 0.1f);
  pgm_drawable_set_size (txt3, 1.0f, 2.5f);
  pgm_drawable_set_position (txt3, 2.75f, 0.25f, 0.0f);
  pgm_drawable_set_fg_color (txt3, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (txt3, 120, 120, 120, 255);
  pgm_drawable_show (txt3);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 4.0f, 3.0f);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add_many (canvas, PGM_DRAWABLE_MIDDLE, txt1, txt2, txt3, NULL);

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (key_press_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  g_signal_connect (G_OBJECT (viewport), "configure-event",
                    G_CALLBACK (configure_event_cb), NULL);
  pgm_viewport_show (viewport);
  pgm_main ();

  /* Deinit */
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  pgm_deinit ();

  return 0;
}
