

/* 
A* -------------------------------------------------------------------
B* This file contains source code for the PyMOL computer program
C* Copyright (c) Schrodinger, LLC. 
D* -------------------------------------------------------------------
E* It is unlawful to modify or remove this copyright notice.
F* -------------------------------------------------------------------
G* Please see the accompanying LICENSE file for further information. 
H* -------------------------------------------------------------------
I* Additional authors of this source file include:
-* 
-* 
-*
Z* -------------------------------------------------------------------
*/
#ifndef _H_MemoryDebug
#define _H_MemoryDebug

#include <vector>

#include "os_std.h"
#include "PyMOLGlobals.h"


/* ================================================================ 
 * Don't touch below unless you know what you are doing */

typedef struct VLARec {
  ov_size size, unit_size;
  float grow_factor;
  int auto_zero;
} VLARec;


/* NOTE: in VLACheck, rec is a zero based array index, not a record count */
#define VLACheck(ptr,type,rec) VLACheck2<type>(ptr, rec)

#define VLAlloc(type,init_size) (type*)VLAMalloc(init_size,sizeof(type),5,0)
#define VLACalloc(type,init_size) (type*)VLAMalloc(init_size,sizeof(type),5,1)
#define VLASize(ptr,type,size) VLASize2<type>(ptr,size)
#define VLASizeForSure(ptr,type,size) VLASizeForSure2<type>(ptr,size)

#define VLACopy(ptr,type) (type*)VLANewCopy(ptr);
#define VLAInsert(ptr,type,index,count) {ptr=(type*)VLAInsertRaw(ptr,index,count);}
#define VLADelete(ptr,type,index,count) {ptr=(type*)VLADeleteRaw(ptr,index,count);}

#define Alloc(type,size) (type*)mmalloc(sizeof(type)*(size))
#define Calloc(type,size) (type*)mcalloc(sizeof(type),size)
#define Realloc(ptr,type,size) (type*)mrealloc(ptr,sizeof(type)*(size))

#define FreeP(ptr) {if(ptr) {mfree(ptr);ptr=NULL;}}
#define DeleteP(ptr) {if(ptr) {delete ptr;ptr=NULL;}}
#define DeleteAP(ptr) {if(ptr) {delete[] ptr;ptr=NULL;}}

void *VLAExpand(void *ptr, ov_size rec);        /* NOTE: rec is index (total-1) */
void *MemoryReallocForSure(void *ptr, unsigned int newSize);
void *MemoryReallocForSureSafe(void *ptr, unsigned int newSize, unsigned int oldSize);

void *VLADeleteRaw(void *ptr, int index, unsigned int count);
void *VLAInsertRaw(void *ptr, int index, unsigned int count);

void *VLAMalloc(ov_size init_size, ov_size unit_size, unsigned int grow_factor, int auto_zero); /*growfactor 1-10 */

void VLAFree(void *ptr);
void *VLASetSize(void *ptr, unsigned int newSize);
void *VLASetSizeForSure(void *ptr, unsigned int newSize);

unsigned int VLAGetSize(const void *ptr);
void *VLANewCopy(const void *ptr);
void MemoryZero(char *p, char *q);


#define mcalloc calloc
#define mmalloc malloc
#define mrealloc realloc
#define mfree free
#define mstrdup strdup
#define ReallocForSure(ptr,type,size) (type*)MemoryReallocForSure(ptr,sizeof(type)*(size))
#define ReallocForSureSafe(ptr,type,size,old_size) (type*)MemoryReallocForSure(ptr,sizeof(type)*(size),sizeof(type)*(old_size))

#ifdef __cplusplus
#define mnew new
#endif

#define MD_FILE_LINE_Call
#define MD_FILE_LINE_Decl
#define MD_FILE_LINE_Nest
#define MD_FILE_LINE_PTR_Call


inline unsigned int VLAGetByteSize(const void *ptr) {
  const VLARec *vla = ((const VLARec *) ptr) - 1;
  return vla->size * vla->unit_size;
}

/*
 * Templated version of the `VLACopy` macro
 */
template <typename T>
T * VLACopy2(const T * vla) {
  return VLACopy((void*)vla, T);
}

/*
 * @brief std::vector version of VLACheck. Checks to see if index i is valid for insertion.
 *        If not, a resize will be attempted.
 * @param vec: vector whose size will be check for valid insertion at index i
 * @param i: index for position where an element may be inserted into vec
 * Note: Use of this function should be limited. Used for safe replacement of VLACheck
 * Note: This function can throw.
 */
template <typename T>
void VecCheck(std::vector<T> &vec, std::size_t i){
  if(i >= vec.size()){
    vec.resize(i + 1);
  }
}

template <typename T>
T* VLACheck2(T*& ptr, size_t pos) {
  if (pos >= ((VLARec*) ptr)[-1].size) {
    ptr = static_cast<T*>(VLAExpand(ptr, pos));
  }
  return ptr;
}

template <typename T>
void VLASize2(T*& ptr, size_t size) {
  ptr = static_cast<T*>(VLASetSize(ptr, size));
}

template <typename T>
void VLASizeForSure2(T*& ptr, size_t size) {
  ptr = static_cast<T*>(VLASetSizeForSure(ptr, size));
}

template <typename T>
void VLAFreeP(T*& ptr) {
  if (ptr) {
    VLAFree(ptr);
    ptr = nullptr;
  }
}

#endif

// vi:sw=2:expandtab
