% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hdi.R
\name{hdi}
\alias{hdi}
\alias{hdi.numeric}
\alias{hdi.data.frame}
\alias{hdi.MCMCglmm}
\alias{hdi.sim.merMod}
\alias{hdi.sim}
\alias{hdi.emmGrid}
\alias{hdi.stanreg}
\alias{hdi.brmsfit}
\alias{hdi.BFBayesFactor}
\title{Highest Density Interval (HDI)}
\usage{
hdi(x, ...)

\method{hdi}{numeric}(x, ci = 0.89, verbose = TRUE, ...)

\method{hdi}{data.frame}(x, ci = 0.89, verbose = TRUE, ...)

\method{hdi}{MCMCglmm}(x, ci = 0.89, verbose = TRUE, ...)

\method{hdi}{sim.merMod}(
  x,
  ci = 0.89,
  effects = c("fixed", "random", "all"),
  parameters = NULL,
  verbose = TRUE,
  ...
)

\method{hdi}{sim}(x, ci = 0.89, parameters = NULL, verbose = TRUE, ...)

\method{hdi}{emmGrid}(x, ci = 0.89, verbose = TRUE, ...)

\method{hdi}{stanreg}(
  x,
  ci = 0.89,
  effects = c("fixed", "random", "all"),
  parameters = NULL,
  verbose = TRUE,
  ...
)

\method{hdi}{brmsfit}(
  x,
  ci = 0.89,
  effects = c("fixed", "random", "all"),
  component = c("conditional", "zi", "zero_inflated", "all"),
  parameters = NULL,
  verbose = TRUE,
  ...
)

\method{hdi}{BFBayesFactor}(x, ci = 0.89, verbose = TRUE, ...)
}
\arguments{
\item{x}{Vector representing a posterior distribution. Can also be a
\code{stanreg}, \code{brmsfit} or a \code{BayesFactor} model.}

\item{...}{Currently not used.}

\item{ci}{Value or vector of probability of the (credible) interval - CI (between 0 and 1)
to be estimated. Default to \code{.89} (89\%).}

\item{verbose}{Toggle off warnings.}

\item{effects}{Should results for fixed effects, random effects or both be returned?
Only applies to mixed models. May be abbreviated.}

\item{parameters}{Regular expression pattern that describes the parameters that
should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}

\item{component}{Should results for all parameters, parameters for the conditional model
or the zero-inflated part of the model be returned? May be abbreviated. Only
applies to \pkg{brms}-models.}
}
\value{
A data frame with following columns:
  \itemize{
    \item \code{Parameter} The model parameter(s), if \code{x} is a model-object. If \code{x} is a vector, this column is missing.
    \item \code{CI} The probability of the credible interval.
    \item \code{CI_low}, \code{CI_high} The lower and upper credible interval limits for the parameters.
  }
}
\description{
Compute the \strong{Highest Density Interval (HDI)} of posterior distributions. All points within this interval have a higher probability density than points outside the interval. The HDI can be used in the context of uncertainty characterisation of posterior distributions as \strong{Credible Interval (CI)}.
}
\details{
Unlike equal-tailed intervals (see \code{eti()}) that typically exclude 2.5\%
from each tail of the distribution and always include the median, the HDI is
\emph{not} equal-tailed and therefore always includes the mode(s) of posterior
distributions.
\cr \cr
By default, \code{hdi()} and \code{eti()} return the 89\% intervals (\code{ci = 0.89}),
deemed to be more stable than, for instance, 95\% intervals (\cite{Kruschke, 2014}).
An effective sample size of at least 10.000 is recommended if 95\% intervals
should be computed (\cite{Kruschke, 2014, p. 183ff}). Moreover, 89 indicates
the arbitrariness of interval limits - its only remarkable property is being
the highest prime number that does not exceed the already unstable 95\%
threshold (\cite{McElreath, 2015}).
\cr \cr
A 90\% equal-tailed interval (ETI) has 5\% of the distribution on either
side of its limits. It indicates the 5th percentile and the 95h percentile.
In symmetric distributions, the two methods of computing credible intervals,
the ETI and the \link[=hdi]{HDI}, return similar results.
\cr \cr
This is not the case for skewed distributions. Indeed, it is possible that
parameter values in the ETI have lower credibility (are less probable) than
parameter values outside the ETI. This property seems undesirable as a summary
of the credible values in a distribution.
\cr \cr
On the other hand, the ETI range does change when transformations are applied
to the distribution (for instance, for a log odds scale to probabilities):
the lower and higher bounds of the transformed distribution will correspond
to the transformed lower and higher bounds of the original distribution.
On the contrary, applying transformations to the distribution will change
the resulting HDI.
}
\examples{
library(bayestestR)

posterior <- rnorm(1000)
hdi(posterior, ci = .89)
hdi(posterior, ci = c(.80, .90, .95))

df <- data.frame(replicate(4, rnorm(100)))
hdi(df)
hdi(df, ci = c(.80, .90, .95))

\dontrun{
library(rstanarm)
model <- stan_glm(mpg ~ wt + gear, data = mtcars, chains = 2, iter = 200, refresh = 0)
hdi(model)
hdi(model, ci = c(.80, .90, .95))

library(emmeans)
hdi(emtrends(model, ~1, "wt"))

library(brms)
model <- brms::brm(mpg ~ wt + cyl, data = mtcars)
hdi(model)
hdi(model, ci = c(.80, .90, .95))

library(BayesFactor)
bf <- ttestBF(x = rnorm(100, 1, 1))
hdi(bf)
hdi(bf, ci = c(.80, .90, .95))
}

}
\references{
\itemize{
  \item Kruschke, J. (2014). Doing Bayesian data analysis: A tutorial with R, JAGS, and Stan. Academic Press.
  \item McElreath, R. (2015). Statistical rethinking: A Bayesian course with examples in R and Stan. Chapman and Hall/CRC.
}
}
\author{
Credits go to \href{https://rdrr.io/cran/ggdistribute/src/R/stats.R}{ggdistribute} and \href{https://github.com/mikemeredith/HDInterval}{HDInterval}.
}
