///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/fem_helper.h"
using namespace rheolef;
using namespace std;

// ============================================================================
//  coordinate system helper
// ============================================================================
const char* 
fem_helper::coord_sys_table [max_coord_sys] = {
    "cartesian",
    "rz",
    "zr"
};
string 
fem_helper::coordinate_system_name (coordinate_type i)
{
    assert_macro (i < max_coord_sys, "invalid coordinate_type " << i);
    return coord_sys_table [i];
}
fem_helper::coordinate_type
fem_helper::coordinate_system (string sys_coord)
{
    for (size_type i = 0; i < max_coord_sys; i++)
        if (sys_coord == coord_sys_table[i]) return coordinate_type(i);
    error_macro ("unexpected coordinate system `" << sys_coord << "'");
    return max_coord_sys;
}
void
fem_helper::check_coord_sys_and_dimension (coordinate_type i, size_type d)
{
    assert_macro (i < max_coord_sys, "invalid coordinate_type " << i);
    check_macro (!((i == axisymmetric_rz || i == axisymmetric_zr) && d > 2),
	"inconsistent `" << coord_sys_table[i] 
	 << "' coordinate system for " << d << "D geometry"); 
}
// =================================================================================
// tensor-valued fields helper
// =================================================================================
const char*
fem_helper::valued_field_table [max_valued_field] = {
    "scalar",
    "vector",
    "tensor",
    "unsymmetric_tensor",
    "tensor4",
    "hybrid"
};
// ---------------------------------------------------------------------------------
// symmetric 2-tensors
// ---------------------------------------------------------------------------------
fem_helper::size_type
fem_helper::symmetric_tensor_index [3][3] = {
	{ 0, 1, 3},
	{ 1, 2, 4},
	{ 3, 4, 5}
};
fem_helper::pair_size_type
fem_helper::symmetric_tensor_subscript [6] = {
	{0,0}, {0,1}, {1,1}, {0,2}, {1,2}, {2,2} };
const char*
fem_helper::symmetric_tensor_subscript_name [6] = {
	"00",  "01",  "11",  "02",  "12",  "22" };

// ---------------------------------------------------------------------------------
// unsymmetric 2-tensors
// ---------------------------------------------------------------------------------
fem_helper::size_type
fem_helper::unsymmetric_tensor_index [3][3] = {
	{ 0, 1, 4},
	{ 2, 3, 5},
	{ 6, 7, 8}
};
fem_helper::pair_size_type
fem_helper::unsymmetric_tensor_subscript [9] = {
	{0,0}, {0,1}, {1,0}, {1,1}, {0,2}, {1,2}, {2,0}, {2,1}, {2,2} };
const char* 
fem_helper::unsymmetric_tensor_subscript_name [9] = {
	"00",  "01",  "10",  "11",  "02",  "12",  "20",  "21",  "22" };

// ---------------------------------------------------------------------------------
// tensors usefull functions
// ---------------------------------------------------------------------------------
fem_helper::pair_size_type 
fem_helper::make_pair_size(size_type a, size_type b)
{
    pair_size_type x;
    x.first = a;
    x.second = b;
    return x;
}
fem_helper::double_pair_size_type 
fem_helper::make_double_pair_size(size_type a, size_type b, size_type c, size_type d)
{
    double_pair_size_type x;
    x.first.first = a;
    x.first.second = b;
    x.second.first = c;
    x.second.second = d;
    return x;
}
string
fem_helper::valued_field_name (valued_field_type i)
{
    assert_macro (i < max_valued_field, "invalid valued_field_type " << i);
    return valued_field_table [i];
}
fem_helper::valued_field_type 
fem_helper::valued_field (string valued)
{
    for (size_type i = 0; i < max_valued_field; i++)
        if (valued == valued_field_table[i]) return valued_field_type(i);
    error_macro ("unexpected valued field `" << valued << "'");
    return max_valued_field;
}
// -----------------------------------------------------------------------
// 2-tensor
// -----------------------------------------------------------------------
fem_helper::size_type
fem_helper::tensor_index (
    valued_field_type valued,
    coordinate_type   sys_coord,
    size_type         i,
    size_type         j)
{
    if (valued == unsymmetric_tensorial) {
      if ((sys_coord == axisymmetric_rz || sys_coord == axisymmetric_zr)
	  && i == 2 && j == 2)
	  return 4;
      return unsymmetric_tensor_index [i][j];
    } else {
      if ((sys_coord == axisymmetric_rz || sys_coord == axisymmetric_zr)
	  && i == 2 && j == 2)
	  return 3;
      return   symmetric_tensor_index [i][j];
    }
}
fem_helper::size_type
fem_helper::tensor_index (
    string      valued,
    string      sys_coord, 
    size_type   i, 
    size_type   j)
{
    return tensor_index (valued_field(valued), coordinate_system(sys_coord), i, j);
}
fem_helper::pair_size_type
fem_helper::tensor_subscript (
    valued_field_type valued,
    coordinate_type   sys_coord,
    size_type         i_comp)
{
    if (valued == unsymmetric_tensorial) {
      if ((sys_coord == axisymmetric_rz || sys_coord == axisymmetric_zr)
	  && i_comp == 4)
	 return make_pair_size(2,2);
      return unsymmetric_tensor_subscript [i_comp];
    } else {
      if ((sys_coord == axisymmetric_rz || sys_coord == axisymmetric_zr)
	 && i_comp == 3)
	 return make_pair_size(2,2);
      return   symmetric_tensor_subscript [i_comp];
    }
}
string
fem_helper::tensor_subscript_name (
    valued_field_type valued,
    coordinate_type   sys_coord,
    size_type         i_comp)
{
    if (valued == unsymmetric_tensorial) {
      if ((sys_coord == axisymmetric_rz || sys_coord == axisymmetric_zr)
          && i_comp == 4)
	return "22";
      return unsymmetric_tensor_subscript_name [i_comp];
    } else {
      if ((sys_coord == axisymmetric_rz || sys_coord == axisymmetric_zr)
	&& i_comp == 3) 
        return "22";
      return   symmetric_tensor_subscript_name [i_comp];
    }
}
fem_helper::pair_size_type
fem_helper::tensor_subscript (
    string     valued,
    string     sys_coord,
    size_type  i_comp)
{
    return tensor_subscript (valued_field(valued),
	         coordinate_system(sys_coord), i_comp);
}
string
fem_helper::tensor_subscript_name (
    string     valued,
    string     sys_coord,
    size_type  i_comp)
{
    return tensor_subscript_name (valued_field(valued),
	         coordinate_system(sys_coord), i_comp);
}
// -----------------------------------------------------------------------
// 4-tensor
// -----------------------------------------------------------------------
fem_helper::size_type
fem_helper::tensor4_index (
    valued_field_type valued,
    coordinate_type   sys_coord,
    size_type         i,
    size_type         j,
    size_type         k,
    size_type         l)
{
    size_type ij = tensor_index (tensorial, sys_coord, i, j);
    size_type kl = tensor_index (tensorial, sys_coord, k, l);
    return symmetric_tensor4_index[ij][kl];
}
fem_helper::size_type
fem_helper::tensor4_index (
    string      valued,
    string      sys_coord, 
    size_type   i,
    size_type   j,
    size_type   k,
    size_type   l)
{
    return tensor4_index (valued_field(valued), coordinate_system(sys_coord), i, j, k, l);
}
fem_helper::double_pair_size_type
fem_helper::tensor4_subscript (
    valued_field_type valued,
    coordinate_type   sys_coord,
    size_type         i_comp)
{
    if (sys_coord == axisymmetric_rz || sys_coord == axisymmetric_zr) {
      return symmetric_tensor4_subscript_rz [i_comp];
    } else {
      return symmetric_tensor4_subscript    [i_comp];
    }
}
string
fem_helper::tensor4_subscript_name (
    valued_field_type valued,
    coordinate_type   sys_coord,
    size_type         i_comp)
{
    if (sys_coord == axisymmetric_rz || sys_coord == axisymmetric_zr) {
      return symmetric_tensor4_subscript_rz_name [i_comp];
    } else {
      return symmetric_tensor4_subscript_name    [i_comp];
    }
}
fem_helper::double_pair_size_type
fem_helper::tensor4_subscript (
    string     valued,
    string     sys_coord,
    size_type  i_comp)
{
    return tensor4_subscript (valued_field(valued),
	         coordinate_system(sys_coord), i_comp);
}
string
fem_helper::tensor4_subscript_name (
    string     valued,
    string     sys_coord,
    size_type  i_comp)
{
    return tensor4_subscript_name (valued_field(valued),
	         coordinate_system(sys_coord), i_comp);
}
// =================================================================================
// symmetric 4-tensors
// Note: 3 symmetries for 4-order tensors
//     A_ijkl=A_ijlk, A_ijkl=A_jikl & A_ijkl=Aklij
// => 6 components in 2d, 21 in 3d and 10 in 2d-axisymmetric(rz)
// =================================================================================

fem_helper::size_type
fem_helper::symmetric_tensor4_index [6][6] = {
	{  0,  1,  3,  6, 10, 15},
	{  1,  2,  4,  7, 11, 16},
	{  3,  4,  5,  8, 12, 17},
	{  6,  7,  8,  9, 13, 18},
	{ 10, 11, 12, 13, 14, 19},
	{ 15, 16, 17, 18, 19, 20}
};
fem_helper::double_pair_size_type
fem_helper::symmetric_tensor4_subscript [21] = {
	{{0,0}, {0,0}},
	{{0,0}, {0,1}},
	{{0,1}, {0,1}},
	{{0,0}, {1,1}},
	{{0,1}, {1,1}},
	{{1,1}, {1,1}},
	{{0,0}, {0,2}},
	{{0,1}, {0,2}},
	{{1,1}, {0,2}},
	{{0,2}, {0,2}},
	{{0,0}, {1,2}},
	{{0,1}, {1,2}},
	{{1,1}, {1,2}},
	{{0,2}, {1,2}},
	{{1,2}, {1,2}},
	{{0,0}, {2,2}},
	{{0,1}, {2,2}},
	{{1,1}, {2,2}},
	{{0,2}, {2,2}},
	{{1,2}, {2,2}},
	{{2,2}, {2,2}} };

const char*
fem_helper::symmetric_tensor4_subscript_name [21] = {
	"00_00",
	"00_01",
	"01_01",
	"00_11",
	"01_11",
	"11_11",
	"00_02",
	"01_02",
	"11_02",
	"02_02",
	"00_12",
	"01_12",
	"11_12",
	"02_12",
	"12_12",
	"00_22",
	"01_22",
	"11_22",
	"02_22",
	"12_22",
	"22_22" };

// special axisymmetic case
fem_helper::double_pair_size_type
fem_helper::symmetric_tensor4_subscript_rz [10] = {
	{{0,0}, {0,0}},
	{{0,0}, {0,1}},
	{{0,1}, {0,1}},
	{{0,0}, {1,1}},
	{{0,1}, {1,1}},
	{{1,1}, {1,1}},
	{{0,0}, {2,2}},
	{{0,1}, {2,2}},
	{{1,1}, {2,2}},
	{{2,2}, {2,2}} };

const char*
fem_helper::symmetric_tensor4_subscript_rz_name [10] = {
	"00_00",
	"00_01",
	"01_01",
	"00_11",
	"01_11",
	"11_11",
	"00_22",
	"01_22",
	"11_22",
	"22_22" };

// ============================================================================
// approximation name -> basis and numbering names
// ============================================================================
std::string 
fem_helper::basis_name (std::string approx_name)
{
    if (approx_name == "P1d" || approx_name == "P1pw") return "P1";
    if (approx_name == "P2d") return "P2";
    return approx_name;
}
std::string 
fem_helper::numbering_name (std::string approx_name)
{
    return approx_name + "_numbering";
}
