///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// compute:
//  \int_\Omega 2 D(u):D(v) dx dy
//
// check value on the unit square [0,1]^2
//
// component 0 : we check the u_x component : u_y = 0
//               and u_x takes monomial values : 1, x, ...
// component 1 : u_x = 0 and u_y is monomial
//
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

string der(const string& approx)
 {
  if (approx=="P1") return "P0";
  else if (approx=="P2") return "P1d";
  error_macro("Unimplemented");
 }


typedef Float (*function_type)(const point&);

struct list_type {
	string        name;
	function_type function;
};
Float monom_1  (const point& x) { return 1; }
Float monom_x  (const point& x) { return x[0]; }
Float monom_y  (const point& x) { return x[1]; }
Float monom_x2 (const point& x) { return x[0]*x[0]; }
Float monom_xy (const point& x) { return x[0]*x[1]; }
Float monom_y2 (const point& x) { return x[1]*x[1]; }

list_type xy_monom_list [] = {
	//   monom
	// id   fct    
	{"1",	monom_1},
	{"r",	monom_x},
	{"z",	monom_y},
	{"r2",	monom_x2},
	{"rz",	monom_xy},
	{"z2",	monom_y2}
};
const unsigned int xy_monom_max = sizeof(xy_monom_list)/sizeof(list_type);

typedef Float T;

const Float infty = -1;

Float  expect_table [2][4] [xy_monom_max] = {
  {
    // a([ur,0],[1,0,0,0]) = 2Du_00
    {//         1       r       z       r2      rz       z2

	      0,      1,      0,    T(4./3),  T(1./2),       0
    },
    // a([ur,0],[0,1,0,0]) = 2*(2Du_01)
    {
		0, 	0,  	1,	0,    T(2./3),	 1
    },
    // a([ur,0],[0,0,1,0]) = 2Du_11
    {
		0, 	0,	0,	0, 	0,	 0
    },
    // a([ur,0],[0,0,0,1]) = 2Du_22
    {
		2, 	1,	1,  T(2./3), T(1/2.),	 T(2./3)
    }
  },
  {
    // a([0,uz],[1,0,0,0]) = 2Du_00
    {//         1       r       z       r2      rz       z2

	       0,      0,      0,	0,      0,       0
    },
    // a([0,uz],[0,1,0,0]) = 2*(2Du_01)
    {
		0, 	1,	0,  	1, T(2./3),	 0
    },
    // a([0,uz],[0,0,1,0]) = 2Du_11
    {
		0, 	0,	1,	0,  T(1./2),	T(4./3)
    },
    // a([0,uz],[0,0,0,1]) = 2Du_22
    {
		0, 	0,	0,	0, 	0,	 0
    }
  }
};
void usage()
{
      cerr << "form_2D_rz_tst: usage: form_2D_rz_tst"
	   << " {-Igeodir}*"
	   << " -|mesh[.geo]"
	   << " [-approx string]"
	   << " [-u-component int]"
	   << " [-tau-component int]"
	   << " [-u-monom string]"
	   << endl;
      cerr << "example:\n";
      cerr << "  form_2D_rz_tst square -u-monom rz -u-component 0 -tau-component 3\n";
      exit (1);
}
int main(int argc, char**argv)
{
    //
    // load geometry and options
    //
    geo omega;  
    string coord_sys = "rz";
    string approx = "P2";
    bool mesh_done = false;
    size_t i_comp_u = 0;
    size_t i_comp_tau = 0;
    string        u_monom_id = "";
    function_type u_monom_function = 0;
    unsigned int  u_monom_idx = 0;

    if (argc <= 1) usage() ;

    for (int i = 1; i < argc; i++ ) {

      if (argv [i][0] == '-' && argv [i][1] == 'I')
	  append_dir_to_rheo_path (argv[i]+2) ;
      else if (strcmp(argv[i], "-approx") == 0)
	  approx = argv[++i];
      else if (strcmp(argv[i], "-u-component") == 0)
	  i_comp_u = atoi(argv[++i]);
      else if (strcmp(argv[i], "-tau-component") == 0)
	  i_comp_tau = atoi(argv[++i]);
      else if (strcmp(argv[i], "-u-monom") == 0) {
	  u_monom_id = argv[++i];
          for (unsigned int i = 0; i < xy_monom_max; i++) {
            if (u_monom_id == xy_monom_list [i].name) {
	      u_monom_function = xy_monom_list [i].function;
	      u_monom_idx = i;
            }
          }
      } else if (strcmp(argv[i], "-") == 0) {
	  // input geo on standard input
	  if (mesh_done) usage() ;
	  cerr << " ! load geo on stdin" << endl ;
	  cin >> omega ;
	  mesh_done = true ;
      } else {
	  // input geo on file
	  if (mesh_done) usage() ;
	  cerr << " ! load " << argv[i] << endl ;
	  omega = geo(argv[i]);
	  mesh_done = true ;
      }
    }
    if (!mesh_done) {
	cerr << "form_2D_rz_tst: no mesh specified" << endl;
	usage() ;
    }
    omega.set_coordinate_system (coord_sys);
    if (!u_monom_function) {
	error_macro("invalid u-monom identifier: " << u_monom_id);
    }
    cerr << "syscoord = " << omega.coordinate_system() << endl;
    cerr << "u_monom = " << u_monom_id << endl;
    space V0h(omega, approx);
    space Qh(omega, der(approx), "tensor");
    space Vh = V0h*V0h;
    field u_monom (Vh, Float(0));
    field v_monom (Qh, Float(0));
	v_monom[i_comp_tau] = 1;
    u_monom[i_comp_u] = interpolate (V0h, u_monom_function);
    form  a1 (Vh,Qh,"2D");
	form invm(Qh,Qh,"inv_mass");
    Float result = a1(u_monom, v_monom);
    string u_vec_monom_id;
    string v_vec_monom_id;
    if (i_comp_u == 0) u_vec_monom_id = u_monom_id + ",0";
    else               u_vec_monom_id = "0," + u_monom_id;
    if (i_comp_tau == 0) v_vec_monom_id = "00";
    else 
    if (i_comp_tau == 1) v_vec_monom_id = "01";
    else 
    if (i_comp_tau == 2) v_vec_monom_id = "11";
    cerr << setprecision(numeric_limits<Float>::digits10)
         << "a(omega," << approx << "[" << u_vec_monom_id << "]," << approx << "[" << v_vec_monom_id << "]) = " << double(result) << endl;
    Float expect = expect_table [i_comp_u][i_comp_tau][u_monom_idx];
    Float tol = 1e-10;

    if (expect == infty) {
        cerr << "ok (expect infty)" << endl;
        return 0;
    } else if (fabs(result - expect) <= tol) {
        cerr << "ok" << endl;
        return 0;
    } else {
        cerr << "but it was expected " << expect << endl;
	cerr << "and error = " << fabs(result - expect) << endl;
	cerr << "and tol = " << tol << endl;
        return 1;
    }
}
