use fips203::ml_kem_512;
use fips203::traits::{Decaps, Encaps, KeyGen, SerDes};
use hex_literal::hex;
use rand_core::SeedableRng;


#[test]
fn wasm_match() {
    let seed = 123; // Note: this should match the value giving in the web form
    let mut rng = rand_chacha::ChaCha8Rng::seed_from_u64(seed);

    // Alice runs KeyGen, and serializes ek for Bob (to bytes)
    let (alice_ek, alice_dk) = ml_kem_512::KG::try_keygen_with_rng(&mut rng).unwrap();
    let alice_ek_bytes = alice_ek.into_bytes();

    // Alice sends ek bytes to Bob
    let bob_ek_bytes = alice_ek_bytes;

    // Bob deserializes ek bytes, runs Encaps, to get ssk and serializes ct for Alice (to bytes)
    let bob_ek = ml_kem_512::EncapsKey::try_from_bytes(bob_ek_bytes).unwrap();
    let (bob_ssk, bob_ct) = bob_ek.try_encaps_with_rng(&mut rng).unwrap();
    let bob_ct_bytes = bob_ct.into_bytes();

    // Bob sends ct bytes to Alice
    let alice_ct_bytes = bob_ct_bytes;

    // Alice deserializes runs Decaps
    let alice_ct = ml_kem_512::CipherText::try_from_bytes(alice_ct_bytes).unwrap();
    let alice_ssk = alice_dk.try_decaps(&alice_ct).unwrap();

    // Alice and Bob now have the same shared secret key
    assert_eq!(bob_ssk, alice_ssk);

    // Double check the correspondence of the serialized keypair (we already have alice_ek)
    let alice_dk_bytes = alice_dk.into_bytes();
    assert!(ml_kem_512::KG::validate_keypair_vartime(&alice_ek_bytes, &alice_dk_bytes));

    assert_eq!(bob_ek_bytes, hex!("5193763f122ed6da1fec7390b4b838dc1720741247e19a51be568cd6042fa9300f93217e14c0917be1317b8276e1e5ccbf3038c2222384ac7d736010e9051122649a8dfba4808c2a29a73d6db9adc070b35874125a205c5de33a75a271c638a16db850bd98c3e616aa513995f5f864cd792fc7b7929285049643afde6274b0c529e82a606789513ed98400ec2635a8bfa4507d4faa9d6dc169c3815a6951b9a7796e74c2ca0d0450dc3094ced76a70b8c3011c77cfa20d9d563622884aee656929f9280a77acbc70258aca1d3132c026043d60ea85dff44477c81e5c5689412236f7c87f76a7b656081e4f0c3cdbca5204152a00fb2508bac5e3f3aba9529f97c882d21283808977846b1b4cb6ae431589ac061a21316a88bb3889039ff434c3ca2887f299814d549d6e7007019a2924e692875489cf69a3a4817bf2157549053f2b284324860ef04bbc1faa7645358465c8019a38a87c91147e76bb7697653850392bb77c662071f5b80170f267de59b3a488bf41176f17b43f1943b81df45e8e377329da01a12c36d0c48a06b18606f9cdd2c25fb9ba68b506a19ca913535ac1c2733d42615ce37217daf865aaec1a4d6a8fc5918ac799acaa2b7a4e0ba50d18935a5ccae362a47afc74ce0b39a9d66a5773018c52a0aeda6e3f82bec470551d1ac0abd7124900631e8ba44232b4176219e386cd57f3b9a2278c9886155d11c9c6055a40a91366e26ab747180b76a230eb65e32ca5dec3c198366b8a06204be6b5357a78c8347123000b89701af98082d2823ac8177c3cb1acc64bc4ba0065f136ce8b92663e202c58085931b88323d41a5e6590bd67c1f3390a2bc460b005a04f85541928c5f7b2482c522c610b9a12646660c8c795b55284c0ae4aeb266af8691e6a360112967d1b5f4bfa24612831c15cadfcdc0b97138a80fa1f9361ab0774853276a9701732949b829264584d3aa2b1fb4f00219f1d0b8c3a3347a8a5015a93aac9586c937a287f3a8235dc27861a94aea8beded8cf6cd418865bac2e0a86b0475cac5382a245cb976271db74aba65425b5d31b1d7645e00a790f5a7d9baa91c518e85bfb04c6ff8a102d6a4afb159777b7f3ce754861b693f6998cb1"), "ek does not match");
    assert_eq!(bob_ct_bytes, hex!("92455fbb204e41647129e71c35e9a7dd72e8e4138d6f2e4285a8652ae33fc653f0bfafb27440968af24134a586dc15b516d0dfe2aea14db6df01218f6eb5672e39503506f4b2916cf87c55edb1536b6b7646cf0bd69651cdcb643cb220bde16d9ab37a2b2295a295f306ac4e79403dbf394c56278f77fbf27e3ff00a02753e64d3cc1fe3c0f2dc2454f79475dc167a14764c84c121180708cf034a1dac4cf3da3f55c363cd8025f26c51e85531d8da08fdb578837892391e7b027126f1e2eff9a937043bf387f01f4a4b70a9ace6f45f3c794925df503a3119b19db1bd372b2954edec2ff4eee81f189dd9424c2d867b4186e586e960f597e178ec5e006f53d54aa10841ee4911f1e2acd4c88cbc3d42f0ccf36d07c689d3f3df0a5c649b74d74b035862aae8176756919ba8c3eee699c2942d1af5820733a433ecb3ea094123881b9b819fa4f9f0eadd86b10c2635d9725fab4038e4eef835f78b37f8ecc55475296a3413189265a108b8c99c8f5ce8d7af3c6f06bd4f1b6ff3b6ace389f9c5961c446f32424eca4d214cdea2bcfdcb53e81be8f3b2f12326652dbfbbad073e5acb75702f717e6420b354de539dca9f86decd6d86bf79f05ee68c2edd363668391fd9e6d526cf400f7e89d4e0fa150f8258d21508876795f855788f3ee02e775003a225fedcf940bba8f0bd8cb41e0141f208cb601be981b8abed26f1ff9b5db9a30271fd8f64d0ede4044d23fba9163ed05ee4e8ce2010067037cbde67e9c58e4b6f48e2b4c3da274092a05c0850405e15b6a2f40ca7b6f52780c3e61d4eebc16ff82d5c1ab0f166e62bf0860688550458878bda1efbdcf76d492dd45cc189e89c17d82a8b2bd8510ba45c3e41e609d39c79e6bb3766069987d8cb7b61556abadb1a33805a6d2be7fab7f6023fb5c9d62870bb6dcf7067899f47ce91ebd06b5f51c2dcdda6dd765b2ed8d8d0f0941a0025dd8b0b1bf2a7ec902d010dc00d2e0429ef24b98cb109182a0b5e7ad140448ed8413820bdfd684f08ba8d7c61b1e13c5381f2e4d654c106d91ea5adb2c0e850ac19e055ff472abfcfd0d71a0ac827"), "ct does not match");
    assert_eq!(alice_dk_bytes, hex!("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"), "dk does not match");
    assert_eq!(
        alice_ssk.into_bytes(),
        hex!("ae36ee54c1a7362316d47e8d7b867d1adf25470064034b8410b4ac4c2ab01de0"),
        "ssk does not match"
    );
}
