use std::path::PathBuf;
use std::sync::Arc;

use capnp::capability::Promise;

use sequoia_ipc::capnp_rpc as capnp_rpc;
use capnp_rpc::pry;
use capnp_rpc::rpc_twoparty_capnp::Side;
use capnp_rpc::{RpcSystem, twoparty};

use sequoia_openpgp as openpgp;
use openpgp::crypto::mpi::Ciphertext;
use openpgp::packet::Key;
use openpgp::packet::PKESK;
use openpgp::packet::key;
use openpgp::parse::Parse;
use openpgp::types::HashAlgorithm;
use openpgp::types::PublicKeyAlgorithm;

use sequoia_ipc as ipc;

use crate::keystore_protocol_capnp::keystore;

use crate::Error;
use crate::Result;
use crate::error::ServerError;

mod backend;
use backend::Servers;

pub struct Keystore {
    #[allow(unused)]
    descriptor: ipc::Descriptor,
    ks: keystore::Client,
}

impl Keystore {
    /// Instantiates a new `Keystore` server.
    pub fn new(descriptor: ipc::Descriptor,
               _local: &tokio::task::LocalSet)
        -> Result<Self>
    {
        // The application may not use the rust logging framework.
        // Try to initialize it so that the user can get some output.
        let _ = env_logger::Builder::from_default_env().try_init();

        let home = descriptor.context().home();
        let home = backend::Server::home(Some(home))?.to_path_buf();
        log::debug!("KeystoreServer::new: keystore directory: {}",
                    home.display());
        Ok(Keystore {
            descriptor,
            ks: capnp_rpc::new_client(KeystoreServer::new(Arc::new(home))),
        })
    }

    /// A generic variant of [`Keystore::new`].
    ///
    /// This is a variant of [`Keystore::new`], which returns a
    /// generic type instead of a [`Keystore`].  This is useful in
    /// conjunction with [`sequoia_ipc::Descriptor::new`].
    pub fn new_descriptor(descriptor: ipc::Descriptor,
                          local: &tokio::task::LocalSet)
        -> Result<Box<dyn ipc::Handler>>
    {
        Self::new(descriptor, local).map(|d| Box::new(d) as Box<dyn ipc::Handler>)
    }
}

impl ipc::Handler for Keystore {
    fn handle(&self,
              network: twoparty::VatNetwork<tokio_util::compat::Compat<tokio::net::tcp::OwnedReadHalf>>)
        -> RpcSystem<Side>
    {
        RpcSystem::new(Box::new(network), Some(self.ks.clone().client))
    }
}

#[derive(Clone, Debug)]
struct KeystoreServer {
    home: Arc<PathBuf>,
}

impl KeystoreServer {
    fn new(home: Arc<PathBuf>) -> Self
    {
        Self {
            home,
        }
    }
}

impl keystore::Server for KeystoreServer {
    srpc!(fn backends(&mut self,
                      _params: keystore::BackendsParams,
                      mut results: keystore::BackendsResults)
        -> Result<()>
    {
        let server = Servers::get(&self.home)?;
        let server = server.lock().unwrap();

        let backends = server.iter();
        let mut ok = results.initn_ok(backends.len() as u32);
        for (i, backend) in backends.enumerate() {
            let server: BackendServer = BackendServer::new(
                Arc::clone(&self.home), backend.id());
            let cap: keystore::backend::Client = capnp_rpc::new_client(server);
            ok.reborrow().set(i as u32, cap.client.hook);
        }

        Ok(())
    });

    srpc!(fn decrypt(&mut self,
                     params: keystore::DecryptParams,
                     mut results: keystore::DecryptResults)
              -> Result<()>
    {
        let pkesks = params
            .get()?
            .get_pkesks()?
            .iter()
            .enumerate()
            .map(|(i, pkesk)| {
                Ok((i, PKESK::from_bytes(pkesk?)?))
            })
            .collect::<Result<Vec<(usize, PKESK)>>>()?;

        log::trace!("KeystoreServer::decrypt({})",
                    pkesks.iter()
                    .map(|(_i, pkesk)| {
                        pkesk.recipient().to_string()
                    })
                    .collect::<Vec<String>>()
                    .join(", "));

        let (wildcard, known): (Vec<&(usize, PKESK)>, Vec<&(usize, PKESK)>)
            = pkesks
            .iter()
            .partition(|(_i, pkesk)| pkesk.recipient().is_wildcard());

        // Keys that are inaccessible, but that could become available
        // with a little help from the user (e.g., unlock a password,
        // insert a device, etc.)
        let mut inaccessible: Vec<InaccessibleDecryptionKey> = Vec::new();

        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if ! known.is_empty() {
            for backend in server.backends.iter_mut() {
                for device in backend.list() {
                    for mut key in device.list() {
                        for (i, pkesk) in known.iter() {
                            let keyid = key.keyid();
                            if pkesk.recipient() != &keyid {
                                log::trace!("PKESK for {}, key: {}, skipping",
                                            pkesk.recipient(), keyid);
                                continue;
                            }

                            if ! key.decryption_capable() {
                                log::trace!("{} cannot be used for decryption",
                                            key.fingerprint());
                                continue;
                            }

                            let mut abort = false;
                            if key.locked() {
                                log::trace!("{} is locked", keyid);
                                abort = true;
                            }
                            if ! key.available() {
                                log::trace!("{} is unavailable", key.fingerprint());
                                abort = true;
                            }
                            if abort {
                                inaccessible.push(InaccessibleDecryptionKey {
                                    key: KeyDescriptor {
                                        server: KeyServer::new(
                                            Arc::clone(&self.home),
                                            backend.id(),
                                            device.id(), key.id()),
                                        key: key.public_key().clone(),
                                    },
                                    pkesk: pkesk.clone(),
                                });
                                continue;
                            }

                            if let Some((sym_algo, sk)) = key.decrypt_pkesk(pkesk) {
                                let mut ok = results.init_ok();

                                ok.set_index(*i as u32);
                                ok.set_fingerprint(
                                    key.fingerprint().as_bytes());
                                ok.set_algo(u8::from(sym_algo));
                                ok.set_session_key(&sk);

                                return Ok(());
                            } else {
                                log::trace!("PKESK for {} matches key {}, \
                                             but failed to decrypt it!",
                                            pkesk.recipient(), keyid);
                            }
                        }
                    }
                }
            }
        }

        if ! wildcard.is_empty() {
            unimplemented!();
        }

        if ! inaccessible.is_empty() {
            log::debug!("Failed to decrypt, but have {} candidate keys \
                         that are inaccessible",
                        inaccessible.len());
            Err(ServerError::InaccessibleDecryptionKey(inaccessible).into())
        } else {
            // XXX: The right error is: no known secret key can decrypt
            // it.
            Err(Error::EOF.into())
        }
    });
}

#[derive(Clone, Debug)]
struct BackendServer {
    home: Arc<PathBuf>,
    id: String,
}

impl BackendServer {
    fn new<S: AsRef<str>>(home: Arc<PathBuf>, id: S) -> Self {
        let server = Self {
            home,
            id: id.as_ref().into(),
        };

        server
    }
}

impl keystore::backend::Server for BackendServer {
    srpc!(fn id(&mut self,
                _params: keystore::backend::IdParams,
                mut _results: keystore::backend::IdResults)
             -> Result<&str>
    {
        Ok(&self.id)
    });

    srpc!(fn list(&mut self, _params: keystore::backend::ListParams,
                  mut results: keystore::backend::ListResults)
              -> Result<()>
    {
        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.id) {
            let devices: Vec<_> = backend.list().collect();

            let mut ok = results.initn_ok(devices.len() as u32);
            for (i, device) in devices.into_iter().enumerate() {
                let server = DeviceServer::new(
                    Arc::clone(&self.home), &self.id, device.id());
                let cap: keystore::device::Client
                    = capnp_rpc::new_client(server);

                ok.set(i as u32, cap.client.hook);
            }

            Ok(())
        } else {
            Err(Error::EOF.into())
        }
    });
}

#[derive(Clone, Debug)]
struct DeviceServer {
    home: Arc<PathBuf>,
    backend: String,
    id: String,
}

impl DeviceServer {
    fn new<B, I>(home: Arc<PathBuf>, backend: B, id: I) -> Self
        where B: AsRef<str>, I: AsRef<str>
    {
        let device = Self {
            home,
            backend: backend.as_ref().into(),
            id: id.as_ref().into(),
        };

        device
    }
}

impl keystore::device::Server for DeviceServer {
    srpc!(fn id(&mut self,
                _params: keystore::device::IdParams,
                mut _results: keystore::device::IdResults)
             -> Result<&str>
    {
        Ok(&self.id)
    });

    srpc!(fn list(&mut self,
                  _params: keystore::device::ListParams,
                  mut results: keystore::device::ListResults)
              -> Result<()>
    {
        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.backend) {
            if let Some(device) = backend.list().find(|d| d.id() == self.id) {
                let keys: Vec<_> = device.list().collect();

                let mut ok = results.initn_ok(keys.len() as u32);
                for (i, key) in keys.into_iter().enumerate() {
                    let server: KeyServer = KeyServer::new(
                        Arc::clone(&self.home), &self.backend,
                        &self.id, key.id());
                    let cap: keystore::key::Client
                        = capnp_rpc::new_client(server);

                    use openpgp::serialize::MarshalInto;
                    let bytes = key.public_key().to_vec()
                        .expect("serializing to a vec is infallible");

                    let mut r = ok.reborrow().get(i as u32);

                    r.set_handle(cap);
                    r.set_public_key(&bytes[..]);
                }

                return Ok(())
            }
        }

        Err(Error::EOF.into())
    });
}

#[derive(Clone, Debug)]
struct KeyServer {
    home: Arc<PathBuf>,
    backend: String,
    device: String,
    key: String,
}

impl KeyServer {
    fn new<B, D, K>(home: Arc<PathBuf>, backend: B, device: D, key: K)
        -> Self
        where B: AsRef<str>, D: AsRef<str>, K: AsRef<str>
    {
        let key = Self {
            home: home,
            backend: backend.as_ref().into(),
            device: device.as_ref().into(),
            key: key.as_ref().into(),
        };

        key
    }
}

impl keystore::key::Server for KeyServer {
    srpc!(fn id(&mut self,
                _params: keystore::key::IdParams,
                 mut _results: keystore::key::IdResults)
             -> Result<&str>
    {
        Ok(&self.key)
    });

    srpc!(fn unlock(&mut self,
                    params: keystore::key::UnlockParams,
                    mut _results: keystore::key::UnlockResults)
                    -> Result<()>
    {
        let password = params.get()?.get_password()?;

        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.backend) {
            if let Some(device) = backend.list().find(|d| d.id() == self.device) {
                if let Some(mut key) = device.list().find(|k| k.id() == self.key) {
                    match key.unlock(&password.into()) {
                        Ok(()) => return Ok(()),
                        Err(err) => return Err(err.into()),
                    }
                }
            }
        }

        return Err(Error::EOF.into());
    });

    srpc!(fn decrypt_ciphertext(
                  &mut self,
                  params: keystore::key::DecryptCiphertextParams,
                  mut results: keystore::key::DecryptCiphertextResults)
              -> Result<()>
    {
        let algo = PublicKeyAlgorithm::from(params.get()?.get_algo());
        let ciphertext = params.get()?.get_ciphertext()?;
        let ciphertext = Ciphertext::parse(algo, ciphertext)?;

        let plaintext_len = match params.get()?.get_plaintext_len() {
            0 => None,
            n => Some(n as usize),
        };

        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.backend) {
            if let Some(device) = backend.list().find(|d| d.id() == self.device) {
                if let Some(mut key) = device.list().find(|k| k.id() == self.key) {
                    if ! key.decryption_capable() {
                        log::trace!("{} is not for decryption",
                                    key.fingerprint());
                        return Err(Error::NotDecryptionCapable(
                            key.fingerprint().to_string()).into());
                    }

                    match key.decrypt_ciphertext(&ciphertext, plaintext_len) {
                        Ok(sk) => {
                            let ok = results.initn_ok(sk.len() as u32);
                            ok.copy_from_slice(&sk);
                            return Ok(())
                        }
                        Err(err) => {
                            return Err(err.into());
                        }
                    }
                }
            }
        }

        return Err(Error::EOF.into());
    });

    srpc!(fn sign_message(
                  &mut self,
                  params: keystore::key::SignMessageParams,
                  mut results: keystore::key::SignMessageResults)
              -> Result<()>
    {
        let hash_algo = HashAlgorithm::from(params.get()?.get_hash_algo());
        let digest = params.get()?.get_digest()?;

        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.backend) {
            if let Some(device) = backend.list().find(|d| d.id() == self.device) {
                if let Some(mut key) = device.list().find(|k| k.id() == self.key) {
                    log::trace!("Signing digest with {} (hash: {})",
                                key.keyid(), hash_algo);

                    if ! key.signing_capable() {
                        log::trace!("{} is not for signing",
                                    key.fingerprint());
                        return Err(Error::NotSigningCapable(
                            key.fingerprint().to_string()).into());
                    }
                    match key.sign(hash_algo, digest) {
                        Ok((pk_algo, sig)) => {
                            let mut ok = results.init_ok();

                            ok.set_pk_algo(u8::from(pk_algo));

                            use openpgp::serialize::MarshalInto;
                            let bytes = sig.to_vec()
                                .expect("serializing to a vec is infallible");
                            let mpis = ok.init_mpis(bytes.len() as u32);
                            mpis.copy_from_slice(&bytes);

                            return Ok(())
                        }
                        Err(err) => {
                            log::info!("Failed to sign {} digest with {}: {}",
                                       hash_algo, key.keyid(), err);
                            return Err(err.into());
                        }
                    }
                }
            }
        }

        log::debug!("Invalid key capability: {}/{}/{} not found",
                    self.backend, self.device, self.key);

        return Err(Error::EOF.into());
    });

    srpc!(fn available(&mut self,
                    _params: keystore::key::AvailableParams,
                    mut _results: keystore::key::AvailableResults)
             -> Result<bool>
    {
        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.backend) {
            if let Some(device) = backend.list().find(|d| d.id() == self.device) {
                if let Some(key) = device.list().find(|k| k.id() == self.key) {
                    return Ok(key.available());
                }
            }
        }

        return Err(Error::EOF.into());
    });

    srpc!(fn locked(&mut self,
                    _params: keystore::key::LockedParams,
                    mut _results: keystore::key::LockedResults)
             -> Result<bool>
    {
        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.backend) {
            if let Some(device) = backend.list().find(|d| d.id() == self.device) {
                if let Some(key) = device.list().find(|k| k.id() == self.key) {
                    return Ok(key.locked());
                }
            }
        }

        return Err(Error::EOF.into());
    });

    srpc!(fn decryption_capable(&mut self,
                                _params: keystore::key::DecryptionCapableParams,
                                mut _results: keystore::key::DecryptionCapableResults)
             -> Result<bool>
    {
        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.backend) {
            if let Some(device) = backend.list().find(|d| d.id() == self.device) {
                if let Some(key) = device.list().find(|k| k.id() == self.key) {
                    return Ok(key.decryption_capable());
                }
            }
        }

        return Err(Error::EOF.into());
    });

    srpc!(fn signing_capable(&mut self,
                                _params: keystore::key::SigningCapableParams,
                                mut _results: keystore::key::SigningCapableResults)
             -> Result<bool>
    {
        let server = Servers::get(&self.home)?;
        let mut server = server.lock().unwrap();

        if let Some(backend) = server.backends.iter_mut().find(|b| b.id() == self.backend) {
            if let Some(device) = backend.list().find(|d| d.id() == self.device) {
                if let Some(key) = device.list().find(|k| k.id() == self.key) {
                    return Ok(key.signing_capable());
                }
            }
        }

        return Err(Error::EOF.into());
    });
}

// Basically, a wrapper around a KeyHandle that we want to return
// later.  This isn't just a wrapper around a `dyn
// sequoia_keystore_backend::KeyHandle` due to lifetimes.
#[derive(Clone)]
pub(crate) struct KeyDescriptor {
    server: KeyServer,
    key: Key<key::PublicParts, key::UnspecifiedRole>,
}

impl std::fmt::Debug for KeyDescriptor {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "server::KeyDescriptor {{ {} }}",
               self.key.fingerprint())
    }
}

impl KeyDescriptor {
    /// Returns a capability.
    pub fn cap(&self) -> keystore::key::Client {
        capnp_rpc::new_client(self.server.clone())
    }

    /// Returns the key handle.
    pub fn public_key(&self) -> &Key<key::PublicParts, key::UnspecifiedRole> {
        &self.key
    }

    pub fn serialize(&self, mut builder: keystore::key_descriptor::Builder) {
        use openpgp::serialize::MarshalInto;

        builder.set_handle(self.cap());

        let bytes = self.public_key().to_vec()
            .expect("serializing to a vec is infallible");
        builder.set_public_key(&bytes[..]);
    }
}

#[derive(Clone)]
pub(crate) struct InaccessibleDecryptionKey {
    key: KeyDescriptor,
    pkesk: PKESK,
}

impl std::fmt::Debug for InaccessibleDecryptionKey {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        write!(f, "server::InaccessibleDecryptionKey {{ {} }}",
               self.key.key.fingerprint())
    }
}

impl InaccessibleDecryptionKey {
    pub fn serialize(&self, mut builder: keystore::inaccessible_decryption_key::Builder) {
        use openpgp::serialize::MarshalInto;

        self.key.serialize(builder.reborrow().init_key_descriptor());

        let bytes = self.pkesk.to_vec()
            .expect("serializing to a vec is infallible");
        builder.set_pkesk(&bytes[..]);
    }
}
