-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Ada.Text_IO;
with Ada.Integer_Text_IO;
with E_Strings.Not_SPARK;

package body SLI.IO is
   --# hide SLI.IO;

   SLI_File_Descriptor : Ada.Text_IO.File_Type;

   procedure Create_File (Name_Of_File : in     E_Strings.T;
                          Status       :    out File_Status) is
   begin
      Ada.Text_IO.Create
        (File => SLI_File_Descriptor,
         Mode => Ada.Text_IO.Out_File,
         Name => E_Strings.Not_SPARK.Get_String (E_Str => Name_Of_File),
         Form => "");
      Status := Ok;
   exception
      when Ada.Text_IO.Status_Error =>
         Status := Status_Error;
      when Ada.Text_IO.Name_Error =>
         Status := Name_Error;
      when Ada.Text_IO.Use_Error =>
         Status := Use_Error;
      when Ada.Text_IO.Device_Error =>
         Status := Device_Error;
   end Create_File;

   procedure Close is
   begin
      if Ada.Text_IO.Is_Open (File => SLI_File_Descriptor) then
         Ada.Text_IO.Close (File => SLI_File_Descriptor);
      end if;
   end Close;

   procedure Put_Char (Item : in Character) is
   begin
      if Ada.Text_IO.Is_Open (File => SLI_File_Descriptor) then
         Ada.Text_IO.Put (File => SLI_File_Descriptor,
                          Item => Item);
      end if;
   end Put_Char;

   procedure Put_String (Item : in String) is
   begin
      if Ada.Text_IO.Is_Open (File => SLI_File_Descriptor) then
         Ada.Text_IO.Put (File => SLI_File_Descriptor,
                          Item => Item);
      end if;
   end Put_String;

   procedure Put_Integer (Item : in Integer) is
   begin
      if Ada.Text_IO.Is_Open (File => SLI_File_Descriptor) then
         Ada.Integer_Text_IO.Put (File  => SLI_File_Descriptor,
                                  Item  => Item,
                                  Width => 0);
      end if;
   end Put_Integer;

   procedure New_Line is
   begin
      if Ada.Text_IO.Is_Open (File => SLI_File_Descriptor) then
         Ada.Text_IO.New_Line (File => SLI_File_Descriptor);
      end if;
   end New_Line;

   procedure Put_Line (Item : in String) is
   begin
      Put_String (Item => Item);
      New_Line;
   end Put_Line;

   procedure E_Strings_Put_String (E_Str : in E_Strings.T) is
   begin
      if E_Strings.Get_Length (E_Str => E_Str) /= 0 and then Ada.Text_IO.Is_Open (File => SLI_File_Descriptor) then
         Ada.Text_IO.Put (File => SLI_File_Descriptor,
                          Item => E_Strings.Not_SPARK.Get_String (E_Str => E_Str));
      end if;
   end E_Strings_Put_String;

   procedure E_Strings_Put_Line (E_Str : in E_Strings.T) is
   begin
      E_Strings_Put_String (E_Str => E_Str);
      New_Line;
   end E_Strings_Put_Line;

end SLI.IO;
