/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#ifndef _h_klib_printf_
#define _h_klib_printf_

#ifndef _h_klib_extern_
#include <klib/extern.h>
#endif

#ifndef _h_klib_defs_
#include <klib/defs.h>
#endif

#include <stdarg.h>

#ifdef __cplusplus
extern "C" {
#endif

/*--------------------------------------------------------------------------
 * FORMAT
 *  a description of the string_printf formatting
 */

/*
  The standard C library formatting approach was taken as a basis.

  This interface differs in some ways, in that it presents a sub-set of
  the std. C format convention, and then extends it for klib.

  The general substitution parameter format is:

    '%' [ <flags> ] [ <field-width> ] [ '.' <precision> ] [ ':' <index> ] \
        [ <storage-class-width> ] <storage-class>

  where:

    flags
        = ' '           : prepend space to a numeral if it does not have a sign
        | '-'           : left-align parameter within field
        | '+'           : always produce a sign on numeric conversion
        | '0'           : left-pad with zeroes rather than spaces
        | '#'           : use "alternate" representation
        | ','           : produce comma-separated triples
        | '\''          :  "      "
        ;

    field-width
        = DECIMAL       : a base-10 numeral
        | '*'           : take field width from args as type 'uint32_t'
        ;

    precision
        = DECIMAL       : a base-10 numeral
        | '*'           : take precision from args as type 'uint32_t'
        ;

    index
        = idx           : a single, zero-based vector element
        | idx '-' idx   : a fully-closed, zero-based interval
        ;

    idx
        = DECIMAL       : a base-10 numeral
        | '$'           : the last element in vector
        | '*'           : take index from args as type 'uint32_t'
        ;

    storage-class-width
        = 'h'           : half the normal size
        | 'l'           : twice the normal size
        | 'z'           : sizeof size_t
        | time-modifier
        ;

    time-modifier
        = 'h'           : date only
        | 'l'           : date and time
        | 'z'           : date, time and zone
        ;

    storage-class
        = 'd' | 'i'     : decimal int32_t
        | 'u'           : decimal uint32_t
        | 'x'           : lower-case hex uint32_t
        | 'X'           : upper-case hex uint32_t
        | 'o'           : octal uint32_t
        | 'b'           : binary uint32_t
        | 'p'           : hex void*
        | 'f'           : double
        | 'e'           : scientific notation double
        | 'g'           : general double
        | 'c'           : UTF-32 character
        | 's'           : UTF-8 string
        | 'S'           : const String* [ see above ]
        | 'N'           : const KSymbol* [ <klib/symbol.h> ]
        | 'V'           : tri-part version [ ver_t ]
        | 'R'           : return code [ rc_t ]
        | 'T'           : const KTime*  [ <klib/time.h> ]
        | '!'           ; operating specific error code ( i.e. errno or GetLastError() )
        ;

  Notes:
     1. field-width and precision measure characters, not bytes
     2. for version numbers, precision gives the number of fields,
        where 1 = major, 2 = major.minor and 3 = major.minor.release.
     3. in the absence of precision, versions are written with the
        minimum number of fields required.
     4. the storage-class-width is interpreted differently for storage-class
        'T' ( const KTime* )

 */

/* string_printf
 *  provides a facility similar to snprintf
 *  formatting is similar but differs somewhat [ see FORMAT at bottom ]
 *
 *  "dst" [ OUT ] and "bsize" [ IN ] - output buffer for string
 *  will be NUL-terminated if possible
 *
 *  "num_writ" [ OUT, NULL OKAY ] - returns the number of non-NUL bytes
 *  written to "dst" or the required "bsize" to complete successfully,
 *  not including the NUL termination.
 *
 *  "fmt" [ IN ] and "args" [ IN, OPTIONAL ] - data to write
 *
 *  returns 0 if all bytes were successfully written and a NUL-byte was
 *  written into the buffer.
 *
 *  returns rcBuffer, rcInsufficient if the buffer was too small. in this
 *  case, it is possible that the only missing byte would be the NUL
 *  termination, and the output string may still be usable since "num_writ"
 *  indicates the actual number of text bytes.
 */
KLIB_EXTERN rc_t CC string_printf ( char *dst, size_t bsize,
    size_t *num_writ, const char *fmt, ... );
KLIB_EXTERN rc_t CC string_vprintf ( char *dst, size_t bsize,
    size_t *num_writ, const char *fmt, va_list args );


#ifdef __cplusplus
}
#endif

#endif /* _h_klib_printf_ */
