use lang:bs;
use core:lang;
use core:asm;
use core:debug;

// Get the name of the SrcName type.
SrcName srcName(SrcPos pos) on Compiler {
	SrcName name(pos);
	name.add("core");
	name.add("lang");
	name.add("SrcName");
	name;
}

// Get the name of the TypePart type.
SrcName namePart(SrcPos pos) on Compiler {
	SrcName name(pos);
	name.add("core");
	name.add("lang");
	name.add("RecPart");
	name;
}

// Generate a name part from a string.
RecPart strRecPart(Str name) {
	RecPart(name.unescape('"'));
}

// Create something that gives you a TypePart.
Expr createTypePart(SrcPos pos, Block block, NamePart part) on Compiler {
	ExprBlock b(pos, block);
	SrcName partT = namePart(pos);

	Expr str = strConstant(pos, part.name);
	Var base(b, partT, SStr("part", pos), Actuals(str));
	b.add(base);
	LocalVarAccess bAccess(pos, base.var);

	if (part as SimplePart) {
		for (Nat i = 0; i < part.params.count; i++) {
			Expr created = if (type = part.params[i].type) {
				createType(pos, b, Name(type.path));
			} else {
				createType(pos, b, Name("void"));
			};
			b.add(namedExpr(b, SStr("add", pos), bAccess, Actuals(created)));
		}
	} else if (part as RecPart) {
		for (Nat i = 0; i < part.params.count; i++) {
			Expr created = createType(pos, b, part.params[i]);
			b.add(namedExpr(b, SStr("add", pos), bAccess, Actuals(created)));
		}
	}

	b.add(bAccess);
	b;
}

// Create something that gives you a TypeName.
Expr createType(SrcPos pos, Block block, Name type) on Compiler {
	ExprBlock b(pos, block);
	SrcName nameT = srcName(pos);
	SrcName partT = namePart(pos);

	Var base(b, nameT, SStr("name", pos), Actuals());
	b.add(base);

	LocalVarAccess bAccess(pos, base.var);
	for (Nat i = 0; i < type.count; i++) {
		Expr created = createTypePart(pos, b, type[i]);
		b.add(namedExpr(b, SStr("add", pos), bAccess, Actuals(created)));
	}

	b.add(bAccess);
	b;
}

// Create a static reference to a Named object in the compiler.
Expr referNamed(SrcPos pos, Block block, SrcName type) on Compiler {
	Named? referTo = block.scope.find(type);
	if (referTo) {
		ReferNamed(pos, referTo);
	} else {
		throw SyntaxError(type.pos, "Could not resolve the name " + type.toS);
	}
}

Expr referNamedAbs(SrcPos pos, Block block, SrcName type) on Compiler {
	Named? referTo = rootScope().find(type);
	if (referTo) {
		ReferNamed(pos, referTo);
	} else {
		throw SyntaxError(type.pos, "Could not resolve the absolute name " + type.toS);
	}
}

// Special case for the current package.
Expr referNamed(SrcPos pos, Block block) on Compiler {
	ReferNamed(pos, ScopeLookup:firstPkg(block.lookup));
}

// Refer to a named object somewhere.
class ReferNamed extends Expr {

	// Object we're referring to.
	Named referTo;

	init(SrcPos pos, Named to) {
		init(pos) {
			referTo = to;
		}
	}

	ExprResult result() {
		ExprResult(Value(typeOf(referTo)));
	}

	void code(CodeGen state, CodeResult result) {
		if (!result.needed) {
			return;
		}

		// This solution is also bad when we want to serialize the code and load it later.
		// We should use a reference to the named item instead.
		var v = result.location(state);
		state.l << mov(v, objPtr(referTo));
		result.created(state);
	}

	Str toS() {
		"named{" # referTo.identifier # "}";
	}
}

Expr reloadName(Block block, SrcName typeName) on Compiler {
	Expr type = createType(typeName.pos, block, typeName);
	Expr callIt = namedExpr(block, name{lang:bs:macro:reload}, Actuals(type));
	callIt;
}


Expr printExpr(Str text, Expr expr) on Compiler {
	if (expr as Operator) {
		print(text.removeIndentation # " <=> " # expr.meaning);
	} else {
	    print(text.removeIndentation # " <=> " # expr);
	}
	expr;
}

void summary(NameSet s) on Compiler {
	Type[] types;
	Function[] fns;

	for (i in s) {
		if (i as Type) {
			types << i;
		} else if (i as Function) {
			fns << i;
		}
	}

	print("Types:");
	for (i in types) {
		print(" " # i.path);
	}

	print("");
	print("Functions:");
	for (i in fns) {
		print(" " # i.path);
	}
}

class PrintAsm extends Expr {
	init(SrcPos pos) {
		init(pos);
	}

	ExprResult result() : override {
		Value();
	}

	void code(CodeGen state, CodeResult result) : override {
		print(state.toS);
	}

	Str toS() : override {
		"dumpAsm";
	}
}
