/*
    sun.c - Sunrise and sunset
    Copyright (C) 2010 Ladislav Vaiz <ok1zia@nagano.cz>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

	Based on http://williams.best.vwh.net/sunrise_sunset_algorithm.htm

Source:
	Almanac for Computers, 1990
	published by Nautical Almanac Office
	United States Naval Observatory
	Washington, DC 20392

Inputs:
	day, month, year:      date of sunrise/sunset
	latitude, longitude:   location for sunrise/sunset
	zenith:                Sun's zenith for sunrise/sunset
	  offical      = 90 degrees 50'
	  civil        = 96 degrees
	  nautical     = 102 degrees
	  astronomical = 108 degrees
	
	NOTE: longitude is positive for East and negative for West
    NOTE: the algorithm assumes the use of a calculator with the
          trig functions in "degree" (rather than "radian") mode. Most
          programming languages assume radian arguments, requiring back
          and forth convertions. The factor is 180/pi. So, for instance,
          the equation RA = atan(0.91764 * tan(L)) would be coded as RA
          = (180/pi)*atan(0.91764 * tan((pi/180)*L)) to give a degree
          answer with a degree input for L.


*/

#include "header.h"

double dsin(double x){
	return sin(x * MY_PI/180.0);
}

double dcos(double x){
	return cos(x * MY_PI/180.0);
}

double dtan(double x){
	return tan(x * MY_PI/180.0);
}

double dasin(double x){
	return asin(x) * 180.0/MY_PI;
}

double dacos(double x){
	return acos(x) * 180.0/MY_PI;
}

double datan(double x){
	return atan(x) * 180.0/MY_PI;
}

static void norm360(double *x){
	while (*x < 0.0) *x += 360.0;
	while (*x >= 360.0) *x -= 360.0;
}

static void norm24(double *x){
	while (*x < 0.0) *x += 24.0;
	while (*x >= 24.0) *x -= 24.0;
}

double sunriseset(int sunrise, double latitude, double longitude){
    struct tm tm;
	time_t now;
	double month, year, day, N1, N2, N3, N, lngHour, t, M, L, RA, sinDec, cosDec, zenith, cosH, H, T, UT;

//	1. first calculate the day of the year

	now = time(NULL);
    gmtime_r(&now, &tm);
	month = tm.tm_mon + 1;
	year = tm.tm_year + 1900;
	day = tm.tm_mday;

	N1 = floor(275 * month / 9);
	N2 = floor((month + 9) / 12);
	N3 = (1 + floor((year - 4 * floor(year / 4) + 2) / 3));
	N = N1 - (N2 * N3) + day - 30;

//	2. convert the longitude to hour value and calculate an approximate time

	lngHour = longitude / 15;
				
	if (sunrise)
		t = N + ((6 - lngHour) / 24);
	else
		t = N + ((18 - lngHour) / 24);

//	3. calculate the Sun's mean anomaly
		
	M = (0.9856 * t) - 3.289;

//	4. calculate the Sun's true longitude
			
	L = M + (1.916 * dsin(M)) + (0.020 * dsin(2 * M)) + 282.634;
	norm360(&L);
//	NOTE: L potentially needs to be adjusted into the range [0,360) by adding/subtracting 360

//	5a. calculate the Sun's right ascension
					
	RA = datan(0.91764 * dtan(L));
//	NOTE: RA potentially needs to be adjusted into the range [0,360) by adding/subtracting 360
	norm360(&RA);

//	5b. right ascension value needs to be in the same quadrant as L

//	double Lquadrant  = (floor( L/90)) * 90;
//	double RAquadrant = (floor(RA/90)) * 90;
//	double A = RA + (Lquadrant - RAquadrant);

//	5c. right ascension value needs to be converted into hours
	RA = RA / 15;

//	6. calculate the Sun's declination
	sinDec = 0.39782 * dsin(L);
	cosDec = dcos(dasin(sinDec));

//	7a. calculate the Sun's local hour angle
	
	zenith = 90.0 + 50/60;
	cosH = (dcos(zenith) - (sinDec * dsin(latitude))) / (cosDec * dcos(latitude));
		
	if (cosH > 1) return -1;  // the sun never rises on this location (on the specified date)
    if (cosH < -1) return -2; // the sun never sets on this location (on the specified date)

//	7b. finish calculating H and convert into hours
	
	if (sunrise)
		H = 360 - dacos(cosH);
	else
		H = dacos(cosH);
		
	H = H / 15;

//	8. calculate local mean time of rising/setting
 	  	
	T = H + RA - (0.06571 * t) - 6.622;

//	9. adjust back to UTC
				
	UT = T - lngHour;
//	NOTE: UT potentially needs to be adjusted into the range [0,24) by adding/subtracting 24
	norm24(&UT);

//	10. convert UT value to local time zone of latitude/longitude
						
   // double localT = UT + localOffset;
	return UT;
}

char *sunrisesetstr(char *str, double latitude, double longitude){
	double rise = sunriseset(1, latitude, longitude);
	double set  = sunriseset(0, latitude, longitude);

	if (rise == -1 || set == -1){
		strcpy(str, "polar night");
		return str;
	}
	
	if (rise == -2 || set == -2){
		strcpy(str, "polar day");
		return str;
	}
	sprintf(str, "%02d:%02d-%02d:%02d", (int)rise, ((int)(rise * 60))% 60, (int)set, ((int)(set * 60))%60);
	return str;
}

