/*
 * unity-npapi-plugin.c
 * Copyright (C) Canonical LTD 2012
 * 
 * Author: Alexandre Abreu <alexandre.abreu@canonical.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <errno.h>
#include <glib.h>

#include "unity-npapi-plugin.h"
#include "unity-npapi-plugin-npn-funcs.h"


static bool fill_plugin_functions (NPPluginFuncs * pFuncs);
static bool fill_browser_functions (NPNetscapeFuncs * pFuncs);


int unity_npapi_initialize_plugin (NPNetscapeFuncs * pBrowserFuncs, NPPluginFuncs * pPluginFuncs)
{
  g_return_val_if_fail (NULL != pBrowserFuncs, EINVAL);
  g_return_val_if_fail (NULL != pPluginFuncs, EINVAL);
  
  if (false == fill_browser_functions (pBrowserFuncs))
    {
      return ENOENT;
    }
  
  if (false == fill_plugin_functions (pPluginFuncs))
    {
      return ENOENT;
    }
  
  return 0;
}

int unity_npapi_cleanup_plugin (void)
{
  return 0;
}


void unity_npapi_add_retained_objects_for_context (unity_npapi_plugin_t * plugin
                                                   , gpointer context
                                                   , NPObject * retained_object)
{
  g_return_if_fail(NULL != plugin);

  if (NULL == plugin->retained_objects)
    {
      return;
    }

  if (! g_hash_table_contains(plugin->retained_objects, context))
    {
      g_hash_table_insert(plugin->retained_objects, context, g_list_alloc ());
    }

  GList * objects = g_hash_table_lookup(plugin->retained_objects, context);
  g_return_if_fail(NULL != objects);

  NPN_RetainObject(retained_object);

  g_hash_table_replace(plugin->retained_objects
                       , context
                       , g_list_append(objects, retained_object));
}

static void release_npobjects (GList * objects)
{
  if (NULL == objects)
    {
      return;
    }

  GList * item = g_list_first(objects);
  while (NULL != item)
    {
      if (NULL != item->data)
        {
          NPN_ReleaseObject((NPObject*) item->data);
        }

      item = g_list_next(item);
    }
}

void unity_npapi_free_retained_objects_for_context (unity_npapi_plugin_t * plugin
                                                    , gpointer context)
{
  g_return_if_fail(NULL != plugin);

  if (NULL == plugin->retained_objects)
    {
      return;
    }
  g_return_if_fail(g_hash_table_contains(plugin->retained_objects, context));

  GList * objects =
    g_hash_table_lookup(plugin->retained_objects, context);

  if (NULL != objects)
    {
      release_npobjects (objects);
      g_list_free(objects);
    }

  g_hash_table_remove(plugin->retained_objects, context);
}

void unity_npapi_free_all_retained_objects (unity_npapi_plugin_t * plugin)
{
  g_return_if_fail(NULL != plugin);

  if (NULL == plugin->retained_objects)
    {
      return;
    }

  GHashTableIter iter;
  g_hash_table_iter_init (&iter, plugin->retained_objects);

  gpointer context = NULL;
  gpointer objects = NULL;
  while (g_hash_table_iter_next (&iter, &context, &objects))
  {
    release_npobjects (objects);
    g_list_free (objects);
  }

  g_hash_table_unref (plugin->retained_objects);
  plugin->retained_objects = NULL;
}


/////////////////////////////////////////////////////
//
//	private
//
/////////////////////////////////////////////////////

static bool fill_browser_functions (NPNetscapeFuncs * pFuncs)
{
  g_return_val_if_fail (NULL != pFuncs, false);

  pBrowser_NPN_Functions = pFuncs;

  return true;
}

static bool fill_plugin_functions (NPPluginFuncs * pFuncs)
{
  g_return_val_if_fail (NULL != pFuncs, false);

  pFuncs->version = (NP_VERSION_MAJOR << 8) | NP_VERSION_MINOR;
  pFuncs->size = sizeof(NPPluginFuncs);
  pFuncs->newp = NPP_New;
  pFuncs->destroy = NPP_Destroy;
  pFuncs->setwindow = NPP_SetWindow;
  pFuncs->newstream = NPP_NewStream;
  pFuncs->destroystream = NPP_DestroyStream;
  pFuncs->asfile = NPP_StreamAsFile;
  pFuncs->writeready = NPP_WriteReady;
  pFuncs->write = NPP_Write;
  pFuncs->print = NPP_Print;
  pFuncs->event = NPP_HandleEvent;
  pFuncs->urlnotify = NPP_URLNotify;
  pFuncs->javaClass = NULL; // not used
  pFuncs->getvalue = NPP_GetValue;
  pFuncs->setvalue = NPP_SetValue;

  return true;
}


