#!/usr/bin/python

## Export the current translations to chromium's extension
## Expects to be called w/ .po folder and chromium locale folder, e.g.:

import sys
import argparse

def exit_with_error(msg):
    sys.stderr.write("*** Error: " + msg + "\n")
    sys.exit(1)

parser = argparse.ArgumentParser()
parser.add_argument("po-folder", help="location of the .po files")
parser.add_argument("chrome-locals-folder", help="location of the chromium locale folder")

args = vars (parser.parse_args(sys.argv[1:]))

po_folder = args['po-folder']
chrome_locale_folder = args['chrome-locals-folder']

# validation
import os
if not os.path.exists(po_folder) or not os.path.exists(chrome_locale_folder):
    exit_with_error("Invalid paths")

import os
thispath = os.path.dirname(os.path.realpath(__file__))

def complete_path(p):
    if not os.path.isabs(p):
        p = os.path.join (thispath, p)
    return p

po_folder = complete_path(po_folder)
chrome_locale_folder = complete_path(chrome_locale_folder)

messages_json_reference_filename = os.path.join(chrome_locale_folder, "en/messages.json")
if not os.path.exists(messages_json_reference_filename):
    exit_with_error("Invalid chromium locale folder structure: no messages.json found for 'en' reference")

def collect_po_files(po_folder):
    po_folder = complete_path(po_folder)
    return [entry for entry in os.listdir(po_folder) if os.path.isfile(os.path.join(po_folder, entry)) and entry.endswith('.po')]

def save_message_json_file():
    pass

def parse_chromium_reference_locale(chromium_locale_folder):
    import json
    messages = {}
    try:
        content = json.loads(open(messages_json_reference_filename).read())
        for k, v in content.iteritems():
            if v.__class__ is type({}) and v.has_key('message'):
                # assume that messages are identical
                if messages.has_key(v['message']):
                    print "Something is wrong, duplicated message strings found '", v['message'], "'"
                else:
                    messages[v['message']] = (k, v)
    except Exception, e:
        exit_with_error("Cannot read & import messages.json file, " + str(e))
    return messages

def parse_po_files(po_folder):
    import polib
    po_files = collect_po_files(po_folder)
    if len(po_files) == 0:
        return {}
    r = []
    for po_file in po_files:
        cur = {}
        filepath = os.path.join(po_folder, po_file)
        lang = os.path.splitext(os.path.basename(po_file))[0]
        cur['lang'] = lang
        po = polib.pofile(filepath)
        for entry in po:
            if entry.obsolete or entry.msgstr == '':
                continue
            cur.setdefault('translations', {})[entry.msgid] = entry.msgstr
        r.append(cur)
    return r

def generate_chromium_json_files(po_folder, chromium_locale_folder):
    po_data = parse_po_files(po_folder)
    if len(po_data) == 0:
        print "No .po files found, or data not parsable"
        return
    en_reference_data = parse_chromium_reference_locale(chromium_locale_folder)

    import copy
    for po_datum in po_data:
        lang = po_datum['lang']
        chromium_cur_lang_folder = os.path.join(chromium_locale_folder, lang)
        if not os.path.exists(chromium_cur_lang_folder):
            os.mkdir(chromium_cur_lang_folder)
        message_filename = os.path.join(chromium_cur_lang_folder, "messages.json")
        lang_translations = po_datum['translations']

        messages = {}
        for k, v in en_reference_data.iteritems():
            message = copy.deepcopy(v[1])
            message['message'] = lang_translations[k]
            messages[v[0]] = message
        import json
        import codecs
        out = codecs.open(message_filename, "w+", "utf-8")
        out.write(json.dumps(messages))

if __name__ == "__main__":
    generate_chromium_json_files(po_folder, chrome_locale_folder)
