/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#ifndef VERSION
#define VERSION "UNKNOWN VERSION"
#endif

#include <librevenge-stream/librevenge-stream.h>
#include <libetonyek/libetonyek.h>
#include <librvngabw/librvngabw.hxx>

#include "writerperfect_utils.hxx"
#include "ABWStringDocumentHandler.hxx"

using libetonyek::EtonyekDocument;

class ABWOutputFileHelper
{
public:
	ABWOutputFileHelper(const char *outFileName) : mOutFileName(outFileName) {}
	~ABWOutputFileHelper() {}

	bool convertTextDocument(librevenge::RVNGInputStream &input)
	{
		writerperfectrvngabw::ABWStringDocumentHandler contentHandler;
		librvngabw::ABWTextGenerator collector(&contentHandler);
		if (!EtonyekDocument::parse(&input, &collector))
			return false;
		if (mOutFileName)
		{
			FILE *fhandle = fopen(mOutFileName, "wb");
			if (!fhandle)
				return false;
			fprintf(fhandle, "%s\n", contentHandler.getData().cstr());
			fclose(fhandle);
		}
		else
			printf("%s\n", contentHandler.getData().cstr());
		return true;
	}

	static EtonyekDocument::Confidence isSupportedFormat(librevenge::RVNGInputStream &input, EtonyekDocument::Type &type)
	{
		return EtonyekDocument::isSupported(&input, &type);
	}
private:
	//! the ouput file name
	char const *mOutFileName;
};

int printUsage(char const *name)
{
	fprintf(stderr, "Usage: %s [OPTION] <Pages> [file.abw]\n", name);
	fprintf(stderr, "\ttries to convert Pages document in a AbiWord file.\n");
	fprintf(stderr, "\tIf given the result is stored in file.abw, \n");
	fprintf(stderr, "\tif not, the result is sent to stdout.\n");
	fprintf(stderr, "\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "\t-h:                Shows this help message\n");
	fprintf(stderr, "\t-v:                Output %s version \n", name);
	fprintf(stderr, "\n");

	return 1;
}

int printVersion(char const *name)
{
	printf("%s %s\n", name, VERSION);
	return 0;
}

int main(int argc, char *argv[])
{
	bool printHelp=false;
	int ch;

	while ((ch = getopt(argc, argv, "hvx")) != -1)
	{
		switch (ch)
		{
		case 'v':
			printVersion(argv[0]);
			return 0;
		case 'h':
		default:
			printHelp=true;
			break;
		}
	}
	if ((argc != 1+optind && argc != 2+optind) || printHelp || !argv[optind])
	{
		printUsage(argv[0]);
		return -1;
	}

	char const *szInputFile = argv[optind];
	char const *szOutFile = (argc == 2+optind) ? argv[optind+1] : 0;

	ABWOutputFileHelper helper(szOutFile);

	shared_ptr<librevenge::RVNGInputStream> input;
	if (librevenge::RVNGDirectoryStream::isDirectory(szInputFile))
		input.reset(new librevenge::RVNGDirectoryStream(szInputFile));
	else
		input.reset(new librevenge::RVNGFileStream(szInputFile));

	EtonyekDocument::Type type = EtonyekDocument::TYPE_UNKNOWN;
	const EtonyekDocument::Confidence confidence = helper.isSupportedFormat(*input, type);
	if ((EtonyekDocument::CONFIDENCE_NONE == confidence) || (EtonyekDocument::TYPE_PAGES != type))
	{
		fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Page Document.\n");
		return 1;
	}

	if (EtonyekDocument::CONFIDENCE_SUPPORTED_PART == confidence)
	{
		input.reset(librevenge::RVNGDirectoryStream::createForParent(szInputFile));

		if (EtonyekDocument::CONFIDENCE_EXCELLENT != helper.isSupportedFormat(*input, type) || (EtonyekDocument::TYPE_PAGES != type))
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid Page Document.\n");
			return 1;
		}
	}

	if (!helper.convertTextDocument(*input))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
