/*
 * Copyright 2015-2018 Yubico AB
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifdef NDEBUG
#undef NDEBUG
#endif
#include <assert.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "openssl-compat.h"

#include <yubihsm.h>

const char ssh_ca_pvtkey_file[] = "ssh_ca_pvtkey.pem";
const char *key_label = "label";
const uint8_t password[] = "password";
const unsigned char template_dat[] = {
  // Timestamp key algorithm RSA2048
  0x01, 0x00, 0x01, 0x09,
  // Timestamp public key
  0x02, 0x01, 0x00, 0xc2, 0x55, 0x62, 0x08, 0xf5, 0xd2, 0xc2, 0x81, 0xb8, 0xa5,
  0x16, 0xfd, 0x27, 0x25, 0xe6, 0x7e, 0x88, 0xcd, 0xc5, 0xd2, 0xcf, 0xdf, 0xd3,
  0xea, 0x2d, 0x35, 0xdf, 0x35, 0x27, 0x93, 0x44, 0x45, 0xa6, 0x14, 0x84, 0xee,
  0xcb, 0x02, 0xc4, 0x7b, 0x67, 0xc5, 0x94, 0x16, 0xde, 0xe4, 0xa6, 0x1f, 0x25,
  0x52, 0x4b, 0x27, 0x9d, 0x4d, 0x09, 0xb1, 0x9b, 0x3e, 0xc5, 0x89, 0xde, 0xe2,
  0x90, 0xda, 0xa0, 0x64, 0xc7, 0xb3, 0xaa, 0xae, 0xc7, 0x23, 0x55, 0x37, 0xa0,
  0xea, 0x63, 0xb4, 0x1b, 0x65, 0x4a, 0x7f, 0x71, 0xc6, 0x5c, 0xc2, 0x34, 0xfe,
  0xa6, 0x02, 0xc9, 0xd6, 0x65, 0x13, 0x5d, 0xca, 0x74, 0x32, 0xf8, 0x7c, 0x01,
  0x4b, 0x67, 0x61, 0xdf, 0x27, 0xdd, 0x1d, 0xed, 0x2f, 0x71, 0xcb, 0x8b, 0x23,
  0x74, 0x4c, 0xfc, 0x99, 0xe2, 0x23, 0xed, 0xa5, 0xd8, 0x41, 0xe2, 0x9f, 0x82,
  0x19, 0xbd, 0xae, 0x50, 0xfb, 0xb9, 0xc7, 0xe6, 0x83, 0x01, 0xac, 0x1c, 0x63,
  0x89, 0xb2, 0xac, 0xa7, 0xfd, 0x01, 0x2a, 0xa3, 0xd4, 0x0d, 0x88, 0xf4, 0xcf,
  0x9f, 0xed, 0xc1, 0x19, 0xc8, 0x64, 0x71, 0xd3, 0x02, 0x6b, 0x9f, 0x0d, 0xc2,
  0xdf, 0x81, 0x5d, 0x53, 0x82, 0x3e, 0xa0, 0xab, 0xf2, 0x93, 0xc9, 0xa4, 0xa8,
  0x3b, 0x71, 0xc1, 0xf4, 0xe3, 0x31, 0xa5, 0xdc, 0xfe, 0xc7, 0x9e, 0x7f, 0xd8,
  0x2d, 0xd9, 0xfc, 0x90, 0xde, 0xa8, 0xdb, 0x77, 0x0b, 0x2f, 0xb0, 0xf4, 0x49,
  0x21, 0x95, 0x95, 0x4b, 0x7e, 0xa0, 0x6f, 0x15, 0x8f, 0x95, 0xdd, 0x72, 0x39,
  0x7a, 0x13, 0xb6, 0xcc, 0xfa, 0x9a, 0x07, 0x2d, 0x41, 0xcf, 0x12, 0xaf, 0x8e,
  0x87, 0x9f, 0x97, 0xf1, 0x1e, 0x00, 0xac, 0xce, 0x2d, 0x12, 0xd4, 0x34, 0x0c,
  0x40, 0x84, 0x33, 0x3a, 0x6c, 0x9f, 0x22, 0x7d, 0x6f, 0x89, 0x87, 0xfb,
  // CA key allowlist (0x0001, 0x00ab, 0x0014, 0x0005, 0x003a)
  0x03, 0x00, 0x0a, 0x00, 0x01, 0x00, 0xab, 0x00, 0x14, 0x00, 0x05, 0x00, 0x3a,
  // Not before
  0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
  // Not after
  0x05, 0x00, 0x04, 0xff, 0xff, 0xff, 0xff,
  // Principals blocklist (root, toor)
  0x06, 0x00, 0x0a, 0x72, 0x6f, 0x6f, 0x74, 0x00, 0x74, 0x6f, 0x6f, 0x72, 0x00};
const char ssh_req_file[] = "ssh_req.dat";

const unsigned char expected_result[] =
  {0x00, 0x00, 0x00, 0x1c, 0x73, 0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x2d, 0x63,
   0x65, 0x72, 0x74, 0x2d, 0x76, 0x30, 0x31, 0x40, 0x6f, 0x70, 0x65, 0x6e, 0x73,
   0x73, 0x68, 0x2e, 0x63, 0x6f, 0x6d, 0x00, 0x00, 0x00, 0x20, 0x91, 0xe7, 0x84,
   0x34, 0x8a, 0xbd, 0x1f, 0x52, 0x2c, 0x4f, 0xa7, 0x59, 0xfb, 0x97, 0xd2, 0x4b,
   0x07, 0xbd, 0xad, 0x1f, 0xaf, 0x53, 0x9a, 0x50, 0x35, 0x71, 0xb0, 0x63, 0x64,
   0xe2, 0x88, 0xcf, 0x00, 0x00, 0x00, 0x03, 0x01, 0x00, 0x01, 0x00, 0x00, 0x01,
   0x01, 0x00, 0xd8, 0x2a, 0x18, 0x56, 0x40, 0x5b, 0xe5, 0xc7, 0xed, 0x94, 0x6f,
   0x1f, 0x18, 0x87, 0x33, 0x10, 0xc6, 0xfa, 0x00, 0x1e, 0xdf, 0xf8, 0xf5, 0xaf,
   0x89, 0xda, 0x5d, 0x05, 0x39, 0xd2, 0x15, 0x55, 0x78, 0x41, 0xb7, 0x89, 0x51,
   0x9c, 0x0b, 0xe0, 0xbc, 0x3c, 0x65, 0x40, 0xdf, 0x84, 0xd2, 0xf1, 0xaf, 0xd8,
   0x0f, 0x0b, 0x40, 0x7e, 0x59, 0x84, 0x92, 0x24, 0xa9, 0xaa, 0x83, 0x70, 0x0b,
   0x6e, 0x6a, 0xbc, 0xb1, 0x60, 0xbe, 0xa1, 0xad, 0xa1, 0x4f, 0x96, 0xe8, 0xa7,
   0xfe, 0xc3, 0x21, 0x41, 0xa7, 0x73, 0xbc, 0x10, 0x0a, 0xdb, 0x4c, 0xfd, 0x7a,
   0xef, 0x85, 0xac, 0x99, 0xe7, 0xfb, 0x94, 0x7e, 0x09, 0xb7, 0xb6, 0x8d, 0x5d,
   0x03, 0x4b, 0x9c, 0x2e, 0xc6, 0xcc, 0x3b, 0x9c, 0xb3, 0xb2, 0xaf, 0x5d, 0x48,
   0xd3, 0x51, 0x33, 0xc2, 0xb8, 0xc2, 0x21, 0x10, 0x40, 0x8e, 0x54, 0x26, 0x2e,
   0xbb, 0x32, 0x6e, 0x69, 0x4b, 0x6d, 0xe9, 0x4a, 0xa3, 0x0b, 0xc6, 0xa3, 0x13,
   0x1c, 0x72, 0x7d, 0x23, 0x4b, 0x29, 0xe9, 0x3b, 0xfb, 0x26, 0x4e, 0xe2, 0xa4,
   0xbc, 0xad, 0xa0, 0x9c, 0xf2, 0xdd, 0xb4, 0x63, 0x21, 0x3b, 0x25, 0xb3, 0xd9,
   0x20, 0xb8, 0x62, 0xdb, 0x0c, 0xd3, 0xdc, 0xdf, 0x9f, 0xdf, 0x0e, 0xea, 0x74,
   0xd0, 0x3f, 0xb7, 0x04, 0x67, 0xac, 0xb7, 0xea, 0xe2, 0xc9, 0x0c, 0xe2, 0x44,
   0x03, 0x3c, 0x6f, 0x9c, 0x56, 0xee, 0x7b, 0x0d, 0x7c, 0xfc, 0xe6, 0x76, 0xac,
   0x7b, 0x10, 0x26, 0xf1, 0xb9, 0xaf, 0x53, 0x6c, 0x74, 0xbb, 0x8a, 0x24, 0xd5,
   0x91, 0xd8, 0xc8, 0x72, 0xfb, 0x6f, 0x52, 0x58, 0x94, 0xeb, 0x8d, 0xc2, 0x12,
   0xbc, 0xd1, 0xde, 0xfb, 0x49, 0xf3, 0x39, 0x51, 0x86, 0xd4, 0x32, 0x9f, 0x36,
   0x1b, 0x37, 0xb7, 0x8a, 0x4f, 0x43, 0x7b, 0xd9, 0xf0, 0x26, 0x5f, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x08, 0x69, 0x64, 0x65, 0x6e, 0x74, 0x69, 0x74, 0x79, 0x00, 0x00, 0x00, 0x12,
   0x00, 0x00, 0x00, 0x05, 0x75, 0x73, 0x65, 0x72, 0x31, 0x00, 0x00, 0x00, 0x05,
   0x75, 0x73, 0x65, 0x72, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x82, 0x00, 0x00, 0x00, 0x15, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74,
   0x2d, 0x58, 0x31, 0x31, 0x2d, 0x66, 0x6f, 0x72, 0x77, 0x61, 0x72, 0x64, 0x69,
   0x6e, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x17, 0x70, 0x65, 0x72,
   0x6d, 0x69, 0x74, 0x2d, 0x61, 0x67, 0x65, 0x6e, 0x74, 0x2d, 0x66, 0x6f, 0x72,
   0x77, 0x61, 0x72, 0x64, 0x69, 0x6e, 0x67, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x16, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x2d, 0x70, 0x6f, 0x72, 0x74,
   0x2d, 0x66, 0x6f, 0x72, 0x77, 0x61, 0x72, 0x64, 0x69, 0x6e, 0x67, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x70, 0x65, 0x72, 0x6d, 0x69, 0x74, 0x2d,
   0x70, 0x74, 0x79, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x70, 0x65,
   0x72, 0x6d, 0x69, 0x74, 0x2d, 0x75, 0x73, 0x65, 0x72, 0x2d, 0x72, 0x63, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x17, 0x00, 0x00,
   0x00, 0x07, 0x73, 0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x00, 0x00, 0x00, 0x03,
   0x01, 0x00, 0x01, 0x00, 0x00, 0x01, 0x01, 0x00, 0xd3, 0x35, 0xad, 0x1a, 0x8a,
   0xe2, 0x4d, 0x09, 0xaa, 0xb4, 0x29, 0x8b, 0x1e, 0xbd, 0xc9, 0x41, 0x22, 0xbe,
   0x11, 0xd3, 0x0e, 0xab, 0x25, 0xed, 0xbc, 0x5e, 0x2f, 0xaa, 0xf2, 0x48, 0x14,
   0x5b, 0x14, 0xee, 0xe3, 0x03, 0x9b, 0xf1, 0x52, 0x87, 0x0d, 0x61, 0x09, 0x51,
   0x60, 0xc9, 0x9b, 0xbb, 0x3f, 0xbe, 0xfe, 0x45, 0x46, 0x88, 0x30, 0x05, 0x04,
   0x6e, 0xd6, 0x0a, 0x17, 0x8b, 0xd8, 0xbc, 0xf3, 0xf1, 0x50, 0x50, 0xcb, 0xd5,
   0x8c, 0x87, 0x17, 0xa5, 0x6c, 0x36, 0xfc, 0x24, 0x30, 0xd0, 0xd2, 0x99, 0x16,
   0x72, 0x0f, 0xa9, 0x47, 0xe4, 0x3f, 0xd0, 0x15, 0xc1, 0x86, 0x66, 0xd9, 0xfa,
   0x47, 0x3d, 0x52, 0x7c, 0x9b, 0xb2, 0x9e, 0xb4, 0xfe, 0xfd, 0x99, 0x6d, 0x33,
   0xf3, 0x6b, 0xa1, 0xd7, 0x5a, 0x21, 0x30, 0x36, 0xdc, 0xd4, 0xa6, 0x26, 0xb2,
   0x22, 0xfe, 0x5c, 0x26, 0xe6, 0xfa, 0xee, 0xbc, 0x54, 0x09, 0xec, 0x90, 0xf8,
   0xfa, 0xd8, 0x91, 0xf2, 0x22, 0xce, 0xa9, 0x6b, 0x06, 0x5e, 0x55, 0x2d, 0x21,
   0x70, 0xed, 0xd3, 0xdb, 0x10, 0xb3, 0xef, 0xc7, 0x09, 0xbc, 0xe2, 0x1d, 0x47,
   0xa8, 0x58, 0xff, 0x13, 0x4a, 0x98, 0x4a, 0x7d, 0xce, 0x8b, 0x8c, 0xdb, 0x52,
   0xc7, 0xaa, 0x66, 0xca, 0x70, 0xb6, 0xc1, 0x11, 0x7b, 0x2d, 0x87, 0x74, 0x0a,
   0x6e, 0xcd, 0xab, 0x41, 0xd0, 0xfb, 0x13, 0xf1, 0xb9, 0xa0, 0x41, 0x59, 0xea,
   0x9d, 0x29, 0x42, 0x7b, 0xff, 0x44, 0x6a, 0xb7, 0xbb, 0x26, 0xac, 0x61, 0x9c,
   0xa9, 0x6b, 0xb7, 0x3f, 0xdc, 0xb1, 0x73, 0x18, 0x98, 0x56, 0x51, 0x60, 0x65,
   0x20, 0x40, 0x51, 0x58, 0xc7, 0x31, 0x86, 0x82, 0x46, 0x25, 0x99, 0x28, 0xfb,
   0x3a, 0xc0, 0x34, 0xd8, 0x9c, 0x93, 0x81, 0x13, 0xdb, 0xc5, 0xa8, 0x71, 0xe3,
   0x4f, 0xee, 0xe6, 0x9f, 0x00, 0x00, 0x01, 0x0f, 0x00, 0x00, 0x00, 0x07, 0x73,
   0x73, 0x68, 0x2d, 0x72, 0x73, 0x61, 0x00, 0x00, 0x01, 0x00, 0x0a, 0xcc, 0x96,
   0xd1, 0xc3, 0xa5, 0x05, 0xfb, 0x20, 0x2f, 0x3a, 0x70, 0x89, 0x73, 0x42, 0x01,
   0x21, 0x14, 0x44, 0xc6, 0x61, 0xbd, 0xc7, 0x5f, 0xf0, 0x8b, 0x04, 0x30, 0xa8,
   0x81, 0x81, 0x17, 0x6e, 0x8a, 0xe5, 0x88, 0x2a, 0x4a, 0xf8, 0x58, 0x82, 0x1c,
   0x67, 0x18, 0x26, 0x9a, 0x24, 0xb5, 0xf9, 0x1c, 0xc9, 0xe8, 0x12, 0xe2, 0x93,
   0xdc, 0x63, 0xc1, 0x4d, 0x39, 0x37, 0x44, 0xb4, 0x06, 0x27, 0x29, 0x53, 0x3b,
   0xa1, 0x72, 0xb1, 0xb0, 0x7f, 0xec, 0x76, 0x88, 0x68, 0x54, 0xba, 0xff, 0xe4,
   0x29, 0x67, 0xbc, 0xae, 0x0b, 0x33, 0xc3, 0x78, 0xf8, 0x90, 0xe0, 0x33, 0xda,
   0x1c, 0x76, 0x5a, 0x18, 0xcf, 0xdf, 0x22, 0x17, 0x7a, 0xba, 0x7d, 0x84, 0x3f,
   0xd1, 0x4a, 0x18, 0x6d, 0x67, 0x8c, 0xa0, 0x64, 0xe9, 0x57, 0xdc, 0xb9, 0x99,
   0x89, 0x85, 0xd6, 0x28, 0x82, 0x33, 0x3f, 0x95, 0xda, 0xfb, 0x8b, 0x92, 0x35,
   0xd2, 0x73, 0x1f, 0xdd, 0x4a, 0x62, 0x0a, 0x67, 0xfb, 0xdc, 0x08, 0x6d, 0x4b,
   0xe4, 0xed, 0x9f, 0x22, 0xda, 0xe0, 0x02, 0x8e, 0x8c, 0xcb, 0x33, 0xe6, 0x08,
   0x91, 0x4d, 0x26, 0xf3, 0xc7, 0xdd, 0xad, 0x08, 0xec, 0x63, 0xf0, 0xe8, 0x09,
   0x14, 0x78, 0xd4, 0xf3, 0xc0, 0xb7, 0xd7, 0x5c, 0x9d, 0x62, 0x00, 0x8c, 0xde,
   0xde, 0xcd, 0x75, 0x5c, 0x9b, 0xfb, 0x85, 0xce, 0x3d, 0x58, 0xb0, 0x4a, 0xc8,
   0xc3, 0xc5, 0x86, 0xe5, 0x3f, 0xf4, 0x86, 0x29, 0x57, 0x2e, 0x7a, 0xd4, 0x64,
   0x29, 0xa8, 0x42, 0xba, 0xf3, 0xb4, 0x92, 0x3f, 0x77, 0xc3, 0x44, 0xaa, 0xcc,
   0x30, 0xb8, 0x82, 0xb2, 0xcb, 0x29, 0x9c, 0xea, 0x84, 0xa5, 0x0f, 0x58, 0x59,
   0x3d, 0x43, 0xe3, 0xc4, 0xdd, 0x18, 0xdf, 0xe4, 0x82, 0x45, 0x22, 0xea, 0xa7,
   0xe2, 0x26, 0xc8, 0x41, 0xfb, 0x37};

int main(void) {
  yh_connector *connector = NULL;
  yh_session *session = NULL;
  yh_rc yrc = YHR_GENERIC_ERROR;

  uint16_t authkey = 1;

  const char *connector_url;

  connector_url = getenv("DEFAULT_CONNECTOR_URL");
  if (connector_url == NULL) {
    connector_url = DEFAULT_CONNECTOR_URL;
  }

  yrc = yh_init();
  assert(yrc == YHR_SUCCESS);

  yrc = yh_init_connector(connector_url, &connector);
  assert(yrc == YHR_SUCCESS);

  yrc = yh_connect(connector, 0);
  assert(yrc == YHR_SUCCESS);

  yrc = yh_create_session_derived(connector, authkey, password,
                                  sizeof(password) - 1, false, &session);
  assert(yrc == YHR_SUCCESS);

  uint8_t session_id;
  yrc = yh_get_session_id(session, &session_id);
  assert(yrc == YHR_SUCCESS);

  printf("Successfully established session %02d\n", session_id);

  FILE *fp = fopen(ssh_ca_pvtkey_file, "rb");
  assert(fp != NULL);

  yh_algorithm algorithm = 0;
  uint8_t key[2048];
  size_t key_material_len = sizeof(key);
  if (!read_file(fp, key, &key_material_len)) {
    assert(false);
  }
  bool ret = read_private_key(key, key_material_len, &algorithm, key,
                              &key_material_len, false);
  assert(ret == true);
  assert(algorithm == YH_ALGO_RSA_2048);

  yh_capabilities capabilities = {{0}};
  yrc = yh_string_to_capabilities("sign-ssh-certificate", &capabilities);
  assert(yrc == YHR_SUCCESS);

  uint16_t domain_five = 0;
  yh_string_to_domains("5", &domain_five);
  assert(yrc == YHR_SUCCESS);

  uint16_t ca_key_id =
    5; // We use ID 5 because it is in the template's allowlist
  yrc = yh_util_get_object_info(session, ca_key_id, YH_ASYMMETRIC_KEY, NULL);
  if (yrc == YHR_SUCCESS) {
    yrc = yh_util_delete_object(session, ca_key_id, YH_ASYMMETRIC_KEY);
    assert(yrc == YHR_SUCCESS);
  } else {
    assert(yrc == YHR_DEVICE_OBJECT_NOT_FOUND);
  }
  yrc = yh_util_import_rsa_key(session, &ca_key_id, key_label, domain_five,
                               &capabilities, algorithm, key,
                               key + (key_material_len / 2));
  assert(yrc == YHR_SUCCESS);

  printf("Key imported with ID %04x\n", ca_key_id);

  uint16_t template_id = 10;
  yrc = yh_util_get_object_info(session, template_id, YH_TEMPLATE, NULL);
  if (yrc == YHR_SUCCESS) {
    yrc = yh_util_delete_object(session, template_id, YH_TEMPLATE);
    assert(yrc == YHR_SUCCESS);
  } else {
    assert(yrc == YHR_DEVICE_OBJECT_NOT_FOUND);
  }
  yrc = yh_util_import_template(session, &template_id, key_label, domain_five,
                                &capabilities, YH_ALGO_TEMPLATE_SSH,
                                template_dat, sizeof(template_dat));
  assert(yrc == YHR_SUCCESS);

  printf("Template imported with ID %04x\n", template_id);

  uint8_t template2[sizeof(template_dat)];
  size_t template2_len = sizeof(template2);
  yrc = yh_util_get_template(session, template_id, template2, &template2_len);
  assert(yrc == YHR_SUCCESS);

  assert(sizeof(template_dat) == template2_len);
  assert(memcmp(template_dat, template2, template2_len) == 0);

  uint8_t ssh_req[2048];
  size_t ssh_req_len = sizeof(ssh_req);

  fp = fopen(ssh_req_file, "rb");
  assert(fp != NULL);

  fseek(fp, 0, SEEK_END);
  ssh_req_len = ftell(fp);
  assert(ssh_req_len <= sizeof(ssh_req));
  fseek(fp, 0, SEEK_SET);

  size_t read = fread(ssh_req, 1, ssh_req_len, fp);
  fclose(fp);
  printf("actually read %zu, expected %zu\n", read, ssh_req_len);
  assert(read == ssh_req_len);

  size_t ssh_cert_len = sizeof(ssh_req) - ssh_req_len;
  yrc =
    yh_util_sign_ssh_certificate(session, ca_key_id, template_id,
                                 YH_ALGO_RSA_PKCS1_SHA1, ssh_req, ssh_req_len,
                                 ssh_req + ssh_req_len, &ssh_cert_len);
  assert(yrc == YHR_SUCCESS);
  assert(memcmp(expected_result, ssh_req + 4 + 256, sizeof(expected_result)) ==
         0);

  BIO *bio;
  BIO *b64;

  b64 = BIO_new(BIO_f_base64());
  bio = BIO_new_fp(stdout, BIO_NOCLOSE);
  bio = BIO_push(b64, bio);

  fprintf(stdout, "ssh-rsa-cert-v01@openssh.com ");
  (void) BIO_set_flags(bio, BIO_FLAGS_BASE64_NO_NL);
  (void) BIO_write(bio, ssh_req + 4 + 256,
                   ssh_req_len + ssh_cert_len - 4 - 256);
  (void) BIO_flush(bio);
  fprintf(stdout, "\n");

  BIO_free_all(bio);

  yrc = yh_util_close_session(session);
  assert(yrc == YHR_SUCCESS);

  yrc = yh_destroy_session(&session);
  assert(yrc == YHR_SUCCESS);

  yh_disconnect(connector);
  assert(yrc == YHR_SUCCESS);

  yrc = yh_exit();
  assert(yrc == YHR_SUCCESS);

  return 0;
}
